/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// test sca_tdf::sca_ltf_zp complex conjugate

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE( my_ltf )
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  my_ltf( sc_core::sc_module_name nm )
  : in("in"), out("out") {}

  void initialize()
  {
    //         (s - 90112.6j) (s - 145805j) (s + 90112.6j) (s + 145805j)
    //  H(s) = ---------------------------------------------------------
    //                                1.0 + 0.0j

    zeros(0) = sca_util::sca_complex( 0.0, 90112.6 );
    zeros(1) = sca_util::sca_complex( 0.0, 145805 );
    zeros(2) = sca_util::sca_complex( 0.0, -145805 );
    zeros(3) = sca_util::sca_complex( 0.0, -90112.6 );
  }

  void processing()
  {
    double tmp = ltf_zp( zeros, poles, in.read() * 1e-20 );
    out.write(tmp);
  }

  void ac_processing()
  {
    using sca_ac_analysis::sca_ac;
    sca_ac(out) = sca_ac_analysis::sca_ac_ltf_zp( zeros, poles, sca_ac(in) * 1e-20 );
  }

private:
  sca_tdf::sca_ltf_zp ltf_zp; // Laplace transfer function
  sca_util::sca_vector<sca_util::sca_complex> poles, zeros; // poles and zeros as complex values
};


SCA_TDF_MODULE(my_step)
{
  sca_tdf::sca_out<double> out; // output port

  SCA_CTOR(my_step) : out("out")
  {}

  void set_attributes()
  {
    out.set_timestep(100.0, sc_core::SC_MS);
    out.set_delay(1);
  }

  void ac_processing()
  {
     using sca_ac_analysis::sca_ac;
     sca_ac(out) = 1.0;
  }

  void processing()
  {
    out.write(1.0);
  }
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1;
  sca_tdf::sca_signal<double> sig2;

  my_step step("step");
  my_ltf  ltf1("ltf1");

  step.out(sig1);
  ltf1.in(sig1);
  ltf1.out(sig2);

  // tracing
  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("conjugate_problem");
  tf->set_mode(sca_util::sca_ac_format(sca_util::SCA_AC_DB_DEG));
  sca_util::sca_trace(tf, sig1, "sig1");
  sca_util::sca_trace(tf, sig2, "sig2");

  try
  {
    // sc_core::sc_start(7, sc_core::SC_SEC);
    sca_ac_analysis::sca_ac_start(5e3, 50e3, 1000, sca_ac_analysis::SCA_LOG);
  }
  catch (const std::exception& e)
  {
    std::cerr << "Error catched: " << e.what() << std::endl;
  }

  sca_util::sca_close_tabular_trace_file(tf);

  test_util::check_results("conjugate_problem", 2);

  TEST_LABEL_END;

  return 0;
}
