/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

/*
 *   assignment to out via operator
 *
 *   Tests:
 *           single rate out / single rate in  (vector,vector,?in?)
 *                                             scalar out / scalar in
 *                                             port out   / scalar in
 *                                             scalar out / port in
 *                                             port out   / port in
 *
 *           single rate out / multi rate in  (vector,vector,?in?)
 *                                             scalar out / vector in
 *                                             port out   / vector in
 *                                             scalar out / port in
 *                                             port out   / port in
 *
 *           multi rate out / single rate in  (vector,vector, sc_core::sc_time,?in?)
 *           //delay is required due causality (sc_time argument)
 *                                             vector out / scalar in
 *                                             port out   / scalar in
 *                                             vector out / port in
 *                                             port out   / port in
 *
 *           multi rate out / multi rate in  (vector,vector,?in?)
 *                                             vector out / vector in
 *                                             port out   / vector in
 *                                             vector out / port in
 *                                             port out   / port in
 *
 */

#include <systemc-ams>
#include "test_utilities.h"

#include <string>
#include <fstream>
#include <sstream>

///////////////////////////////////////////////////////////////////////////////

static const double DELAY_MAX_IN_SEC = 0.85e-3;
static const double DELAY_IN_SEC = 0.83e-3;

SCA_TDF_MODULE(ltf_nd_src)
{
  // single rate
  sca_tdf::sca_out<double> outp_const_1; // constant value (1.0)
  sca_tdf::sca_out<double> outp_ramp_1;  // ramp  y=t
  sca_tdf::sca_out<double> outp_t2_1;    // y=0.5*t*t
  sca_tdf::sca_out<double> outp_sin_1;
  sca_tdf::sca_out<double> outp_t3_1;    // y=0.5/3*t*t*t

  // multi rate
  sca_tdf::sca_out<double> outp_const_r;
  sca_tdf::sca_out<double> outp_ramp_r;
  sca_tdf::sca_out<double> outp_t2_r;
  sca_tdf::sca_out<double> outp_sin_r;
  sca_tdf::sca_out<double> outp_t3_r;

  // single rate
  sca_tdf::sca_out<double> outp0;
  sca_tdf::sca_out<double> outp1;
  sca_tdf::sca_out<double> outp2;
  sca_tdf::sca_out<double> outp3;
  sca_tdf::sca_out<double> outp4;

  // multi rate
  sca_tdf::sca_out<double> outpr0;
  sca_tdf::sca_out<double> outpr1;
  sca_tdf::sca_out<double> outpr2;
  sca_tdf::sca_out<double> outpr3;
  sca_tdf::sca_out<double> outpr4;

  struct params
  {
    int rate;
    sca_core::sca_time sample_time;
    double f_sin;

    params()
    {
      rate = 5;
      sample_time = sca_core::sca_time(1.0, sc_core::SC_US);
      f_sin = 13.3e3;
    }
  };

  void set_attributes()
  {
    set_timestep(p.sample_time);

    outp_const_r.set_rate(p.rate);
    outp_ramp_r.set_rate(p.rate);
    outp_t2_r.set_rate(p.rate);
    outp_sin_r.set_rate(p.rate);
    outp_t3_r.set_rate(p.rate);

    outpr0.set_rate(p.rate);
    outpr1.set_rate(p.rate);
    outpr2.set_rate(p.rate);
    outpr3.set_rate(p.rate);
    outpr4.set_rate(p.rate);
  }

  void initialize();
  void processing();

  ltf_nd_src( sc_core::sc_module_name nm, params pa = params() )
  : p(pa)
  {}

  params p;

private:
  sca_tdf::sca_ltf_nd  nd10, nd11, nd12, nd13, nd14;
  sca_tdf::sca_ltf_nd  nd20, nd21, nd22, nd23, nd24;
  sca_tdf::sca_ltf_nd  nd30, nd31, nd32, nd33, nd34;
  sca_tdf::sca_ltf_nd  nd40, nd41, nd42, nd43, nd44;

  sca_util::sca_vector<double> N1, D1;
  sca_util::sca_vector<double> N2, D2;
  sca_util::sca_vector<double> N3, D3;
  sca_util::sca_vector<double> N4, D4;
};

void ltf_nd_src::initialize()
{
  N1(0) = 1.0;
  D1(0) = 1.0;  // y=x

  N2(0) = 1.0;
  D2(1) = 1.0;  // y=t*x - x=const=1.0

  N3(0) = 1.0;
  D3(2) = 1.0;  // y=0.5*t*t*x - x=const=1.0

  N4(0) = 1.0;
  D4(3) = 1.0;  // y=0.5/3.0*t*t*t*x - x=const=1.0

  nd24.set_max_delay(800.0, sc_core::SC_US);
}

// time domain implementation
void ltf_nd_src::processing()
{
  double inp;
  inp= std::sin(2.0 * M_PI * p.f_sin * get_time().to_seconds());

  // output to scalar port sca_tdf::sca_out<double>
  // input double
  outp0 = nd10(N1, D1, 1.0); // single rate
  outp1 = nd11(N2, D2, 1.0);
  outp2 = nd12(N3, D3, 1.0);
  outp3 = nd13(N1, D1, inp);

  // test tstep
  outp4 = nd14(N4, D4, 1.0, 1.0, sc_core::sc_time(0.1, sc_core::SC_US));
  outp4 = nd14(N4, D4, 1.0, 1.0, sc_core::sc_time(0.2, sc_core::SC_US));
  outp4 = nd14(N4, D4, 1.0);

  // output to port sca_tdf::sca_out<double >
  outpr0 = nd20(N1, D1, sc_core::sc_time(0.8, sc_core::SC_US), 1.0); // multi rate with 0.8us delay
  outpr1 = nd21(N2, D2, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);
  outpr2 = nd22(N3, D3, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);
  outpr3 = nd23(N1, D1, sc_core::sc_time(0.8, sc_core::SC_US), inp);

  // check tstep
  outpr4 = nd24(N4, D4, sc_core::sc_time(0.8, sc_core::SC_US), 1.0, 1.0, sc_core::sc_time(0.1, sc_core::SC_US));
  outpr4 = nd24(N4, D4, sc_core::sc_time(0.8, sc_core::SC_US), 1.0, 1.0, sc_core::sc_time(0.2, sc_core::SC_US));
  outpr4 = nd24(N4, D4, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);

  double outv0, outv1, outv2, outv3, outv4;

  // output to double
  // input double
  outv0 = nd30(N1, D1, 1.0);
  outv1 = nd31(N2, D2, 1.0);
  outv2 = nd32(N3, D3, 1.0);
  outv3 = nd33(N1, D1, inp);

  // check tstep
  outv4 = nd34(N4, D4, 1.0, 1.0, sc_core::sc_time(0.1, sc_core::SC_US));
  outv4 = nd34(N4, D4, 1.0, 1.0, sc_core::sc_time(0.7, sc_core::SC_US));
  outv4 = nd34(N4, D4, 1.0);

  outp_const_1 = outv0;
  outp_ramp_1  = outv1;
  outp_t2_1    = outv2;
  outp_sin_1   = outv3;
  outp_t3_1    = outv4;

  sca_util::sca_vector<double> out_vec0(p.rate);
  sca_util::sca_vector<double> out_vec1(p.rate);
  sca_util::sca_vector<double> out_vec2(p.rate);
  sca_util::sca_vector<double> out_vec3(p.rate);
  sca_util::sca_vector<double> out_vec4(p.rate);

  // output to vector sca_util::sca_vector<double>
  // input double value
  out_vec0 = nd40(N1, D1, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);
  out_vec1 = nd41(N2, D2, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);
  out_vec2 = nd42(N3, D3, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);
  out_vec3 = nd43(N1, D1, sc_core::sc_time(0.8, sc_core::SC_US), inp);
  out_vec4 = nd44(N4, D4, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);

  for (long i = 0; i <p.rate; i++)
  {
    outp_const_r[i] = out_vec0(i);
    outp_ramp_r[i]  = out_vec1(i);
    outp_t2_r[i]    = out_vec2(i);
    outp_sin_r[i]   = out_vec3(i);
    outp_t3_r[i]    = out_vec4(i);
  }
}

////////////////////////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE (ltf_nd_test)
{
  // single rate
  sca_tdf::sca_in<double> in_const_1; // constant value (1.0)
  sca_tdf::sca_in<double> in_ramp_1;  // ramp  y=t
  sca_tdf::sca_in<double> in_t2_1;    // y=0.5*t*t
  sca_tdf::sca_in<double> in_sin_1;

  sca_tdf::sca_out<double> out_const_1;
  sca_tdf::sca_out<double> out_ramp_1;
  sca_tdf::sca_out<double> out_t2_1;
  sca_tdf::sca_out<double> out_sin_1;
  sca_tdf::sca_out<double> out_t3_1;

  sca_tdf::sca_out<double> out_const_3;
  sca_tdf::sca_out<double> out_ramp_3;
  sca_tdf::sca_out<double> out_t2_3;
  sca_tdf::sca_out<double> out_sin_3;
  sca_tdf::sca_out<double> out_t3_3;

  sca_tdf::sca_out<double> out_const_5;
  sca_tdf::sca_out<double> out_ramp_5;
  sca_tdf::sca_out<double> out_t2_5;
  sca_tdf::sca_out<double> out_sin_5;
  sca_tdf::sca_out<double> out_t3_5;

  sca_tdf::sca_out<double> out_const_7;
  sca_tdf::sca_out<double> out_ramp_7;
  sca_tdf::sca_out<double> out_t2_7;
  sca_tdf::sca_out<double> out_sin_7;
  sca_tdf::sca_out<double> out_t3_7;

  // multi rate
  sca_tdf::sca_in<double> in_const_r;
  sca_tdf::sca_in<double> in_ramp_r;
  sca_tdf::sca_in<double> in_t2_r;
  sca_tdf::sca_in<double> in_sin_r;

  // single rate
  sca_tdf::sca_in<double> in_10;
  sca_tdf::sca_in<double> in_11;
  sca_tdf::sca_in<double> in_12;
  sca_tdf::sca_in<double> in_13;

  sca_tdf::sca_out<double> out_20;
  sca_tdf::sca_out<double> out_21;
  sca_tdf::sca_out<double> out_22;
  sca_tdf::sca_out<double> out_23;
  sca_tdf::sca_out<double> out_24;

  sca_tdf::sca_out<double> out_40;
  sca_tdf::sca_out<double> out_41;
  sca_tdf::sca_out<double> out_42;
  sca_tdf::sca_out<double> out_43;
  sca_tdf::sca_out<double> out_44;

  sca_tdf::sca_out<double> out_60;
  sca_tdf::sca_out<double> out_61;
  sca_tdf::sca_out<double> out_62;
  sca_tdf::sca_out<double> out_63;
  sca_tdf::sca_out<double> out_64;

  sca_tdf::sca_out<double> out_80;
  sca_tdf::sca_out<double> out_81;
  sca_tdf::sca_out<double> out_82;
  sca_tdf::sca_out<double> out_83;
  sca_tdf::sca_out<double> out_84;

  // multi rate
  sca_tdf::sca_in<double> in_r_0;
  sca_tdf::sca_in<double> in_r_1;
  sca_tdf::sca_in<double> in_r_2;
  sca_tdf::sca_in<double> in_r_3;

  sca_tdf::sca_out<double> out_r_90;
  sca_tdf::sca_out<double> out_r_91;
  sca_tdf::sca_out<double> out_r_92;
  sca_tdf::sca_out<double> out_r_93;
  sca_tdf::sca_out<double> out_r_94;

  sca_tdf::sca_out<double> out_r_100;
  sca_tdf::sca_out<double> out_r_101;
  sca_tdf::sca_out<double> out_r_102;
  sca_tdf::sca_out<double> out_r_103;
  sca_tdf::sca_out<double> out_r_104;

  sca_tdf::sca_out<double> out_r_110;
  sca_tdf::sca_out<double> out_r_111;
  sca_tdf::sca_out<double> out_r_112;
  sca_tdf::sca_out<double> out_r_113;
  sca_tdf::sca_out<double> out_r_114;

  sca_tdf::sca_out<double> out_r_120;
  sca_tdf::sca_out<double> out_r_121;
  sca_tdf::sca_out<double> out_r_122;
  sca_tdf::sca_out<double> out_r_123;
  sca_tdf::sca_out<double> out_r_124;

  sca_tdf::sca_out<double> out_r_130;
  sca_tdf::sca_out<double> out_r_131;
  sca_tdf::sca_out<double> out_r_132;
  sca_tdf::sca_out<double> out_r_133;
  sca_tdf::sca_out<double> out_r_134;

  sca_tdf::sca_out<double> out_r_140;
  sca_tdf::sca_out<double> out_r_141;
  sca_tdf::sca_out<double> out_r_142;
  sca_tdf::sca_out<double> out_r_143;
  sca_tdf::sca_out<double> out_r_144;

  sca_tdf::sca_out<double> out_r_150;
  sca_tdf::sca_out<double> out_r_151;
  sca_tdf::sca_out<double> out_r_152;
  sca_tdf::sca_out<double> out_r_153;
  sca_tdf::sca_out<double> out_r_154;

  sca_tdf::sca_out<double> out_r_160;
  sca_tdf::sca_out<double> out_r_161;
  sca_tdf::sca_out<double> out_r_162;
  sca_tdf::sca_out<double> out_r_163;
  sca_tdf::sca_out<double> out_r_164;

  struct params // parameter
  {
    unsigned long rate, rate2;

    params() // default for parameter
    {
      rate = 5;
      rate2 = 3;
    }
  };

  void initialize();     // initialization
  void processing();     // time domain processing method
  void set_attributes();
  params p;


  // constructor
  ltf_nd_test(sc_core::sc_module_name nm, params pa = params())
  : p(pa)
  {}

  // definition of local variables
private:
  sca_tdf::sca_ltf_nd nd10, nd11, nd12, nd13, nd14;
  sca_tdf::sca_ltf_nd nd20, nd21, nd22, nd23, nd24;
  sca_tdf::sca_ltf_nd nd30, nd31, nd32, nd33, nd34;
  sca_tdf::sca_ltf_nd nd40, nd41, nd42, nd43, nd44;
  sca_tdf::sca_ltf_nd nd50, nd51, nd52, nd53, nd54;
  sca_tdf::sca_ltf_nd nd60, nd61, nd62, nd63, nd64;
  sca_tdf::sca_ltf_nd nd70, nd71, nd72, nd73, nd74;
  sca_tdf::sca_ltf_nd nd80, nd81, nd82, nd83, nd84;
  sca_tdf::sca_ltf_nd nd90, nd91, nd92, nd93, nd94;
  sca_tdf::sca_ltf_nd nd100, nd101, nd102, nd103, nd104;
  sca_tdf::sca_ltf_nd nd110, nd111, nd112, nd113, nd114;
  sca_tdf::sca_ltf_nd nd120, nd121, nd122, nd123, nd124;
  sca_tdf::sca_ltf_nd nd130, nd131, nd132, nd133, nd134;
  sca_tdf::sca_ltf_nd nd140, nd141, nd142, nd143, nd144;
  sca_tdf::sca_ltf_nd nd150, nd151, nd152, nd153, nd154;
  sca_tdf::sca_ltf_nd nd160, nd161, nd162, nd163, nd164;
  sca_tdf::sca_ltf_nd nd_delay0, nd_delay1, nd_delay2, nd_delay3, nd_delay4;
  sca_util::sca_vector<double> N, D, N2, D2;

  double cnt;
};

void ltf_nd_test::set_attributes()
{
  in_const_r.set_rate(p.rate);
  in_ramp_r.set_rate(p.rate);
  in_t2_r.set_rate(p.rate);
  in_sin_r.set_rate(p.rate);

  in_r_0.set_rate(p.rate);
  in_r_1.set_rate(p.rate);
  in_r_2.set_rate(p.rate);
  in_r_3.set_rate(p.rate);

  out_r_90.set_rate(p.rate);
  out_r_91.set_rate(p.rate);
  out_r_92.set_rate(p.rate);
  out_r_93.set_rate(p.rate);
  out_r_94.set_rate(p.rate);

  out_r_100.set_rate(p.rate);
  out_r_101.set_rate(p.rate);
  out_r_102.set_rate(p.rate);
  out_r_103.set_rate(p.rate);
  out_r_104.set_rate(p.rate);

  out_r_110.set_rate(p.rate);
  out_r_111.set_rate(p.rate);
  out_r_112.set_rate(p.rate);
  out_r_113.set_rate(p.rate);
  out_r_114.set_rate(p.rate);

  out_r_120.set_rate(p.rate);
  out_r_121.set_rate(p.rate);
  out_r_122.set_rate(p.rate);
  out_r_123.set_rate(p.rate);
  out_r_124.set_rate(p.rate);

  out_r_130.set_rate(p.rate2);
  out_r_131.set_rate(p.rate2);
  out_r_132.set_rate(p.rate2);
  out_r_133.set_rate(p.rate2);
  out_r_134.set_rate(p.rate2);

  out_r_140.set_rate(p.rate2);
  out_r_141.set_rate(p.rate2);
  out_r_142.set_rate(p.rate2);
  out_r_143.set_rate(p.rate2);
  out_r_144.set_rate(p.rate2);

  out_r_150.set_rate(p.rate2);
  out_r_151.set_rate(p.rate2);
  out_r_152.set_rate(p.rate2);
  out_r_153.set_rate(p.rate2);
  out_r_154.set_rate(p.rate2);

  out_r_160.set_rate(p.rate2);
  out_r_161.set_rate(p.rate2);
  out_r_162.set_rate(p.rate2);
  out_r_163.set_rate(p.rate2);
  out_r_164.set_rate(p.rate2);

  nd10.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd11.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd12.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd13.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd14.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd20.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd21.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd22.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd23.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd24.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd30.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd31.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd32.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd33.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd34.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd40.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd41.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd42.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd43.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd44.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd50.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd51.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd52.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd53.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd54.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd60.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd61.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd62.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd63.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd64.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd70.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd71.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd72.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd73.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd74.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd80.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd81.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd82.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd83.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd84.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd90.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd91.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd92.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd93.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd94.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd100.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd101.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd102.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd103.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd104.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd110.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd111.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd112.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd113.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd114.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd120.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd121.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd122.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd123.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd124.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd130.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd131.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd132.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd133.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd134.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd140.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd141.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd142.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd143.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd144.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd150.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd151.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd152.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd153.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd154.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);

  nd160.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd161.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd162.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd163.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
  nd164.set_max_delay(DELAY_MAX_IN_SEC, sc_core::SC_SEC);
}

void ltf_nd_test::initialize()
{
  N(0) = 1.0;
  D(0) = 1.0;

  N2(0) = 1.0;
  D2(1) = 1.0;  // H(s)=1/s

  nd_delay0.set_max_delay((2 * in_ramp_r.get_rate()-1) * in_ramp_r.get_timestep());
  nd_delay1.set_max_delay((2 * in_ramp_r.get_rate()-1) * in_ramp_r.get_timestep());
  nd_delay2.set_max_delay((2 * in_ramp_r.get_rate()-1) * in_ramp_r.get_timestep());
  nd_delay3.set_max_delay((2 * in_ramp_r.get_rate()-1) * in_ramp_r.get_timestep());
  nd_delay4.set_max_delay((2 * in_ramp_r.get_rate()-1) * in_ramp_r.get_timestep());
}

// time domain implementation
void ltf_nd_test::processing()
{
  double x0;
  double x1;
  double x2;
  double x3;

  x0 = in_const_1.read();
  x1 = in_ramp_1.read();
  x2 = in_t2_1.read();
  x3 = in_sin_1.read();

  ////////// single rate in / single rate out //////////////

  // double out / double in
  const double y10 = nd10(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x0);
  const double y11 = nd11(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x1);
  const double y12 = nd12(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x2);
  const double y13 = nd13(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x3);
  const double y14 = nd14(N2, D2, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x2);

  out_const_1 = y10;
  out_ramp_1  = y11;
  out_t2_1    = y12;
  out_sin_1   = y13;
  out_t3_1    = y14;

  // port out / double in
  out_20 = nd20(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x0);
  out_21 = nd21(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x1);
  out_22 = nd22(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x2);
  out_23 = nd23(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x3);
  out_24 = nd24(N2, D2, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x2);

  // double out / port in
  const double y30 = nd30(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_10);
  const double y31 = nd31(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_11);
  const double y32 = nd32(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_12);
  const double y33 = nd33(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_13);
  const double y34 = nd34(N2, D2, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_12);

  out_const_3 = y30;
  out_ramp_3  = y31;
  out_t2_3    = y32;
  out_sin_3   = y33;
  out_t3_3    = y34;

  // port out / port in
  out_40 = nd40(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_10);
  out_41 = nd41(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_11);
  out_42 = nd42(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_12);
  out_43 = nd43(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_13);
  out_44 = nd44(N2, D2, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_12);

  ///////// multi rate in (rate=5) / single rate out ////////////////

  // -800  -600  -400  -200  0   time stamps (ps) of vector elements

  sca_util::sca_vector<double> x_vec0(p.rate);
  sca_util::sca_vector<double> x_vec1(p.rate);
  sca_util::sca_vector<double> x_vec2(p.rate);
  sca_util::sca_vector<double> x_vec3(p.rate);
  sca_util::sca_vector<double> x_vec4(p.rate);

  sca_util::sca_vector<double> x_vec0_delayed(p.rate);
  sca_util::sca_vector<double> x_vec1_delayed(p.rate);
  sca_util::sca_vector<double> x_vec2_delayed(p.rate);
  sca_util::sca_vector<double> x_vec3_delayed(p.rate);
  sca_util::sca_vector<double> x_vec4_delayed(p.rate);

  // this assignment corresponds to a delay of -0.8us (negative!!!)
  // assign time 0 -> -800, 200 -> -600, ...
  for (unsigned long i = 0; i < p.rate; i++)
  {
    x_vec0(i) = in_const_r.read(i);
    x_vec1(i) = in_ramp_r.read(i);
    x_vec2(i) = in_t2_r.read(i);
    x_vec3(i) = in_sin_r.read(i);
    x_vec4(i) = in_t2_r.read(i);
  }

  // remove negative delay of -0.8us
  // init init init init 0  | 200 400 600 800 1us | ...
  x_vec0_delayed = nd_delay0(N, D,(in_const_r.get_rate()-1) * in_const_r.get_timestep(), x_vec0);
  x_vec1_delayed = nd_delay1(N, D,(in_ramp_r.get_rate()-1)  * in_ramp_r.get_timestep(),  x_vec1);
  x_vec2_delayed = nd_delay2(N, D,(in_t2_r.get_rate()-1)    * in_t2_r.get_timestep(),    x_vec2);
  x_vec3_delayed = nd_delay3(N, D,(in_sin_r.get_rate()-1)   * in_sin_r.get_timestep(),   x_vec3);
  x_vec4_delayed = nd_delay4(N, D,(in_t2_r.get_rate()-1)    * in_t2_r.get_timestep(),    x_vec4);

  // double out / vector in (rate=5)
  // use delayed vector, due the last vector value corresponds to the out port value
  const double y50 = nd50(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x_vec0_delayed);
  const double y51 = nd51(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x_vec1_delayed);
  const double y52 = nd52(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x_vec2_delayed);
  const double y53 = nd53(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x_vec3_delayed);
  const double y54 = nd54(N2, D2, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x_vec4_delayed);

  out_const_5 = y50;
  out_ramp_5  = y51;
  out_t2_5    = y52;
  out_sin_5   = y53;
  out_t3_5    = y54;

  ////////////////////////////////////////////////////////////////////////

  // port out / vector in
  // use delayed vector, due the last vector value corresponds to the out port value
  out_60 = nd60(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x_vec0_delayed);
  out_61 = nd61(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x_vec1_delayed);
  out_62 = nd62(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x_vec2_delayed);
  out_63 = nd63(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x_vec3_delayed);
  out_64 = nd64(N2, D2, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x_vec4_delayed);

  ///////////////////////////////////////////////////////////////////////

  // double out / port in (rate=5)
  const double y70 = nd70(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_0);
  const double y71 = nd71(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_1);
  const double y72 = nd72(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_2);
  const double y73 = nd73(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_3);
  const double y74 = nd74(N2, D2, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_2);

  out_const_7 = y70;
  out_ramp_7  = y71;
  out_t2_7    = y72;
  out_sin_7   = y73;
  out_t3_7    = y74;

  //port out (single rate) / port in (rate=5)
  out_80 = nd80(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_0);
  out_81 = nd81(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_1);
  out_82 = nd82(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_2);
  out_83 = nd83(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_3);
  out_84 = nd84(N2, D2, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_2);

  ////////// single rate in / multi rate out (rate=5) /////////////

  // vector out / double in
  sca_util::sca_vector<double> y90(p.rate);
  sca_util::sca_vector<double> y91(p.rate);
  sca_util::sca_vector<double> y92(p.rate);
  sca_util::sca_vector<double> y93(p.rate);
  sca_util::sca_vector<double> y94(p.rate);

  y90 = nd90(N, D, sc_core::sc_time(DELAY_IN_SEC - 0.8e-6, sc_core::SC_SEC), x0);
  y91 = nd91(N, D, sc_core::sc_time(DELAY_IN_SEC - 0.8e-6, sc_core::SC_SEC), x1);
  y92 = nd92(N, D, sc_core::sc_time(DELAY_IN_SEC - 0.8e-6, sc_core::SC_SEC), x2);
  y93 = nd93(N, D, sc_core::sc_time(DELAY_IN_SEC - 0.8e-6, sc_core::SC_SEC), x3);
  y94 = nd94(N2, D2, sc_core::sc_time(DELAY_IN_SEC - 0.8e-6, sc_core::SC_SEC), x2);

  // this assignmet corresponds to a delay of 0.8us
  // -800 -> 0, -600 -> 200, -400 -> 400, -200 -> 600, 0 -> 800
  for (unsigned long i = 0; i < p.rate; i++)
  {
    out_r_90[i] = y90(i);
    out_r_91[i] = y91(i);
    out_r_92[i] = y92(i);
    out_r_93[i] = y93(i);
    out_r_94[i] = y94(i);
  }

  // overall delay 0.8us+0.83ms0.8us = 0.83ms

  /////////////////////////////////////////////////////////////////////////

  // port out (rate=5) / double in
  out_r_100 = nd100(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x0);
  out_r_101 = nd101(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x1);
  out_r_102 = nd102(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x2);
  out_r_103 = nd103(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x3);
  out_r_104 = nd104(N2, D2, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), x2);

  //overall delay 0.83ms

  //////////////////////

  // vector out (rate=5) / port in (single rate)
  sca_util::sca_vector<double> y110(p.rate);
  sca_util::sca_vector<double> y111(p.rate);
  sca_util::sca_vector<double> y112(p.rate);
  sca_util::sca_vector<double> y113(p.rate);
  sca_util::sca_vector<double> y114(p.rate);

  y110 = nd110(N, D, sc_core::sc_time(DELAY_IN_SEC - 0.8e-6, sc_core::SC_SEC), in_10);
  y111 = nd111(N, D, sc_core::sc_time(DELAY_IN_SEC - 0.8e-6, sc_core::SC_SEC), in_11);
  y112 = nd112(N, D, sc_core::sc_time(DELAY_IN_SEC - 0.8e-6, sc_core::SC_SEC), in_12);
  y113 = nd113(N, D, sc_core::sc_time(DELAY_IN_SEC - 0.8e-6, sc_core::SC_SEC), in_13);
  y114 = nd114(N2, D2, sc_core::sc_time(DELAY_IN_SEC - 0.8e-6, sc_core::SC_SEC), in_12);

  // this assignment corresponds to 0.8us delay
  // -800 -> 0, -600 -> 200, -400 -> 400, -200 -> 600, 0 -> 800
  for (unsigned long i = 0; i < p.rate; i++)
  {
    out_r_110[i] = y110(i);
    out_r_111[i] = y111(i);
    out_r_112[i] = y112(i);
    out_r_113[i] = y113(i);
    out_r_114[i] = y114(i);
  }

  // overall delay 0.8us+0.83ms-0.8us = 0.83ms

  //////////////////////////////////////////////////////////

  // port out (rate 5) / port in (single rate)
  out_r_120 = nd120(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_10);
  out_r_121 = nd121(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_11);
  out_r_122 = nd122(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_12);
  out_r_123 = nd123(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_13);
  out_r_124 = nd124(N2, D2, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_12);

  // overall delay 0.83ms

  ////////// multi rate in / multi rate out /////////////

  // vector out (rate=3) / vector in (rate=5)
  sca_util::sca_vector<double> y130(p.rate2);
  sca_util::sca_vector<double> y131(p.rate2);
  sca_util::sca_vector<double> y132(p.rate2);
  sca_util::sca_vector<double> y133(p.rate2);
  sca_util::sca_vector<double> y134(p.rate2);

  // 0     1     2     3    4              sample index
  // 0    200   400   600  800             time of inport sample
  //-800 -600  -400  -200  0               time of vector sample
  // 0    200   400   600  800             time of delayed vector
  //    -666      -333     0               time of out vector
  //                       0   333  666    time of out port

  // we use the delayed vector to remove the -0.8us (negative) delay from
  // the inport assignment
  y130 = nd130(N, D, sc_core::sc_time(DELAY_IN_SEC - 2.0e-6 / 3.0, sc_core::SC_SEC), x_vec0_delayed);
  y131 = nd131(N, D, sc_core::sc_time(DELAY_IN_SEC - 2.0e-6 / 3.0, sc_core::SC_SEC), x_vec1_delayed);
  y132 = nd132(N, D, sc_core::sc_time(DELAY_IN_SEC - 2.0e-6 / 3.0, sc_core::SC_SEC), x_vec2_delayed);
  y133 = nd133(N, D, sc_core::sc_time(DELAY_IN_SEC - 2.0e-6 / 3.0, sc_core::SC_SEC), x_vec3_delayed);
  y134 = nd134(N2, D2, sc_core::sc_time(DELAY_IN_SEC - 2.0e-6 / 3.0, sc_core::SC_SEC), x_vec4_delayed);

  // this assignment corresponds to a delay of 0.666us
  // -666 -> 0, -333 -> 333, 0 -> 666
  for (unsigned long i = 0; i < p.rate2; i++)
  {
    out_r_130[i] = y130(i);
    out_r_131[i] = y131(i);
    out_r_132[i] = y132(i);
    out_r_133[i] = y133(i);
    out_r_134[i] = y134(i);
  }

  // -> the overall delay will be 0.666us+0.83ms - 0.666us =0.83ms

  ///////////////////////////////////////////////////////////////////////////
  ///////////////////////////////////////////////////////////////////////////

  // 0     1     2     3    4              sample index
  // 0    200   400   600  800             time of inport sample
  //-800 -600  -400  -200  0               time of vector sample
  //                       0    333   666  time of outport sample

  // to achieve, that inport==outport we must delay the vector by 800ps
  // port out (rate=3) / vector in (rate=5)
  // a delay is required due the vector time ends at 0 and the port time at 1us/3*2
  out_r_140 = nd140(N, D, sc_core::sc_time(DELAY_IN_SEC + 0.8e-6, sc_core::SC_SEC), x_vec0);
  out_r_141 = nd141(N, D, sc_core::sc_time(DELAY_IN_SEC + 0.8e-6, sc_core::SC_SEC), x_vec1);
  out_r_142 = nd142(N, D, sc_core::sc_time(DELAY_IN_SEC + 0.8e-6, sc_core::SC_SEC), x_vec2);
  out_r_143 = nd143(N, D, sc_core::sc_time(DELAY_IN_SEC + 0.8e-6, sc_core::SC_SEC), x_vec3);
  out_r_144 = nd144(N2, D2, sc_core::sc_time(DELAY_IN_SEC + 0.8e-6, sc_core::SC_SEC), x_vec2);

  // vector out (rate=3) / port in (rate=5)

  sca_util::sca_vector<double> y150(p.rate2);
  sca_util::sca_vector<double> y151(p.rate2);
  sca_util::sca_vector<double> y152(p.rate2);
  sca_util::sca_vector<double> y153(p.rate2);
  sca_util::sca_vector<double> y154(p.rate2);

  // inport time stamps:     0 200 400 600 800
  // out vector timestamps: -666 -333 0 (init init 0) | 333 666
  y150 = nd150(N, D, sc_core::sc_time(DELAY_IN_SEC - 2.0e-6 / 3.0, sc_core::SC_SEC), in_r_0);
  y151 = nd151(N, D, sc_core::sc_time(DELAY_IN_SEC - 2.0e-6 / 3.0, sc_core::SC_SEC), in_r_1);
  y152 = nd152(N, D, sc_core::sc_time(DELAY_IN_SEC - 2.0e-6 / 3.0, sc_core::SC_SEC), in_r_2);
  y153 = nd153(N, D, sc_core::sc_time(DELAY_IN_SEC - 2.0e-6 / 3.0, sc_core::SC_SEC), in_r_3);
  y154 = nd154(N2, D2, sc_core::sc_time(DELAY_IN_SEC - 2.0e-6 / 3.0, sc_core::SC_SEC), in_r_2);

  // assignment corresponds to 0.666us delay
  // -666 -> 0, -333 -> 333, 0 -> 666
  for (unsigned long i = 0; i < p.rate2; i++)
  {
    out_r_150.write(y150(i), i);
    out_r_151.write(y151(i), i);
    out_r_152.write(y152(i), i);
    out_r_153.write(y153(i), i);
    out_r_154.write(y154(i), i);
  }

  // overall delay 0.666us+0.83ms-0.666us = 0.83ms

  ////////////////////////////////////////////////////////////////////

  // port out (rate=3) / port in (rate=5)
  out_r_160  = nd160(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_0);
  out_r_161  = nd161(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_1);
  out_r_162  = nd162(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_2);
  out_r_163  = nd163(N, D, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_3);
  out_r_164  = nd164(N2, D2, sc_core::sc_time(DELAY_IN_SEC, sc_core::SC_SEC), in_r_2);
}


bool check_val(double val, double e_val, double rel_err, double abs_err)
{
  if ( (std::fabs(val - e_val) > abs_err) && (std::fabs(val - e_val) / e_val > rel_err))
  {
    std::cout << " Expect value: " << e_val << " read value: " << val << "  diff: " << val - e_val << std::endl;
    return true;
  }

  return false;
}

int check_values(double time, double delay, double cval, double rampv, double t2_val, double sin_val,
                 double t3_val, double rel_err = 1e-10, double abs_err = 1e-10)
{
  double ctime = time-delay;
  double e_cval = 0.0;
  double e_rampv = 0.0;
  double e_t2_val = 0.0;
  double e_sin_val = 0.0;
  double e_t3_val = 0.0;

  if (ctime >= 0.0)
  {
    e_cval = 1.0;
    e_rampv = ctime;
    e_t2_val = 0.5 * ctime * ctime;
    e_sin_val = std::sin(2.0 * M_PI * 13.3e3 * ctime);
    e_t3_val = 0.5 / 3.0 * ctime * ctime * ctime;
  }
  else
  {
    return 0;
  }

  if (check_val(cval, e_cval, rel_err, abs_err))
  {
    return 1;
  }

  if (check_val(rampv, e_rampv, rel_err, abs_err))
  {
    return 2;
  }

  if (check_val(t2_val, e_t2_val, rel_err, abs_err))
  {
    return 3;
  }

  if (check_val(t3_val, e_t3_val, rel_err, abs_err))
  {
    return 5;
  }

  return 0;
}

bool check_file(std::string fname, const std::vector<double>& delays)
{
  std::ifstream fin;
  fin.open(fname.c_str());

  if (fin.fail())
  {
    std::ostringstream str;
    str << "Can't open file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }

  std::string line_in;
  unsigned long line_cnt = 0;
  if (!std::getline(fin, line_in))
  {
    std::ostringstream str;
    str << "Can't get header line from file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }
  line_cnt++;

  std::vector<std::string> names;
  std::istringstream nistr(line_in);

  std::string name;
  if (!(nistr >> name) && (name!="%time"))
  {
    std::ostringstream str;
    str << "Wrong header line in file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }

  while (nistr >> name)
  {
    names.push_back(name);
  }

  while (std::getline(fin, line_in))
  {
    std::istringstream istr(line_in);
    line_cnt++;

    double ctime;
    istr >> ctime;

    if (istr.fail())
    {
      std::ostringstream str;
      str << "Can't get time in file: " << fname << " line: " << line_cnt;
      SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
    }

    for (unsigned long i = 0; i < names.size(); i += 5)
    {
      double cval, rval, t2val, sinval, t3val;

      istr >> cval;
      istr >> rval;
      istr >> t2val;
      istr >> sinval;
      istr >> t3val;

      if (istr.fail())
      {
        std::ostringstream str;
        str << "Failed to read values from file: " << fname << " line: " << line_cnt;
        str << " for: " << names[i] << " " << names[i+1] << " "
            << names[i+2] << " " << names[i+3]<< " " << names[i+4];
        str << " at time: " << ctime;
        SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
      }

      int err = check_values(ctime, delays[i / 5], cval, rval, t2val, sinval, t3val);
      if (err != 0)
      {
        std::ostringstream str;
        str << "Wrong value in file: " << fname << " line: " << line_cnt;
        str << " for signal: " << names[i+err-1] << " at time " << ctime;
        str << " delay: " << delays[i / 5];
        SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
      }
    }
  }
  return false;
}

int sc_main(int argn, char* argc[])
{
  sc_core::sc_set_time_resolution(1.0, sc_core::SC_FS);

  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> s_out_const1, s_out_ramp1, s_out_t21, s_out_sin1, s_out_t31;
  sca_tdf::sca_signal<double> s_out_constr, s_out_rampr, s_out_t2r, s_out_sinr, s_out_t3r;

  sca_tdf::sca_signal<double> s_out_const_1, s_out_ramp_1, s_out_t2_1, s_out_sin_1, s_out_t3_1;
  sca_tdf::sca_signal<double> s_out_const_3, s_out_ramp_3, s_out_t2_3, s_out_sin_3, s_out_t3_3;
  sca_tdf::sca_signal<double> s_out_const_5, s_out_ramp_5, s_out_t2_5, s_out_sin_5, s_out_t3_5;
  sca_tdf::sca_signal<double> s_out_const_7, s_out_ramp_7, s_out_t2_7, s_out_sin_7, s_out_t3_7;

  sca_tdf::sca_signal<double> s_out_10, s_out_r_0;
  sca_tdf::sca_signal<double> s_out_11, s_out_r_1;
  sca_tdf::sca_signal<double> s_out_12, s_out_r_2;
  sca_tdf::sca_signal<double> s_out_13, s_out_r_3;
  sca_tdf::sca_signal<double> s_out_14, s_out_r_4;

  sca_tdf::sca_signal<double> s_out_20, s_out_40, s_out_60, s_out_80;
  sca_tdf::sca_signal<double> s_out_21, s_out_41, s_out_61, s_out_81;
  sca_tdf::sca_signal<double> s_out_22, s_out_42, s_out_62, s_out_82;
  sca_tdf::sca_signal<double> s_out_23, s_out_43, s_out_63, s_out_83;
  sca_tdf::sca_signal<double> s_out_24, s_out_44, s_out_64, s_out_84;

  sca_tdf::sca_signal<double> s_out_r90, s_out_r100, s_out_r110, s_out_r120;
  sca_tdf::sca_signal<double> s_out_r91, s_out_r101, s_out_r111, s_out_r121;
  sca_tdf::sca_signal<double> s_out_r92, s_out_r102, s_out_r112, s_out_r122;
  sca_tdf::sca_signal<double> s_out_r93, s_out_r103, s_out_r113, s_out_r123;
  sca_tdf::sca_signal<double> s_out_r94, s_out_r104, s_out_r114, s_out_r124;

  sca_tdf::sca_signal<double> s_out_r130, s_out_r140, s_out_r150, s_out_r160;
  sca_tdf::sca_signal<double> s_out_r131, s_out_r141, s_out_r151, s_out_r161;
  sca_tdf::sca_signal<double> s_out_r132, s_out_r142, s_out_r152, s_out_r162;
  sca_tdf::sca_signal<double> s_out_r133, s_out_r143, s_out_r153, s_out_r163;
  sca_tdf::sca_signal<double> s_out_r134, s_out_r144, s_out_r154, s_out_r164;

  // instantiate and connect components

  ltf_nd_src* i_src = new ltf_nd_src("i_src");
  i_src->outp_const_1(s_out_const1);
  i_src->outp_ramp_1(s_out_ramp1);
  i_src->outp_t2_1(s_out_t21);
  i_src->outp_sin_1(s_out_sin1);
  i_src->outp_t3_1(s_out_t31);

  i_src->outp_const_r(s_out_constr);
  i_src->outp_ramp_r(s_out_rampr);
  i_src->outp_t2_r(s_out_t2r);
  i_src->outp_sin_r(s_out_sinr);
  i_src->outp_t3_r(s_out_t3r);

  i_src->outp0(s_out_10);
  i_src->outp1(s_out_11);
  i_src->outp2(s_out_12);
  i_src->outp3(s_out_13);
  i_src->outp4(s_out_14);

  i_src->outpr0(s_out_r_0);
  i_src->outpr1(s_out_r_1);
  i_src->outpr2(s_out_r_2);
  i_src->outpr3(s_out_r_3);
  i_src->outpr4(s_out_r_4);

  ltf_nd_test* i_nd_test = new ltf_nd_test("i_nd_test");
  i_nd_test->in_const_1(s_out_const1);
  i_nd_test->in_ramp_1(s_out_ramp1);
  i_nd_test->in_t2_1(s_out_t21);
  i_nd_test->in_sin_1(s_out_sin1);

  i_nd_test->in_const_r(s_out_constr);
  i_nd_test->in_ramp_r(s_out_rampr);
  i_nd_test->in_t2_r(s_out_t2r);
  i_nd_test->in_sin_r(s_out_sinr);

  i_nd_test->in_10(s_out_10);
  i_nd_test->in_11(s_out_11);
  i_nd_test->in_12(s_out_12);
  i_nd_test->in_13(s_out_13);

  i_nd_test->in_r_0(s_out_r_0);
  i_nd_test->in_r_1(s_out_r_1);
  i_nd_test->in_r_2(s_out_r_2);
  i_nd_test->in_r_3(s_out_r_3);

  i_nd_test->out_const_1(s_out_const_1);
  i_nd_test->out_ramp_1(s_out_ramp_1);
  i_nd_test->out_t2_1(s_out_t2_1);
  i_nd_test->out_sin_1(s_out_sin_1);
  i_nd_test->out_t3_1(s_out_t3_1);

  i_nd_test->out_20(s_out_20);
  i_nd_test->out_21(s_out_21);
  i_nd_test->out_22(s_out_22);
  i_nd_test->out_23(s_out_23);
  i_nd_test->out_24(s_out_24);

  i_nd_test->out_const_3(s_out_const_3);
  i_nd_test->out_ramp_3(s_out_ramp_3);
  i_nd_test->out_t2_3(s_out_t2_3);
  i_nd_test->out_sin_3(s_out_sin_3);
  i_nd_test->out_t3_3(s_out_t3_3);

  i_nd_test->out_40(s_out_40);
  i_nd_test->out_41(s_out_41);
  i_nd_test->out_42(s_out_42);
  i_nd_test->out_43(s_out_43);
  i_nd_test->out_44(s_out_44);

  i_nd_test->out_const_5(s_out_const_5);
  i_nd_test->out_ramp_5(s_out_ramp_5);
  i_nd_test->out_t2_5(s_out_t2_5);
  i_nd_test->out_sin_5(s_out_sin_5);
  i_nd_test->out_t3_5(s_out_t3_5);

  i_nd_test->out_60(s_out_60);
  i_nd_test->out_61(s_out_61);
  i_nd_test->out_62(s_out_62);
  i_nd_test->out_63(s_out_63);
  i_nd_test->out_64(s_out_64);

  i_nd_test->out_const_7(s_out_const_7);
  i_nd_test->out_ramp_7(s_out_ramp_7);
  i_nd_test->out_t2_7(s_out_t2_7);
  i_nd_test->out_sin_7(s_out_sin_7);
  i_nd_test->out_t3_7(s_out_t3_7);

  i_nd_test->out_80(s_out_80);
  i_nd_test->out_81(s_out_81);
  i_nd_test->out_82(s_out_82);
  i_nd_test->out_83(s_out_83);
  i_nd_test->out_84(s_out_84);

  i_nd_test->out_r_90(s_out_r90);
  i_nd_test->out_r_91(s_out_r91);
  i_nd_test->out_r_92(s_out_r92);
  i_nd_test->out_r_93(s_out_r93);
  i_nd_test->out_r_94(s_out_r94);

  i_nd_test->out_r_100(s_out_r100);
  i_nd_test->out_r_101(s_out_r101);
  i_nd_test->out_r_102(s_out_r102);
  i_nd_test->out_r_103(s_out_r103);
  i_nd_test->out_r_104(s_out_r104);

  i_nd_test->out_r_110(s_out_r110);
  i_nd_test->out_r_111(s_out_r111);
  i_nd_test->out_r_112(s_out_r112);
  i_nd_test->out_r_113(s_out_r113);
  i_nd_test->out_r_114(s_out_r114);

  i_nd_test->out_r_120(s_out_r120);
  i_nd_test->out_r_121(s_out_r121);
  i_nd_test->out_r_122(s_out_r122);
  i_nd_test->out_r_123(s_out_r123);
  i_nd_test->out_r_124(s_out_r124);

  i_nd_test->out_r_130(s_out_r130);
  i_nd_test->out_r_131(s_out_r131);
  i_nd_test->out_r_132(s_out_r132);
  i_nd_test->out_r_133(s_out_r133);
  i_nd_test->out_r_134(s_out_r134);

  i_nd_test->out_r_140(s_out_r140);
  i_nd_test->out_r_141(s_out_r141);
  i_nd_test->out_r_142(s_out_r142);
  i_nd_test->out_r_143(s_out_r143);
  i_nd_test->out_r_144(s_out_r144);

  i_nd_test->out_r_150(s_out_r150);
  i_nd_test->out_r_151(s_out_r151);
  i_nd_test->out_r_152(s_out_r152);
  i_nd_test->out_r_153(s_out_r153);
  i_nd_test->out_r_154(s_out_r154);

  i_nd_test->out_r_160(s_out_r160);
  i_nd_test->out_r_161(s_out_r161);
  i_nd_test->out_r_162(s_out_r162);
  i_nd_test->out_r_163(s_out_r163);
  i_nd_test->out_r_164(s_out_r164);

  /////// Tracing -> for each rate a different file

  sca_util::sca_trace_file* tf1 = sca_util::sca_create_tabular_trace_file("tdf_nd_1_1.dat");

  sca_util::sca_trace(tf1, s_out_const1, "s_out_const1");
  sca_util::sca_trace(tf1, s_out_ramp1, "s_out_ramp1");
  sca_util::sca_trace(tf1, s_out_t21, "s_out_t21");
  sca_util::sca_trace(tf1, s_out_sin1, "s_out_sin1");
  sca_util::sca_trace(tf1, s_out_t31, "s_out_t31");

  sca_util::sca_trace(tf1, s_out_10, "s_out_10");
  sca_util::sca_trace(tf1, s_out_11, "s_out_11");
  sca_util::sca_trace(tf1, s_out_12, "s_out_12");
  sca_util::sca_trace(tf1, s_out_13, "s_out_13");
  sca_util::sca_trace(tf1, s_out_14, "s_out_14");

  sca_util::sca_trace(tf1, s_out_const_1, "s_out_const_1");
  sca_util::sca_trace(tf1, s_out_ramp_1, "s_out_ramp_1");
  sca_util::sca_trace(tf1, s_out_t2_1, "s_out_t2_1");
  sca_util::sca_trace(tf1, s_out_sin_1, "s_out_sin_1");
  sca_util::sca_trace(tf1, s_out_t3_1, "s_out_t3_1");

  sca_util::sca_trace(tf1, s_out_const_3, "s_out_const_3");
  sca_util::sca_trace(tf1, s_out_ramp_3, "s_out_ramp_3");
  sca_util::sca_trace(tf1, s_out_t2_3, "s_out_t2_3");
  sca_util::sca_trace(tf1, s_out_sin_3, "s_out_sin_3");
  sca_util::sca_trace(tf1, s_out_t3_3, "s_out_t3_3");

  sca_util::sca_trace(tf1, s_out_const_5, "s_out_const_5");
  sca_util::sca_trace(tf1, s_out_ramp_5, "s_out_ramp_5");
  sca_util::sca_trace(tf1, s_out_t2_5, "s_out_t2_5");
  sca_util::sca_trace(tf1, s_out_sin_5, "s_out_sin_5");
  sca_util::sca_trace(tf1, s_out_t3_5, "s_out_t3_5");

  sca_util::sca_trace(tf1, s_out_const_7, "s_out_const_7");
  sca_util::sca_trace(tf1, s_out_ramp_7, "s_out_ramp_7");
  sca_util::sca_trace(tf1, s_out_t2_7, "s_out_t2_7");
  sca_util::sca_trace(tf1, s_out_sin_7, "s_out_sin_7");
  sca_util::sca_trace(tf1, s_out_t3_7, "s_out_t3_7");

  sca_util::sca_trace(tf1, s_out_20, "s_out_20");
  sca_util::sca_trace(tf1, s_out_21, "s_out_21");
  sca_util::sca_trace(tf1, s_out_22, "s_out_22");
  sca_util::sca_trace(tf1, s_out_23, "s_out_23");
  sca_util::sca_trace(tf1, s_out_24, "s_out_24");

  sca_util::sca_trace(tf1, s_out_40, "s_out_40");
  sca_util::sca_trace(tf1, s_out_41, "s_out_41");
  sca_util::sca_trace(tf1, s_out_42, "s_out_42");
  sca_util::sca_trace(tf1, s_out_43, "s_out_43");
  sca_util::sca_trace(tf1, s_out_44, "s_out_44");

  sca_util::sca_trace(tf1, s_out_60, "s_out_60");
  sca_util::sca_trace(tf1, s_out_61, "s_out_61");
  sca_util::sca_trace(tf1, s_out_62, "s_out_62");
  sca_util::sca_trace(tf1, s_out_63, "s_out_63");
  sca_util::sca_trace(tf1, s_out_64, "s_out_64");

  sca_util::sca_trace(tf1, s_out_80, "s_out_80");
  sca_util::sca_trace(tf1, s_out_81, "s_out_81");
  sca_util::sca_trace(tf1, s_out_82, "s_out_82");
  sca_util::sca_trace(tf1, s_out_83, "s_out_83");
  sca_util::sca_trace(tf1, s_out_84, "s_out_84");

  sca_util::sca_trace_file* tfr = sca_util::sca_create_tabular_trace_file("tdf_nd_1_rate.dat");

  sca_util::sca_trace(tfr, s_out_constr, "s_out_constr");
  sca_util::sca_trace(tfr, s_out_rampr, "s_out_rampr");
  sca_util::sca_trace(tfr, s_out_t2r, "s_out_t2r");
  sca_util::sca_trace(tfr, s_out_sinr, "s_out_sinr");
  sca_util::sca_trace(tfr, s_out_t3r, "s_out_t3r");

  sca_util::sca_trace(tfr, s_out_r_0, "s_out_r_0");
  sca_util::sca_trace(tfr, s_out_r_1, "s_out_r_1");
  sca_util::sca_trace(tfr, s_out_r_2, "s_out_r_2");
  sca_util::sca_trace(tfr, s_out_r_3, "s_out_r_3");
  sca_util::sca_trace(tfr, s_out_r_4, "s_out_r_4");

  sca_util::sca_trace(tfr, s_out_r90, "s_out_r90");
  sca_util::sca_trace(tfr, s_out_r91, "s_out_r91");
  sca_util::sca_trace(tfr, s_out_r92, "s_out_r92");
  sca_util::sca_trace(tfr, s_out_r93, "s_out_r93");
  sca_util::sca_trace(tfr, s_out_r94, "s_out_r94");

  sca_util::sca_trace(tfr, s_out_r100, "s_out_r100");
  sca_util::sca_trace(tfr, s_out_r101, "s_out_r101");
  sca_util::sca_trace(tfr, s_out_r102, "s_out_r102");
  sca_util::sca_trace(tfr, s_out_r103, "s_out_r103");
  sca_util::sca_trace(tfr, s_out_r104, "s_out_r104");

  sca_util::sca_trace(tfr, s_out_r110, "s_out_r110");
  sca_util::sca_trace(tfr, s_out_r111, "s_out_r111");
  sca_util::sca_trace(tfr, s_out_r112, "s_out_r112");
  sca_util::sca_trace(tfr, s_out_r113, "s_out_r113");
  sca_util::sca_trace(tfr, s_out_r114, "s_out_r114");

  sca_util::sca_trace(tfr, s_out_r120, "s_out_r120");
  sca_util::sca_trace(tfr, s_out_r121, "s_out_r121");
  sca_util::sca_trace(tfr, s_out_r122, "s_out_r122");
  sca_util::sca_trace(tfr, s_out_r123, "s_out_r123");
  sca_util::sca_trace(tfr, s_out_r124, "s_out_r124");

  sca_util::sca_trace_file* tfr2 = sca_util::sca_create_tabular_trace_file("tdf_nd_1_rate2.dat");

  sca_util::sca_trace(tfr2, s_out_r130, "s_out_r130");
  sca_util::sca_trace(tfr2, s_out_r131, "s_out_r131");
  sca_util::sca_trace(tfr2, s_out_r132, "s_out_r132");
  sca_util::sca_trace(tfr2, s_out_r133, "s_out_r133");
  sca_util::sca_trace(tfr2, s_out_r134, "s_out_r134");

  sca_util::sca_trace(tfr2, s_out_r140, "s_out_r140");
  sca_util::sca_trace(tfr2, s_out_r141, "s_out_r141");
  sca_util::sca_trace(tfr2, s_out_r142, "s_out_r142");
  sca_util::sca_trace(tfr2, s_out_r143, "s_out_r143");
  sca_util::sca_trace(tfr2, s_out_r144, "s_out_r144");

  sca_util::sca_trace(tfr2, s_out_r150, "s_out_r150");
  sca_util::sca_trace(tfr2, s_out_r151, "s_out_r151");
  sca_util::sca_trace(tfr2, s_out_r152, "s_out_r152");
  sca_util::sca_trace(tfr2, s_out_r153, "s_out_r153");
  sca_util::sca_trace(tfr2, s_out_r154, "s_out_r154");

  sca_util::sca_trace(tfr2, s_out_r160, "s_out_r160");
  sca_util::sca_trace(tfr2, s_out_r161, "s_out_r161");
  sca_util::sca_trace(tfr2, s_out_r162, "s_out_r162");
  sca_util::sca_trace(tfr2, s_out_r163, "s_out_r163");
  sca_util::sca_trace(tfr2, s_out_r164, "s_out_r164");

  ////////////////////////////////////////////////

  sc_core::sc_start(2.0, sc_core::SC_MS); //start time domain simulation for 5ms

  sca_util::sca_close_tabular_trace_file(tf1);
  sca_util::sca_close_tabular_trace_file(tfr);
  sca_util::sca_close_tabular_trace_file(tfr2);  

  std::vector<double> delays;
  delays.resize(20, 0.0);

  for (unsigned long i = 2; i < 10; i++)
  {
    delays[i] = DELAY_IN_SEC;
  }

  // delay is not shown, due we propagate the first value to zero -> thus at time
  // zero the constant is valid -> the ramp starts at zero (begin of integration with 1)
  check_file("tdf_nd_1_1.dat", delays);

  delays.clear();
  delays.resize(20, 0.0);

  for (unsigned long i = 2; i < 6; i++)
  {
    delays[i] = DELAY_IN_SEC;
  }

  check_file("tdf_nd_1_rate.dat", delays);

  delays.resize(0);
  delays.resize(20, 0.0);

  for (unsigned long i = 0; i < 5; i++)
  {
    delays[i] = DELAY_IN_SEC;
  }

  check_file("tdf_nd_1_rate2.dat", delays);

  if (sc_core::sc_report_handler::get_count(sc_core::SC_ERROR) <= 0)
  {
    SC_REPORT_INFO("ND_DELAY_CHECK", "Test was successful");
  }

  TEST_LABEL_END;

  delete i_src, i_nd_test;
  
  return 0;
}
