/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  test02.cpp  - eln regression test with three tdf_rswitch and others

#include <systemc-ams>
#include "test_utilities.h"

SC_MODULE(test02)
{
  // port declaration
  sca_tdf::sca_in<double> inp1;
  sca_tdf::sca_in<bool>   inp2;

  // node declaration
  sca_eln::sca_node_ref n0; // reference node
  sca_eln::sca_node n1;
  sca_eln::sca_node n2;
  sca_eln::sca_node n3;
  sca_eln::sca_node n4;
  sca_eln::sca_node n5;

  struct params // parameter
  {
    params() // default for parameter
    {
    }
  };

  // component declaration
  sca_eln::sca_tdf_vsource *vin;
  sca_eln::sca_tdf_rswitch *tdf_rswitch1, *tdf_rswitch2, *tdf_rswitch3;
  sca_eln::sca_r           *r1, *r2;
  sca_eln::sca_c           *c1, *c2;

  test02( sc_core::sc_module_name nm, params pa = params() )
  : inp1("inp1"), inp2("inp2"), 
    n0("n0"), n1("n1"), n2("n2"), n3("n3"), n4("n4"), n5("n5"),
    p(pa)
  {
    architecture(); // generate netlist
  }

  ~test02();

 private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation

void test02::architecture()
{
  vin = new sca_eln::sca_tdf_vsource("vin");
  vin->inp(inp1);
  vin->n(n0);
  vin->p(n3);

  tdf_rswitch1 = new sca_eln::sca_tdf_rswitch("tdf_rswitch1");
  tdf_rswitch1->n(n0);
  tdf_rswitch1->p(n2);
  tdf_rswitch1->ctrl(inp2);
  tdf_rswitch1->ron = 2.0e3;
  tdf_rswitch1->roff = 1.0e3;
  tdf_rswitch1->off_state = false;

  tdf_rswitch2 = new sca_eln::sca_tdf_rswitch("tdf_rswitch2");
  tdf_rswitch2->n(n5);
  tdf_rswitch2->p(n4);
  tdf_rswitch2->ctrl(inp2);
  tdf_rswitch2->ron = 1.0e1;
  tdf_rswitch2->roff = 1.0e0;
  tdf_rswitch2->off_state = false;

  tdf_rswitch3 = new sca_eln::sca_tdf_rswitch("tdf_rswitch3");
  tdf_rswitch3->n(n1);
  tdf_rswitch3->p(n3);
  tdf_rswitch3->ctrl(inp2);
  tdf_rswitch3->ron = 1.0e0;
  tdf_rswitch3->roff = 1.0e2;
  tdf_rswitch3->off_state = false;

  r1 = new sca_eln::sca_r("r1");
  r1->value = 1.0;
  r1->n(n4);
  r1->p(n2);

  r2 = new sca_eln::sca_r("r2");
  r2->value = 5.0;
  r2->n(n2);
  r2->p(n1);

  c1 = new sca_eln::sca_c("c1");
  c1->value = 1.0e-2;
  c1->n(n4);
  c1->p(n2);

  c2 = new sca_eln::sca_c("c2");
  c2->value = 1.0e-4;
  c2->n(n0);
  c2->p(n1);
}

test02::~test02()
{
  delete vin, tdf_rswitch1, tdf_rswitch2, tdf_rswitch3;
  delete r1, r2, c1, c2;
}

void check_td_results(std::string fname)
{
  bool test_passed_with_success = true;
  std::ifstream tdf(fname.c_str());
  std::string line;
  long test_cnt = 0;

  getline(tdf, line);
  std::cout << "1..14" << std::endl;
  while (getline(tdf, line))
  {
    std::istringstream str(line);

    double time;
    str >> time;

    double i_vin, i1, i2, i3, i4, i5, i6, i7;
    str >> i_vin >> i1 >> i2 >> i3 >> i4 >> i5 >> i6 >> i7;

    if (time == 0.02)
    {
      if ( !(test_util::check_value(i5, 0.00779746373228)) )
      {
        std::cerr << "Error TD-check failed for i5 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(i6, 0.0)) )
      {
        std::cerr << "Error TD-check failed for i6 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(i7, 0.0216354894905)) )
      {
        std::cerr << "Error TD-check failed for i7 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;
    }

    if (time == 0.06)
    {
      if ( !(test_util::check_value(i5, 0.00499449457481)) )
      {
        std::cerr << "Error TD-check failed for i5 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(i6, 0.0)) )
      {
        std::cerr << "Error TD-check failed for i6 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(i7, -0.0139616224934)) )
      {
        std::cerr << "Error TD-check failed for i7 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;
    }

    if ( !( (test_util::check_value(i7, i2+i4, 1e-10, 1e-10) &&
        test_util::check_value(i2, i1+i3+i5, 1e-10, 1e-10) &&
        test_util::check_value(i6, i1+i3, 1e-10, 1e-10) &&
        test_util::check_value(-i_vin, i4+i5, 1e-10, 1e-10))) )
    {
      std::cerr << "Error TD-check failed for verification of currents at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;

  };

  if (test_passed_with_success) std::cout << "results ok." << std::endl;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> signal1;
  sca_tdf::sca_signal<bool>   signal2;

  test_util::const_src::params src_params;
  src_params.ampl = 10;

  test_util::const_src* src = new test_util::const_src("source", src_params);
  src->outp(signal1);

  test_util::pulse_src* swch = new test_util::pulse_src("sca_sdf_rswitch");
  swch->outp(signal2);

  test02* cir = new test02("circuit");
  cir->inp1(signal1);
  cir->inp2(signal2);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("output3.dat");
  sca_util::sca_trace(tf, cir->vin, "i_through_vin");
  sca_util::sca_trace(tf, cir->r1, "i1");
  sca_util::sca_trace(tf, cir->r2, "i2");
  sca_util::sca_trace(tf, cir->c1, "i3");
  sca_util::sca_trace(tf, cir->c2, "i4");
  sca_util::sca_trace(tf, cir->tdf_rswitch1, "i5");
  sca_util::sca_trace(tf, cir->tdf_rswitch2, "i6");
  sca_util::sca_trace(tf, cir->tdf_rswitch3, "i7");

  sc_core::sc_start(80.0, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  check_td_results("output3.dat");

  TEST_LABEL_END;

  delete src, swch, cir;

  return 0;
}
