/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  test01.cpp  - eln regression test
//  test01 with one tdf-capacitor et al.

#include <systemc-ams>
#include "test_utilities.h"

SC_MODULE(test01)
{
  // port declaration
  sca_tdf::sca_in<double> inp1;
  sca_tdf::sca_in<double> inp2;

  // node declaration
  sca_eln::sca_node n1;
  sca_eln::sca_node_ref  n2;   // reference node
  sca_eln::sca_node n3;

  // component declaration
  sca_eln::sca_tdf_isource *i_in;
  sca_eln::sca_c           *c1;
  sca_eln::sca_tdf_l       *tdf_l1;
  sca_eln::sca_r           *r1, *r_l;

  struct params // parameter
  {
    params() // default for parameter
    {
    }
  };

  test01( sc_core::sc_module_name nm, params pa = params() )
  : inp1("inp1"), inp2("inp2"),
    n1("n1"), n2("n2"), n3("n3"),
    p(pa)
  {
    architecture(); // generate netlist
  }

  ~test01();

 private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation

void test01::architecture()
{
  i_in = new sca_eln::sca_tdf_isource("i_in");
  i_in->inp(inp1);
  i_in->p(n1);
  i_in->n(n2);

  tdf_l1 = new sca_eln::sca_tdf_l("tdf_l1");
  tdf_l1->scale = 50e-3;
  tdf_l1->n(n3);
  tdf_l1->p(n1);
  tdf_l1->inp(inp2);

  c1 = new sca_eln::sca_c("c1");
  c1->value = 200e-9;
  c1->n(n2);
  c1->p(n1);

  r_l = new sca_eln::sca_r("r_l");
  r_l->value = 1.0e-1;
  r_l->n(n2);
  r_l->p(n3);

  r1 = new sca_eln::sca_r("r1");
  r1->value = 1.25e3;
  r1->n(n2);
  r1->p(n1);
}

test01::~test01()
{
  delete i_in, tdf_l1, c1, r_l, r1;
}

void check_td_results(std::string fname)
{
  bool test_passed_with_success = true;
  std::ifstream tdf(fname.c_str());
  std::string line;
  long test_cnt = 0;

  getline(tdf, line);
  std::cout << "1..4" << std::endl;
  while (getline(tdf, line))
  {
    std::istringstream str(line);

    double time;
    str >> time;

    double node1, node2, node3, i_in, i1, i2, i3, i4;
    str >> node1 >> node2 >> node3 >> i_in >> i1 >> i2 >> i3 >> i4;

    if (time == 0.039)
    {
      if ( !(test_util::check_value(i3, -0.997457699302)))
      {
        std::cerr << "Error TD-check failed for i3 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(-i_in, i1+i3+i4) && test_util::check_value(-i_in, i1+i2+i4)))
      {
        std::cerr << "Error TD-check failed for verification of currents at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;
    }

    if (time == 0.074)
    {
      if ( !(test_util::check_value(i3, -0.999743070407)))
      {
        std::cerr << "Error TD-check failed for i3 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(-i_in, i1+i3+i4) && test_util::check_value(-i_in, i1+i2+i4)))
      {
        std::cerr << "Error TD-check failed for verification of currents at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;
    }
  };

  if (test_passed_with_success) std::cout << "results ok." << std::endl;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> signal1;
  sca_tdf::sca_signal<double> signal2;

  test_util::const_src::params src_params1;
  src_params1.ampl = 1.0;
  src_params1.sample_time = sc_core::sc_time(1.0, sc_core::SC_MS);

  test_util::const_src* src1 = new test_util::const_src("source1", src_params1);
  src1->outp(signal1);

  test_util::sin_src::params src_params2;
  src_params2.ampl = 0.5;
  src_params2.freq = 1000;
  src_params2.offset = 1.0;
  src_params2.sample_time = sc_core::sc_time(1.0, sc_core::SC_MS);

  test_util::sin_src* src2 = new test_util::sin_src("source2", src_params2);
  src2->outp(signal2);

  test01* cir = new test01("test01");
  cir->inp1(signal1);
  cir->inp2(signal2);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("output4.dat");
  sca_util::sca_trace(tf, cir->n1, "node1");
  sca_util::sca_trace(tf, cir->n2, "node2");
  sca_util::sca_trace(tf, cir->n3, "node3");
  sca_util::sca_trace(tf, cir->i_in, "i_in");
  sca_util::sca_trace(tf, cir->r1, "i1");
  sca_util::sca_trace(tf, cir->r_l, "i2");
  sca_util::sca_trace(tf, cir->tdf_l1, "i3");
  sca_util::sca_trace(tf, cir->c1, "i4");

  sc_core::sc_start(80.0, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  check_td_results("output4.dat");

  TEST_LABEL_END;

  delete src1, src2, cir;

  return 0;
}
