/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// Current Source test. Variable Capacitance

#include <systemc-ams>
#include "test_utilities.h"

SC_MODULE(test05)
{
  sc_core::sc_in<double> inp1;

  sca_eln::sca_node n1;
  sca_eln::sca_node n2;
  sca_eln::sca_node_ref n_gnd;

  sca_eln::sca_isource *src1;
  sca_eln::sca_r       *r1;
  sca_eln::sca_de_c    *var_c1;

  struct params
  {
    params()
    {
    }
  };

  test05( sc_core::sc_module_name nem, params pa = params() )
  : n1("n1"), n2("n2"), n_gnd("n_gnd"), p(pa)
  {
    architecture(); // generate netlist
  }

  ~test05();

private:
  params p;
  void architecture();
};

void test05::architecture() // netlist implementation
{
  src1 = new sca_eln::sca_isource("src1");
  src1->amplitude = 0;
  src1->frequency = 0;
  src1->offset = 0.005;
  src1->n(n1);
  src1->p(n_gnd);

  r1 = new sca_eln::sca_r("r1");
  r1->p(n1);
  r1->n(n2);
  r1->value = 1.0e3;
  r1->set_timestep(1.0, sc_core::SC_US);

  var_c1 = new sca_eln::sca_de_c("var_c1");
  var_c1->scale = 1.0e-6;
  var_c1->p(n2);
  var_c1->inp(inp1);
  var_c1->n(n_gnd);
}

test05::~test05()
{
  delete src1, r1, var_c1;
}

bool check_val_local(double val1, double val2, double abs_err = 1e-10, double rel_err = 1e-3)
{
  if ( (std::fabs(val1) < abs_err) || (std::fabs(val2) < abs_err) )
  {
    if (std::fabs(val1 - val2) > abs_err) return false;
  }
  else
  {
    if (std::fabs((val1 - val2) / (val1 + val2) / 2.0) > rel_err)
    {
      return false;
    }
  }
  return true;
}

void check_td_results(std::string fname)
{
  bool test_passed_with_success = true;
  std::ifstream tdf(fname.c_str());
  std::string line;

  getline(tdf, line);
  while (getline(tdf, line))
  {
    std::istringstream str(line);

    double time;
    str >> time;

    double Ic;
    str >> Ic;

    double Vc;
    str >> Vc;

    double theory;
    double m = 1000 * 1.0e-6;
    double b = 2 * 1.0e-6;

    theory = (Ic / m) - ((Ic * b / m) * (1 / ((m * time) + b)));

    if (time >= 1e-6)
    {
      if (!check_val_local(Vc, theory))
      {
        std::cerr << "Error  " << std::endl;
        std::cout << "Time " << time << std::endl;
        std::cout << "Expected " << theory << " Obtained " << Vc << std::endl;
        test_passed_with_success = false;
      }
    }
  };
  if (test_passed_with_success) std::cout  << "Trace comparison check ok" << std::endl;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> signal1;
  sc_core::sc_signal<double> signal2;

  test_util::conv_tdf2de* conver = new test_util::conv_tdf2de("conver");
  conver->tdf_i(signal1);
  conver-> sc_o(signal2);

  test_util::ramp::params ramp_params;
  ramp_params.inclination = 1000.0;
  ramp_params.cutoff = 2.0;
  ramp_params.sample_time = sc_core::sc_time(1.0, sc_core::SC_US);

  test_util::ramp* ramp2 = new test_util::ramp("ramp", ramp_params);
  ramp2->output(signal1);

  test05* cir = new test05("circuit");
  cir->inp1(signal2);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("test05");
  sca_util::sca_trace(tf, cir->r1, "R");
  sca_util::sca_trace(tf, cir->n2, "Vc");
  sca_util::sca_trace(tf, cir->inp1, "imp");

  sc_core::sc_start(8, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  check_td_results("test05.dat");

  TEST_LABEL_END;

  delete conver, ramp2, cir;

  return 0;
}
