/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  set_sink_occ_change_sink.cpp
//    -- Test module set_timestep() in the context of change_attributes() after
//      set_timestep() on module's TDF-CT_CUT out port in set_attributes().

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(src)
{
  sca_tdf::sca_out<double> out;

  SCA_CTOR(src)
  : out("out"), v(0.0)
  {}

  void set_attributes()
  {
    accept_attribute_changes();
    this->set_timestep( 2.0, sc_core::SC_US );
  }

  void processing()
  {
    out.write(v);
    v = v + 1.0;
  }

private:
  double v;
};


SCA_TDF_MODULE(passthrough)
{
  sca_tdf::sca_in<double> inp;
  sca_tdf::sca_out<double, sca_tdf::SCA_CT_CUT> outcc;

  SCA_CTOR(passthrough)
  : inp("inp"), outcc("outcc"), v(0.0), period(10)
  {}

  void set_attributes()
  {
    does_attribute_changes();
    accept_attribute_changes();
    outcc.set_timestep( 2.0, sc_core::SC_US );
    outcc.set_delay(1);
  }

  void initialize()
  {
    outcc.initialize(0, 0);
  }

  void processing()
  {
    v = inp.read();
    outcc.write(v);
  }

  void change_attributes()
  {
    if ( !( (int)v % period) )
    {
      this->set_timestep( ( (int)v / period) + 1, sc_core::SC_US );
    }
  }

private:
  double v;
  int period;
};

SCA_TDF_MODULE(sink)
{
  sca_tdf::sca_in<double> inp;

  SCA_CTOR(sink)
  : inp("inp"), v(0.0)
  {}

  void set_attributes()
  {
    inp.set_timestep( 0.5, sc_core::SC_US );
  }

  void processing()
  {
    v = inp.read();
  }

private:
  double v;
};

int sc_main(int argc, char* argv[])
{
  using namespace sc_core;

  TEST_LABEL_START;

  sca_tdf::sca_signal <double> sig1("sig1");
  sca_tdf::sca_signal <double> sig2("sig2");

  src src1("src1");
  src1.out(sig1);

  passthrough passthrough1("passthrough1");
  passthrough1.inp(sig1);
  passthrough1.outcc(sig2);

  sink sink1("sink1");
  sink1.inp(sig2);

  sca_util::sca_trace_file* atf = sca_util::sca_create_tabular_trace_file("set_sink_occ_change_sink");
  sca_util::sca_trace(atf, sig1, "sig1");
  sca_util::sca_trace(atf, sig2, "sig2");

  atf->set_mode( sca_util::sca_multirate( sca_util::SCA_DONT_INTERPOLATE ) );

  sc_time tstop(100.0, SC_US);

  try
  {
    sc_start(tstop);
  }
  catch (const std::exception& e)
  {
    std::cerr << e.what() << std::endl;
  }

  sc_stop();

  sca_util::sca_close_tabular_trace_file(atf);

  test_util::check_results("set_sink_occ_change_sink", 2);

  TEST_LABEL_END;

  return sc_report_handler::get_count(SC_ERROR);
}

// vim: expandtab : tabstop=2 : softtabstop=2 : shiftwidth=2
