/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

/*
 *   assignment to out via operator using calculate()
 *
 *   Tests:
 *           single rate out / single rate in  (vector,vector,?in?)
 *                                             scalar out / scalar in
 *                                             port out   / scalar in
 *                                             scalar out / port in
 *                                             port out   / port in
 *
 *           single rate out / multi rate in  (vector,vector,?in?)
 *                                             scalar out / vector in
 *                                             port out   / vector in
 *                                             scalar out / port in
 *                                             port out   / port in
 *
 *           multi rate out / single rate in  (vector,vector, sc_time,?in?)
 *           //delay is required due causality (sc_time argument)
 *                                             vector out / scalar in
 *                                             port out   / scalar in
 *                                             vector out / port in
 *                                             port out   / port in
 *
 *           multi rate out / multi rate in  (vector,vector,?in?)
 *                                             vector out / vector in
 *                                             port out   / vector in
 *                                             vector out / port in
 *                                             port out   / port in
 *
 */


#include <systemc-ams>
#include "test_utilities.h"

#include <string>

///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(ltf_nd_src)
{
  // single rate
  sca_tdf::sca_out<double> outp_const_1; // constant value (1.0)
  sca_tdf::sca_out<double> outp_ramp_1;  // ramp  y=t
  sca_tdf::sca_out<double> outp_t2_1;    // y=0.5*t*t
  sca_tdf::sca_out<double> outp_sin_1;
  sca_tdf::sca_out<double> outp_t3_1;    // y=0.5/3*t*t*t

  // single rate
  sca_tdf::sca_out<double> outp0;
  sca_tdf::sca_out<double> outp1;
  sca_tdf::sca_out<double> outp2;
  sca_tdf::sca_out<double> outp3;
  sca_tdf::sca_out<double> outp4;

  struct params
  {
    int rate;
    sca_core::sca_time sample_time;
    double f_sin;

    params()
    {
      sample_time = sca_core::sca_time(1.0, sc_core::SC_US);
      f_sin = 13.3e3;
    }
  };

  void set_attributes()
  {
    set_timestep(p.sample_time);

    does_attribute_changes();
  }

  void initialize();
  void processing();

  void change_attributes()
  {
    switch(activation)
    {
      case 0: // mstep=1ms
        break;

      case 5:
        break;

      case 6:
        break;

      case 7:
        break;

      case 508: // change timestep to 2sec and calculate to end
        set_timestep(2.0, sc_core::SC_US);
        break;

      case 509:
        break;

      case 510:
        break;

      case 511:
        break;

      case 512:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 513:
        break;

      case 514:  // 3 zero timestep module iterations, calculation to end
      case 515:
      case 516:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 517: // 4 zero timestep
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 518: // module timestep 1sec 4 zero tstep iterations
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 519: // calculate to end
        break;

      case 520: // 2 zero module timestep iterations
      case 521:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 522:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 523:
      case 524:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 525:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 526:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 527:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 528:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 529:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 530:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 531: // reset to begin of case 28
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 532:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 533: // normal full timestep
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 534: // reset to tstep before last timestep (before last iteration of 32)
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 535: // repeat as normal full timestep
        request_next_activation(0.0, sc_core::SC_SEC);
        break;

      case 600:
        sc_core::sc_stop();
        break;

      default: break;
    }
  }

  ltf_nd_src(sc_core::sc_module_name nm, params pa = params())
  : outp_const_1("outp_const_1"), outp_ramp_1("outp_ramp_1"), 
    outp_t2_1("outp_t2_1"), outp_sin_1("outp_sin_1"), outp_t3_1("outp_t3_1"),
    outp0("outp0"), outp1("outp1"), outp2("outp2"), outp3("outp3"), outp4("outp4"),
    nd10("nd10"), nd11("nd11"), nd12("nd12"), nd13("nd13"), nd14("nd14"),
    nd30("nd30"), nd31("nd31"), nd32("nd32"), nd33("nd33"), nd34("nd34"),
    activation(0), p(pa)
  {
  }

 private:
  sca_tdf::sca_ltf_nd  nd10, nd11, nd12, nd13, nd14;
  sca_tdf::sca_ltf_nd  nd30, nd31, nd32, nd33, nd34;

  sca_util::sca_vector<double> N1, D1;
  sca_util::sca_vector<double> N2, D2;
  sca_util::sca_vector<double> N3, D3;
  sca_util::sca_vector<double> N4, D4;

  int activation;
  params p;
};

void ltf_nd_src::initialize()
{
  N1(0) = 1.0;
  D1(0) = 1.0;  // y=x

  N2(0) = 1.0;
  D2(1) = 1.0;  // y=t*x - x=const=1.0

  N3(0) = 1.0;
  D3(2) = 1.0;  // y=0.5*t*t*x - x=const=1.0

  N4(0) = 1.0;
  D4(3) = 1.0;  // y=0.5/3.0*t*t*t*x - x=const=1.0
}

// time domain implementation
void ltf_nd_src::processing()
{
  double inp;
  inp = std::sin(2.0 * M_PI * p.f_sin * get_time().to_seconds());

  // output to scalar port sca_tdf::sca_out<double>
  // input double
  outp0 = nd10(N1, D1, 1.0); // single rate
  outp1 = nd11(N2, D2, 1.0);
  outp2 = nd12(N3, D3, 1.0);
  outp3 = nd13(N1, D1, inp);
  outp4 = nd14(N4, D4, 1.0);

  double outv0, outv1, outv2, outv3, outv4;

  // output to double
  // input double
  outv0 = nd30(N1, D1, 1.0);
  outv1 = nd31(N2, D2, 1.0);
  outv2 = nd32(N3, D3, 1.0);
  outv3 = nd33(N1, D1, inp);
  outv4 = nd34(N4, D4, 1.0);

  outp_const_1 = outv0;
  outp_ramp_1  = outv1;
  outp_t2_1    = outv2;
  outp_sin_1   = outv3;
  outp_t3_1    = outv4;

  activation++;
}

////////////////////////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE (ltf_nd_test)
{
  // single rate
  sca_tdf::sca_in<double> in_const_1; // constant value (1.0)
  sca_tdf::sca_in<double> in_ramp_1;  // ramp  y=t
  sca_tdf::sca_in<double> in_t2_1;    // y=0.5*t*t
  sca_tdf::sca_in<double> in_sin_1;

  sca_tdf::sca_out<double> out_const_1;
  sca_tdf::sca_out<double> out_ramp_1;
  sca_tdf::sca_out<double> out_t2_1;
  sca_tdf::sca_out<double> out_sin_1;
  sca_tdf::sca_out<double> out_t3_1;

  sca_tdf::sca_out<double> out_const_3;
  sca_tdf::sca_out<double> out_ramp_3;
  sca_tdf::sca_out<double> out_t2_3;
  sca_tdf::sca_out<double> out_sin_3;
  sca_tdf::sca_out<double> out_t3_3;

  sca_tdf::sca_out<double> out_const_5;
  sca_tdf::sca_out<double> out_ramp_5;
  sca_tdf::sca_out<double> out_t2_5;
  sca_tdf::sca_out<double> out_sin_5;
  sca_tdf::sca_out<double> out_t3_5;

  // single rate
  sca_tdf::sca_in<double> in_10;
  sca_tdf::sca_in<double> in_11;
  sca_tdf::sca_in<double> in_12;
  sca_tdf::sca_in<double> in_13;

  sca_tdf::sca_out<double> out_20;
  sca_tdf::sca_out<double> out_21;
  sca_tdf::sca_out<double> out_22;
  sca_tdf::sca_out<double> out_23;
  sca_tdf::sca_out<double> out_24;

  sca_tdf::sca_out<double> out_40;
  sca_tdf::sca_out<double> out_41;
  sca_tdf::sca_out<double> out_42;
  sca_tdf::sca_out<double> out_43;
  sca_tdf::sca_out<double> out_44;

  sca_tdf::sca_out<double> out_60;
  sca_tdf::sca_out<double> out_61;
  sca_tdf::sca_out<double> out_62;
  sca_tdf::sca_out<double> out_63;
  sca_tdf::sca_out<double> out_64;

  sca_tdf::sca_out<double> out_70;
  sca_tdf::sca_out<double> out_71;
  sca_tdf::sca_out<double> out_72;
  sca_tdf::sca_out<double> out_73;
  sca_tdf::sca_out<double> out_74;

  struct params // parameter
  {
    unsigned long vlength;
    unsigned long vlength2;

    params() // default for parameter
    {
      vlength = 5;
      vlength2 = 3;
    }
  };

  void initialize();     // initialization
  void processing();     // time domain processing method
  void set_attributes();

  // constructor
  ltf_nd_test(sc_core::sc_module_name nm, params pa = params()) 
  : in_const_1("in_const_1"), in_ramp_1("in_ramp_1"), in_t2_1("in_t2_1"), in_sin_1("in_sin_1"),
    out_const_1("out_const_1"), out_ramp_1("out_ramp_1"), out_t2_1("out_t2_1"), out_sin_1("out_sin_1"), out_t3_1("out_t3_1"),
    out_const_3("out_const_3"), out_ramp_3("out_ramp_3"), out_t2_3("out_t2_3"), out_sin_3("out_sin_3"), out_t3_3("out_t3_3"),
    out_const_5("out_const_5"), out_ramp_5("out_ramp_5"), out_t2_5("out_t2_5"), out_sin_5("out_sin_5"), out_t3_5("out_t3_5"),
    in_10("in_10"), in_11("in_11"), in_12("in_12"), in_13("in_13"),
    out_20("out_20"), out_21("out_21"), out_22("out_22"), out_23("out_23"), out_24("out_24"),
    out_40("out_40"), out_41("out_41"), out_42("out_42"), out_43("out_43"), out_44("out_44"),
    out_60("out_60"), out_61("out_61"), out_62("out_62"), out_63("out_63"), out_64("out_64"),
    out_70("out_70"), out_71("out_71"), out_72("out_72"), out_73("out_73"), out_74("out_74"),
    nd10("nd10"), nd11("nd11"), nd12("nd12"), nd13("nd13"), nd14("nd14"),
    nd20("nd20"), nd21("nd21"), nd22("nd22"), nd23("nd23"), nd24("nd24"),
    nd30("nd30"), nd31("nd31"), nd32("nd32"), nd33("nd33"), nd34("nd34"),
    nd40("nd40"), nd41("nd41"), nd42("nd42"), nd43("nd43"), nd44("nd44"),
    nd50("nd50"), nd51("nd51"), nd52("nd52"), nd53("nd53"), nd54("nd54"),
    nd60("nd60"), nd61("nd61"), nd62("nd62"), nd63("nd63"), nd64("nd64"),
    nd70("nd70"), nd71("nd71"), nd72("nd72"), nd73("nd73"), nd74("nd74"),
    nd_vec0("nd_vec0"), nd_vec1("nd_vec1"), nd_vec2("nd_vec2"), nd_vec3("nd_vec3"), nd_vec4("nd_vec4"),
    p(pa)
  {}

  // definition of local variables
private:
  sca_tdf::sca_ltf_nd nd10, nd11, nd12, nd13, nd14;
  sca_tdf::sca_ltf_nd nd20, nd21, nd22, nd23, nd24;
  sca_tdf::sca_ltf_nd nd30, nd31, nd32, nd33, nd34;
  sca_tdf::sca_ltf_nd nd40, nd41, nd42, nd43, nd44;
  sca_tdf::sca_ltf_nd nd50, nd51, nd52, nd53, nd54;
  sca_tdf::sca_ltf_nd nd60, nd61, nd62, nd63, nd64;
  sca_tdf::sca_ltf_nd nd70, nd71, nd72, nd73, nd74;

  sca_tdf::sca_ltf_nd nd_vec0, nd_vec1, nd_vec2, nd_vec3, nd_vec4;
  sca_util::sca_vector<double> N, D, N2, D2;
  params p;
};

void ltf_nd_test::set_attributes()
{
  nd10.set_max_delay(6.0, sc_core::SC_US);
  nd11.set_max_delay(6.0, sc_core::SC_US);
  nd12.set_max_delay(3.3, sc_core::SC_US);
  nd13.set_max_delay(6.0, sc_core::SC_US);
  nd14.set_max_delay(6.0, sc_core::SC_US);

  nd20.set_max_delay(6.0, sc_core::SC_US);
  nd21.set_max_delay(6.0, sc_core::SC_US);
  nd22.set_max_delay(3.3, sc_core::SC_US);
  nd23.set_max_delay(6.0, sc_core::SC_US);
  nd24.set_max_delay(6.0, sc_core::SC_US);

  nd30.set_max_delay(6.0, sc_core::SC_US);
  nd31.set_max_delay(6.0, sc_core::SC_US);
  nd32.set_max_delay(6.0, sc_core::SC_US);
  nd33.set_max_delay(6.0, sc_core::SC_US);
  nd34.set_max_delay(6.0, sc_core::SC_US);

  nd40.set_max_delay(6.0, sc_core::SC_US);
  nd41.set_max_delay(6.0, sc_core::SC_US);
  nd42.set_max_delay(6.0, sc_core::SC_US);
  nd43.set_max_delay(3.3, sc_core::SC_US);
  nd44.set_max_delay(6.0, sc_core::SC_US);

  nd50.set_max_delay(6.0, sc_core::SC_US);
  nd51.set_max_delay(6.0, sc_core::SC_US);
  nd52.set_max_delay(3.3, sc_core::SC_US);
  nd53.set_max_delay(6.0, sc_core::SC_US);
  nd54.set_max_delay(6.0, sc_core::SC_US);

  nd60.set_max_delay(6.0, sc_core::SC_US);
  nd61.set_max_delay(6.0, sc_core::SC_US);
  nd62.set_max_delay(3.3, sc_core::SC_US);
  nd63.set_max_delay(6.0, sc_core::SC_US);
  nd64.set_max_delay(6.0, sc_core::SC_US);

  nd70.set_max_delay(6.0, sc_core::SC_US);
  nd71.set_max_delay(6.0, sc_core::SC_US);
  nd72.set_max_delay(3.3, sc_core::SC_US);
  nd73.set_max_delay(6.0, sc_core::SC_US);
  nd74.set_max_delay(6.0, sc_core::SC_US);

  accept_attribute_changes();
}

void ltf_nd_test::initialize()
{
  N(0) = 1.0; // 1.0/1.0
  D(0) = 1.0;

  N2(0) = 1.0;
  D2(1) = 1.0;  // H(s)=1/s
}

// time domain implementation
void ltf_nd_test::processing()
{
  double x0;
  double x1;
  double x2;
  double x3;

  x0 = in_const_1.read();
  x1 = in_ramp_1.read();
  x2 = in_t2_1.read();
  x3 = in_sin_1.read();

  // create vector input
  sca_util::sca_vector<double> x_vec0(p.vlength);
  sca_util::sca_vector<double> x_vec1(p.vlength);
  sca_util::sca_vector<double> x_vec2(p.vlength);
  sca_util::sca_vector<double> x_vec3(p.vlength);
  sca_util::sca_vector<double> x_vec4(p.vlength);

  x_vec0 = nd_vec0(N, D, x0);
  x_vec1 = nd_vec1(N, D, x1);
  x_vec2 = nd_vec2(N, D, x2);
  x_vec3 = nd_vec3(N, D, x3);
  x_vec4 = nd_vec4(N, D, x2);

  ////////// single rate in / single rate out //////////////

  // double out / double in
  const double y10 = nd10.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x0);
  const double y11 = nd11.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x1);
  const double y12 = nd12.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x2);
  const double y13 = nd13.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x3);
  const double y14 = nd14.calculate(N2, D2, sca_core::sca_time(3.3, sc_core::SC_US), x2);

  out_const_1 = y10;
  out_ramp_1  = y11;
  out_t2_1    = y12;
  out_sin_1   = y13;
  out_t3_1    = y14;

  // port out / double in
  out_20 = nd20.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x0);
  out_21 = nd21.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x1);
  out_22 = nd22.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x2);
  out_23 = nd23.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x3);
  out_24 = nd24.calculate(N2, D2, sca_core::sca_time(3.3, sc_core::SC_US), x2);

  // double out / port in
  const double y30 = nd30.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), in_10);
  const double y31 = nd31.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), in_11);
  const double y32 = nd32.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), in_12);
  const double y33 = nd33.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), in_13);
  const double y34 = nd34.calculate(N2, D2, sca_core::sca_time(3.3, sc_core::SC_US), in_12);

  out_const_3 = y30;
  out_ramp_3  = y31;
  out_t2_3    = y32;
  out_sin_3   = y33;
  out_t3_3    = y34;

  // port out / port in
  out_40 = nd40.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), in_10);
  out_41 = nd41.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), in_11);
  out_42 = nd42.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), in_12);
  out_43 = nd43.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), in_13);
  out_44 = nd44.calculate(N2, D2, sca_core::sca_time(3.3, sc_core::SC_US), in_12);

  // double out / vector in
  // use delayed vector, due the last vector value corresponds to the out port value
  const double y50 = nd50.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x_vec0);
  const double y51 = nd51.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x_vec1);
  const double y52 = nd52.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x_vec2);
  const double y53 = nd53.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x_vec3);
  const double y54 = nd54.calculate(N2, D2, sca_core::sca_time(3.3, sc_core::SC_US), x_vec4);

  out_const_5 = y50;
  out_ramp_5  = y51;
  out_t2_5    = y52;
  out_sin_5   = y53;
  out_t3_5    = y54;

  ////////////////////////////////////////////////////////////////////////

  // port out / vector in
  out_60 = nd60.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x_vec0);
  out_61 = nd61.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x_vec1);
  out_62 = nd62.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x_vec2);
  out_63 = nd63.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x_vec3);
  out_64 = nd64.calculate(N2, D2, sca_core::sca_time(3.3, sc_core::SC_US), x_vec4);

  // create vector input
  sca_util::sca_vector<double> x_vec0_out(p.vlength2);
  sca_util::sca_vector<double> x_vec1_out(p.vlength2);
  sca_util::sca_vector<double> x_vec2_out(p.vlength2);
  sca_util::sca_vector<double> x_vec3_out(p.vlength2);
  sca_util::sca_vector<double> x_vec4_out(p.vlength2);

  // vector out / vector in
  x_vec0_out = nd70.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x_vec0);
  x_vec1_out = nd71.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x_vec1);
  x_vec2_out = nd72.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x_vec2);
  x_vec3_out = nd73.calculate(N, D, sca_core::sca_time(3.3, sc_core::SC_US), x_vec3);
  x_vec4_out = nd74.calculate(N2, D2, sca_core::sca_time(3.3, sc_core::SC_US), x_vec4);

  out_70 = x_vec0_out(p.vlength2 - 1);
  out_71 = x_vec1_out(p.vlength2 - 1);
  out_72 = x_vec2_out(p.vlength2 - 1);
  out_73 = x_vec3_out(p.vlength2 - 1);
  out_74 = x_vec4_out(p.vlength2 - 1);
}

bool check_val(double val, double e_val, double rel_err, double abs_err)
{
  if ( (std::fabs(val - e_val) > abs_err) && (std::fabs(val - e_val) / e_val > rel_err))
  {
    std::cout << " Expect value: " << e_val << " read value: " << val << "  diff: " << val - e_val << std::endl;
    return true;
  }

  return false;
}

int check_values(double time, double delay, double cval, double rampv, double t2_val, double sin_val,
                 double t3_val, double rel_err = 1e-10, double abs_err = 1e-10)
{
  double ctime = time-delay;
  double e_cval = 0.0;
  double e_rampv = 0.0;
  double e_t2_val = 0.0;
  double e_sin_val = 0.0;
  double e_t3_val = 0.0;

  if (ctime >= 0.0)
  {
    e_cval = 1.0;
    e_rampv = ctime;
    e_t2_val = 0.5 * ctime * ctime;
    e_sin_val = std::sin(2.0 * M_PI * 13.3e3 * ctime);
    e_t3_val = 0.5 / 3.0 * ctime * ctime * ctime;
  }
  else
  {
    return 0;
  }

  if (check_val(cval, e_cval, rel_err, abs_err))
  {
    return 1;
  }

  if (check_val(rampv, e_rampv, rel_err, abs_err))
  {
    return 2;
  }

  if (check_val(t2_val, e_t2_val, rel_err, abs_err))
  {
    return 3;
  }

  if (check_val(t3_val, e_t3_val, rel_err, abs_err))
  {
    return 5;
  }
  return 0;
}

bool check_file(std::string fname, const std::vector<double>& delays)
{
  std::ifstream fin;
  fin.open(fname.c_str());

  if (fin.fail())
  {
    std::ostringstream str;
    str << "Can't open file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }

  std::string line_in;
  unsigned long line_cnt = 0;
  if (!std::getline(fin, line_in))
  {
    std::ostringstream str;
    str << "Can't get header line from file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }
  line_cnt++;

  std::vector<std::string> names;
  std::istringstream nistr(line_in);

  std::string name;
  if (!(nistr >> name) && (name != "%time"))
  {
    std::ostringstream str;
    str << "Wrong header line in file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }

  while (nistr >> name)
  {
    names.push_back(name);
  }

  while (std::getline(fin, line_in))
  {
    std::istringstream istr(line_in);
    line_cnt++;

    double ctime;
    istr >> ctime;

    if (istr.fail())
    {
      std::ostringstream str;
      str << "Can't get time in file: " << fname << " line: " << line_cnt;
      SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
    }

    for (unsigned long i = 0; i < names.size(); i += 5)
    {
      double cval, rval, t2val, sinval, t3val;

      istr >> cval;
      istr >> rval;
      istr >> t2val;
      istr >> sinval;
      istr >> t3val;

      if (istr.fail())
      {
        std::ostringstream str;
        str << "Failed to read values from file: " << fname << " line: " << line_cnt;
        str << " for: " << names[i] << " " << names[i+1] << " "
            << names[i+2] << " " << names[i+3]<< " " << names[i+4];
        str << " at time: " << ctime;
        SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
      }

      int err = check_values(ctime, delays[i / 5], cval, rval, t2val, sinval, t3val);
      if (err != 0)
      {
        std::ostringstream str;
        str << "Wrong value in file: " << fname << " line: " << line_cnt;
        str << " for signal: " << names[i+err-1] << " at time " << ctime;
        str << " delay: " << delays[i / 5] << " d index: " << i / 5;
        SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
      }
    }
  }
  return false;
}

int sc_main(int argn, char* argc[])
{
  sc_core::sc_set_time_resolution(1.0, sc_core::SC_FS);

  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> s_out_const1, s_out_ramp1, s_out_t21, s_out_sin1, s_out_t31;
  sca_tdf::sca_signal<double> s_out_const_1, s_out_ramp_1, s_out_t2_1, s_out_sin_1, s_out_t3_1;
  sca_tdf::sca_signal<double> s_out_const_3, s_out_ramp_3, s_out_t2_3, s_out_sin_3, s_out_t3_3;
  sca_tdf::sca_signal<double> s_out_const_5, s_out_ramp_5, s_out_t2_5, s_out_sin_5, s_out_t3_5;

  sca_tdf::sca_signal<double> s_out_10;
  sca_tdf::sca_signal<double> s_out_11;
  sca_tdf::sca_signal<double> s_out_12;
  sca_tdf::sca_signal<double> s_out_13;
  sca_tdf::sca_signal<double> s_out_14;

  sca_tdf::sca_signal<double> s_out_20, s_out_40, s_out_60;
  sca_tdf::sca_signal<double> s_out_21, s_out_41, s_out_61;
  sca_tdf::sca_signal<double> s_out_22, s_out_42, s_out_62;
  sca_tdf::sca_signal<double> s_out_23, s_out_43, s_out_63;
  sca_tdf::sca_signal<double> s_out_24, s_out_44, s_out_64;

  sca_tdf::sca_signal<double> s_out_70, s_out_71, s_out_72, s_out_73, s_out_74;

  // instantiate and connect components

  ltf_nd_src* i_src = new ltf_nd_src("i_src");
  i_src->outp_const_1(s_out_const1);
  i_src->outp_ramp_1(s_out_ramp1);
  i_src->outp_t2_1(s_out_t21);
  i_src->outp_sin_1(s_out_sin1);
  i_src->outp_t3_1(s_out_t31);

  i_src->outp0(s_out_10);
  i_src->outp1(s_out_11);
  i_src->outp2(s_out_12);
  i_src->outp3(s_out_13);
  i_src->outp4(s_out_14);

  ltf_nd_test* i_nd_test = new ltf_nd_test("i_nd_test");
  i_nd_test->in_const_1(s_out_const1);
  i_nd_test->in_ramp_1(s_out_ramp1);
  i_nd_test->in_t2_1(s_out_t21);
  i_nd_test->in_sin_1(s_out_sin1);

  i_nd_test->in_10(s_out_10);
  i_nd_test->in_11(s_out_11);
  i_nd_test->in_12(s_out_12);
  i_nd_test->in_13(s_out_13);

  i_nd_test->out_const_1(s_out_const_1);
  i_nd_test->out_ramp_1(s_out_ramp_1);
  i_nd_test->out_t2_1(s_out_t2_1);
  i_nd_test->out_sin_1(s_out_sin_1);
  i_nd_test->out_t3_1(s_out_t3_1);

  i_nd_test->out_20(s_out_20);
  i_nd_test->out_21(s_out_21);
  i_nd_test->out_22(s_out_22);
  i_nd_test->out_23(s_out_23);
  i_nd_test->out_24(s_out_24);

  i_nd_test->out_const_3(s_out_const_3);
  i_nd_test->out_ramp_3(s_out_ramp_3);
  i_nd_test->out_t2_3(s_out_t2_3);
  i_nd_test->out_sin_3(s_out_sin_3);
  i_nd_test->out_t3_3(s_out_t3_3);

  i_nd_test->out_40(s_out_40);
  i_nd_test->out_41(s_out_41);
  i_nd_test->out_42(s_out_42);
  i_nd_test->out_43(s_out_43);
  i_nd_test->out_44(s_out_44);

  i_nd_test->out_const_5(s_out_const_5);
  i_nd_test->out_ramp_5(s_out_ramp_5);
  i_nd_test->out_t2_5(s_out_t2_5);
  i_nd_test->out_sin_5(s_out_sin_5);
  i_nd_test->out_t3_5(s_out_t3_5);

  i_nd_test->out_60(s_out_60);
  i_nd_test->out_61(s_out_61);
  i_nd_test->out_62(s_out_62);
  i_nd_test->out_63(s_out_63);
  i_nd_test->out_64(s_out_64);

  i_nd_test->out_70(s_out_70);
  i_nd_test->out_71(s_out_71);
  i_nd_test->out_72(s_out_72);
  i_nd_test->out_73(s_out_73);
  i_nd_test->out_74(s_out_74);

  /////// Tracing -> for each rate a different file

  sca_util::sca_trace_file* tf1 = sca_util::sca_create_tabular_trace_file("tdf_nd_1_1.dat");

  sca_util::sca_trace(tf1, s_out_const1, "s_out_const1");
  sca_util::sca_trace(tf1, s_out_ramp1, "s_out_ramp1");
  sca_util::sca_trace(tf1, s_out_t21, "s_out_t21");
  sca_util::sca_trace(tf1, s_out_sin1, "s_out_sin1");
  sca_util::sca_trace(tf1, s_out_t31, "s_out_t31");

  sca_util::sca_trace(tf1, s_out_10, "s_out_10");
  sca_util::sca_trace(tf1, s_out_11, "s_out_11");
  sca_util::sca_trace(tf1, s_out_12, "s_out_12");
  sca_util::sca_trace(tf1, s_out_13, "s_out_13");
  sca_util::sca_trace(tf1, s_out_14, "s_out_14");

  sca_util::sca_trace(tf1, s_out_const_1, "s_out_const_1");
  sca_util::sca_trace(tf1, s_out_ramp_1, "s_out_ramp_1");
  sca_util::sca_trace(tf1, s_out_t2_1, "s_out_t2_1");
  sca_util::sca_trace(tf1, s_out_sin_1, "s_out_sin_1");
  sca_util::sca_trace(tf1, s_out_t3_1, "s_out_t3_1");

  sca_util::sca_trace(tf1, s_out_const_3, "s_out_const_3");
  sca_util::sca_trace(tf1, s_out_ramp_3, "s_out_ramp_3");
  sca_util::sca_trace(tf1, s_out_t2_3, "s_out_t2_3");
  sca_util::sca_trace(tf1, s_out_sin_3, "s_out_sin_3");
  sca_util::sca_trace(tf1, s_out_t3_3, "s_out_t3_3");

  sca_util::sca_trace(tf1, s_out_const_5, "s_out_const_5");
  sca_util::sca_trace(tf1, s_out_ramp_5, "s_out_ramp_5");
  sca_util::sca_trace(tf1, s_out_t2_5, "s_out_t2_5");
  sca_util::sca_trace(tf1, s_out_sin_5, "s_out_sin_5");
  sca_util::sca_trace(tf1, s_out_t3_5, "s_out_t3_5");

  sca_util::sca_trace(tf1, s_out_20, "s_out_20");
  sca_util::sca_trace(tf1, s_out_21, "s_out_21");
  sca_util::sca_trace(tf1, s_out_22, "s_out_22");
  sca_util::sca_trace(tf1, s_out_23, "s_out_23");
  sca_util::sca_trace(tf1, s_out_24, "s_out_24");

  sca_util::sca_trace(tf1, s_out_40, "s_out_40");
  sca_util::sca_trace(tf1, s_out_41, "s_out_41");
  sca_util::sca_trace(tf1, s_out_42, "s_out_42");
  sca_util::sca_trace(tf1, s_out_43, "s_out_43");
  sca_util::sca_trace(tf1, s_out_44, "s_out_44");

  sca_util::sca_trace(tf1, s_out_60, "s_out_60");
  sca_util::sca_trace(tf1, s_out_61, "s_out_61");
  sca_util::sca_trace(tf1, s_out_62, "s_out_62");
  sca_util::sca_trace(tf1, s_out_63, "s_out_63");
  sca_util::sca_trace(tf1, s_out_64, "s_out_64");

  sca_util::sca_trace(tf1, s_out_70, "s_out_70");
  sca_util::sca_trace(tf1, s_out_71, "s_out_71");
  sca_util::sca_trace(tf1, s_out_72, "s_out_72");
  sca_util::sca_trace(tf1, s_out_73, "s_out_73");
  sca_util::sca_trace(tf1, s_out_74, "s_out_74");

  ////////////////////////////////////////////////

  sc_core::sc_start(); //start time domain simulation for 5ms

  sca_util::sca_close_tabular_trace_file(tf1);
  
  std::vector<double> delays;
  delays.resize(20, 3.3e-6);

  delays[0] = 0.0;
  delays[1] = 0.0;

  // delay is not shown, due we propagate the first value to zero -> thus at time
  // zero the constant is valid -> the ramp starts at zero (begin of integration with 1)
  check_file("tdf_nd_1_1.dat", delays);

  ///////////////////////////////////////////////////////////////////////////

  if (sc_core::sc_report_handler::get_count(sc_core::SC_ERROR) <= 0)
  {
    SC_REPORT_INFO("ND_CHECK", "Test was successful");
  }

  TEST_LABEL_END;

  delete i_src, i_nd_test;

  return 0;
}
