/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// test sca_ss for SystemC-AMS 2.0 dtdf -
// correct integration with variable timesteps, test for SC_ZERO_TIME
// with sca_tdf::sca_in in zero timestep iteration at the beginning

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE( my_ss )
{
  sca_tdf::sca_in<sca_util::sca_vector<double> > in;
  sca_tdf::sca_out<double> out;

  my_ss( sc_core::sc_module_name nm )
  : in("in"), out("out"), ss("ss"), activation(-1)
  {
  }

  void set_attributes()
  {
    set_timestep(1.0, sc_core::SC_SEC);
    does_attribute_changes();
  }

  void initialize()
  {
    A(0, 0) = 0.0;
    B(0, 0) = 1.0;  // ds=x
    C(0, 0) = 1.0;  // y=s  -> y=integr(x)
    D(0, 0) = 0.0;

    std::cout << "@" << get_time() << ": activation = " << activation
        << " initialize: SS tstep follows TDF timestep = " << get_timestep() << std::endl;
  }

  void processing()
  {
    sca_util::sca_vector<double> tmp = ss(A, B, C, D, in);
    std::cout << "@" << get_time() << ": activation = " << activation << "        mstep: " << get_timestep()  << "            input = " << in.read()(0) << "  output = " << tmp(0) << std::endl;
    out.write(tmp(0));
    activation++;
  }

  void change_attributes()
  {
    switch(activation)
    {
      case 0: // check iteration at start
        set_timestep(0.0, sc_core::SC_SEC);
        break;

      case 1:
        set_timestep(0.0, sc_core::SC_SEC);
        break;

      case 2:
        set_timestep(0.0, sc_core::SC_SEC);
        break;

      case 3:
        set_timestep(1.0, sc_core::SC_SEC);
        break;

      case 4:
        set_timestep(1.0, sc_core::SC_SEC);
        break;

      case 5:
        sc_core::sc_stop();
        break;

      default:  break;
    }
  }

 private:
  sca_tdf::sca_ss ss; // Laplace transfer function
  sca_util::sca_vector<double> state; // numerator and denominator coefficients
  sca_util::sca_matrix<double> A, B, C, D;
  int activation;
};

SCA_TDF_MODULE(my_step)
{
  sca_tdf::sca_out<sca_util::sca_vector<double> > out; // output port

  SCA_CTOR(my_step) : out("out")
  {}

  void set_attributes()
  {
    out.set_delay(0);
    accept_attribute_changes();
  }

  void initialize()
  {
    last_time = sca_core::sca_max_time();
    repeat_cnt = 0;
  }

  void processing()
  {
    if (last_time == get_time()) repeat_cnt++;
    else                         repeat_cnt = 0;
    last_time = get_time();

    sca_util::sca_vector<double> tmp(1);

    if (repeat_cnt == 1) tmp(0) = 2.0;
    else                 tmp(0) = 1.0;

    out.write(tmp);
  }

  void reinitialize()
  {
  }

  void change_attributes()
  {
  }

 private:
  sca_core::sca_time last_time;
  long repeat_cnt;

};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<sca_util::sca_vector<double> > sig1;
  sca_tdf::sca_signal<double> sig2;

  my_step step("step");
  my_ss  ss1("ss1");

  step.out(sig1);
  ss1.in(sig1);
  ss1.out(sig2);

  sc_core::sc_start();

  TEST_LABEL_END;

  return 0;
}
