/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// test ltf_nd for SystemC-AMS 2.0 dtdf -
// correct integration with variable timesteps, test for SC_ZERO_TIME
// with sca_tdf::sca_in in zero timestep iteration at the beginning

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE( my_ltf )
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  my_ltf( sc_core::sc_module_name nm )
  : in("in"), out("out"),
    activation(-1), gain(1.0)
  {}

  void set_attributes()
  {
    set_timestep(1.0, sc_core::SC_SEC);
    does_attribute_changes();
  }

  void initialize()
  {
    num(0) = 1.0;  // H(s) = 1/s
    den(0) = 0.0;
    den(1) = 1.0;

    std::cout << "@" << get_time() << ": activation = " << activation
              << " initialize: LTF tstep follows TDF timestep = " << get_timestep() << std::endl;
  }

  void processing()
  {
    double tmp = ltf_nd(num, den, in, gain);

    std::cout << "@" << get_time()
              << ": activation = " << activation
              << "        mstep: " << get_timestep()
              << "            input = " << in.read()
              << "  output = " << tmp
              << std::endl;

    out.write(tmp);
    activation++;
  }

  void change_attributes()
  {
    switch (activation)
    {
      case 0: // check iteration at start
        set_timestep(0.0, sc_core::SC_SEC);
        break;

      case 1:
        set_timestep(0.0, sc_core::SC_SEC);
        break;

      case 2:
        set_timestep(0.0, sc_core::SC_SEC);
        break;

      case 3:
        set_timestep(1.0, sc_core::SC_SEC);
        break;

      case 4:
        set_timestep(1.0, sc_core::SC_SEC);
        break;

      case 5:
        sc_core::sc_stop();
        break;

      default:  break;
    }
  }

 private:
  sca_tdf::sca_ltf_nd ltf_nd; // Laplace transfer function
  sca_util::sca_vector<double> num, den, state; // numerator and denominator coefficients
  double gain;
  int activation;
};

SCA_TDF_MODULE(my_step)
{
  sca_tdf::sca_out<double> out; // output port

  SCA_CTOR(my_step) : out("out")
  {}

  void set_attributes()
  {
    out.set_delay(0);
    accept_attribute_changes();
  }

  void initialize()
  {
    last_time = sca_core::sca_max_time();
    repeat_cnt = 0;
  }

  void processing()
  {
    if (last_time == get_time()) repeat_cnt++;
    else                         repeat_cnt = 0;
    last_time = get_time();

    if (repeat_cnt == 1) out.write(2.0);
    else                 out.write(1.0);
  }

  void reinitialize()
  {}

  void change_attributes()
  {}

  sca_core::sca_time last_time;
  long repeat_cnt;
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1;
  sca_tdf::sca_signal<double> sig2;

  my_step step("step");
  my_ltf  ltf1("ltf1");

  step.out(sig1);
  ltf1.in(sig1);
  ltf1.out(sig2);

  sc_core::sc_start();

  TEST_LABEL_END;

  return 0;
}
