/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// user's guide example my_dga showing use of converter ports

#include <systemc-ams>
#include "test_utilities.h"

test_util::sorted_stream sstr;

#define DISP(obj, val) \
  sstr.set_time(sc_core::sc_time_stamp(), 1); \
  sstr << obj.name() << ": " << val << " at " << sc_core::sc_time_stamp() << std::endl; \
  sstr.flush_stamp()

SC_MODULE( my_ctrl )
{
  sc_core::sc_out<bool> out;

  SC_CTOR( my_ctrl ) : out("out")
  { 
    SC_THREAD( ctrl_signal );
  }

  void ctrl_signal()
  {
    while (true)
    { 
      wait(2, sc_core::SC_MS);
      out.write(true); DISP(out, "true");
      wait(2, sc_core::SC_MS);
      out.write(false); DISP(out, "false");
    }
  }
};
    
SCA_TDF_MODULE( my_source )
{
  sca_tdf::sca_out<double> out;

  void processing()
  {
    if (val >= max) step = -1;
    if (val <= -max) step = 1;
  
    val = val + step;
    out.write(val);
    DISP(out, val);
  }

  SCA_CTOR( my_source ) : out("out"), val(2.0), step(1), max(3.0)
  {
  }

 private:
  int step;
  double val, max;
};


SCA_TDF_MODULE(my_dga)
{    
  sca_tdf::sca_in<double>   in; // input port
  sca_tdf::sca_out<double> out; // output port

  // control signal from the discrete-event domain
  sca_tdf::sca_de::sca_in<bool> high_gain_state; // input converter port

  SCA_CTOR(my_dga) : in("in"), out("out"), high_gain_state("high_gain_state"),
    high_gain(100.0), low_gain(1.0) {}

  void set_attributes()
  {
    set_timestep(1.0, sc_core::SC_MS);
  }

  void processing()
  {
    double gain = high_gain_state.read() ? high_gain : low_gain; 
    out.write( gain * in.read() );
  } 

 private:
  double high_gain, low_gain;      
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1, sig2;
  sc_core::sc_signal<bool> de_sig;
  
  my_source src("src");
    src.out(sig1);

  my_ctrl ctrl("ctrl");
    ctrl.out(de_sig);

  my_dga dga("dga");
    dga.in(sig1);
    dga.out(sig2);
    dga.high_gain_state(de_sig);
    
  sc_core::sc_start(12, sc_core::SC_MS);

  sstr.print();

  TEST_LABEL_END;

  return 0;
}
