/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// bask_mod now uses parameters

#include <systemc-ams>

#include "mixer.h"
#include "sin_src.h"

SC_MODULE(bask_mod)
{
  sca_tdf::sca_in<bool>    in;
  sca_tdf::sca_out<double> out;

  sin_src sine;
  mixer   mix;

  bask_mod( sc_core::sc_module_name nm, 
            double baseband_freq,
            double carrier_freq,
            double carrier_ampl = 1.0,
            unsigned long samples_per_period = 20 )
  : in("in"), out("out"), 
    sine("sine", 
       carrier_ampl, 
       carrier_freq, 
       sca_core::sca_time( (1.0 / (samples_per_period * carrier_freq) ), sc_core::SC_SEC) ),
    mix("mix", (int)ceil( static_cast<double>(samples_per_period) * carrier_freq / baseband_freq ) ),
    carrier("carrier")
  {
    using namespace sc_core; // essential for sc_assert to work, when using OSCI systemc-2.2.0

    // Plausibility checks
    sc_assert(carrier_freq > baseband_freq); // wouldn't make sense otherwise!
    sc_assert(samples_per_period > 2);       // Nyquist criterion satisfied?
    sc_assert(carrier_ampl > 0.0);           // Otherwise the output is 0 all the way!
  
    sine.out(carrier);
    mix.in_wav(carrier);
    mix.in_bin(in);
    mix.out(out);
  }

 private:
  sca_tdf::sca_signal<double> carrier;
};
