/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

#include "zdiode_dtdf.h"
#include "test_utilities.h"

SCA_TDF_MODULE(dtdf_rectifier_ctrl)
{
  sca_tdf::sca_in<double>  vin;
  sca_tdf::sca_out<double> rout;
  sca_tdf::sca_out<double> vth;

  void set_attributes()
  {
    rout.set_delay(1);
    vth.set_delay(1);

    vth_p = 0.7;  // threshold voltage
    vth_n = -53.0;
    ron = 1e-3;   // on resistance
    roff = 1e8;   // off resistance

    does_attribute_changes();
  }

  void initialize()
  {
    //iteration at time zero is possible
    rout.initialize(roff);
    vth.initialize(0.0);
    ron_state = true;

    recalculate = 0;
    activations = 0;
    iterations = 0;
    max_iterations = 0;
  }

  void change_attributes()
  {
    activations++;
    if (recalculate > 0)
    {
      // invalid / repeat last timestep
      request_next_activation(sc_core::SC_ZERO_TIME);

      if (recalculate > 10)
      {
        SC_REPORT_ERROR("dtdf_rectifier_ctrl", "Convergence failed. Simulation will terminate.");
      }

      iterations++;
      if (recalculate > max_iterations) max_iterations = iterations;
    }
  }

  void processing()
  {
    double vin_tmp = vin.read();

    // switch between states
    if (ron_state)
    {
      // hysteresis due curves do not fit exactly
      if ((vin_tmp > vth_p) || (vin_tmp < vth_n))
      {
        recalculate = 0;
      }
      else
      {
        recalculate++;
        ron_state = false;
      }
    }
    else
    {
      if ((vin_tmp>vth_p) || (vin_tmp<vth_n))
      {
        recalculate++;
        ron_state = true;
      }
      else recalculate = 0;
    }

    // set resistance and threshold in dependency of the state
    if (ron_state)
    {
      rout.write(ron);
      if (vin_tmp > vth_p) vth.write(vth_p);
      else                 vth.write(vth_n);
    }
    else
    {
      rout.write(roff);
      vth.write(0.0);
    }
  }

  void end_of_simulation()
  {
    std::ostringstream str;

    str << "Repeated timesteps: " << iterations << " from: " << activations;
    str << " maximum repeat of a timestep: " << max_iterations;
    SC_REPORT_INFO("zdiode", str.str().c_str());
  }

  SCA_CTOR(dtdf_rectifier_ctrl) {}

  double ron, roff, vth_p, vth_n;

private:
  bool ron_state;
  unsigned long recalculate, iterations, activations, max_iterations;
};

zdiode_dtdf::zdiode_dtdf(sc_core::sc_module_name, const params& _p)
{
  rctrl = new dtdf_rectifier_ctrl("rctrl");
  rctrl->vin(s_vin);
  rctrl->rout(s_rout);
  rctrl->vth(s_vth);
  rctrl->vth_p=_p.vth_p;
  rctrl->vth_n=_p.vth_n;
  rctrl->ron=_p.ron;
  rctrl->roff=_p.roff;

  vth = new sca_eln::sca_tdf_vsource("vth");
  vth->p(n1);
  vth->n(n);
  vth->inp(s_vth);

  rrect = new sca_eln::sca_tdf_r("rrect");
  rrect->p(p);
  rrect->n(n1);
  rrect->inp(s_rout);

  vrect = new sca_eln::sca_tdf_vsink("vrect");
  vrect->p(p);
  vrect->n(n);
  vrect->outp(s_vin);
}

zdiode_dtdf::~zdiode_dtdf()
{
  delete rctrl, vth, rrect, vrect;
}

// function for current tracing - current through rectifier
namespace sca_util
{
  inline void sca_trace(sca_util::sca_trace_file* tf, const zdiode_dtdf& rect, const std::string& nm)
  {
    sca_util::sca_trace(tf, *(rect.vth), nm);
  }
}

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_eln::sca_node net1;
  sca_eln::sca_node net2;
  sca_eln::sca_node_ref agnd;

  zdiode_dtdf z1("z1"); // use build-in params
    z1.p(net1);
    z1.n(agnd);

  sca_eln::sca_r r1("r1", 1e3); // 1kOhm
    r1.n(net1);
    r1.p(net2);

  sca_eln::sca_vsource v1("v1", 0, 0, 1, 50.0); // 1V, 50Hz
    v1.p(net2);
    v1.n(agnd);
    v1.set_timestep(1.0, sc_core::SC_MS);


  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("zdiode_dtdf");
    sca_util::sca_trace(tf, net1, "net1");
    sca_util::sca_trace(tf, net2, "net2");

  sc_core::sc_start(120, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  test_util::check_results("zdiode_dtdf", 3);

  TEST_LABEL_END;

  return 0;
}
