/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/
#include <systemc-ams>
#include "test_utilities.h"

using namespace std;

///// primitive modules /////////

SCA_TDF_MODULE(adder)
{
  sca_tdf::sca_in<double>  in1;
  sca_tdf::sca_in<double>  in2;
  sca_tdf::sca_out<double> out;

  void processing()
  {
    long expected_value = cnt-2;
    if (cnt < 2)
    {
      expected_value = -1;
    }

    if (int(in1.read()) != expected_value)
    {
      std::ostringstream str;
      str << "Unexpected value at in1: " << in1.read() << "  " << expected_value << std::endl;
      SC_REPORT_ERROR("Check Error", str.str().c_str());
    }

    double expected_dvalue = 2 * (cnt2 - 1) + 0.5;
    if (cnt < 1)
    {
      expected_dvalue = -3;
    }

    if (in2.read() != expected_dvalue)
    {
      std::ostringstream str;
      str << "Unexpected value at in2: " << in2.read() << "  " << expected_dvalue << std::endl;
      SC_REPORT_ERROR("Check Error", str.str().c_str());
    }
    out.write(double(cnt2));
    cnt++;
    cnt2++;
  }

  SCA_CTOR(adder) 
  : in1("in1"), in2("in2"), out("out"), cnt(0), cnt2(1000)
  {}

 private:
  long cnt;
  long cnt2;
};

////////////////////////////////////////////////

SCA_TDF_MODULE(source)
{
  sca_tdf::sc_in<int>  flag_in;
  sca_tdf::sc_out<int> flag_out;
  sca_tdf::sca_out<double> out;

  void set_attributes()
  {
    out.set_timestep(sc_core::sc_time(1.0, sc_core::SC_SEC));
  }

  void processing()
  {
    out.write(state);

    if (get_timestep() != sc_core::sc_time(1.0, sc_core::SC_SEC))
    {
      SC_REPORT_ERROR("Check Error", "Unexpected timestep returned");
    }

    if (get_time() != state * get_timestep())
    {
      SC_REPORT_ERROR("Check Error", "Unexpected execution time");
    }

    state++;
    int tmp = flag_in.read();

    flag_out.write(cnt3);
    cnt3++;

    int expected_value = fcnt;
    if (fcnt%5) expected_value--;
    if (flag_in.read() != expected_value)
    {
      SC_REPORT_ERROR("Check Error", "Unexpected value @ inf");
    }
    fcnt++;
  }

  SCA_CTOR(source) 
  : flag_in("flag_in"), flag_out("flag_out"), out("out"), 
    cnt(0), cnt3(3000), state(0), fcnt(5000)
  {}

 private:
  long cnt;
  long cnt3;
  long state;
  long fcnt;
};

////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(sink)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sc_out<double> deout;

  void set_attributes()
  {
    deout.set_delay(1);
  }

  void initialize()
  {
    deout.initialize(-1.0);
  }

  void processing()
  {
    if (in.read() != (2 * cnt + 1))
    {
      std::cout << in.read() << std::endl;
      SC_REPORT_ERROR("Check Error", "Unexpected value");
    }

    if (get_timestep() != sca_core::sca_time(2.0, sc_core::SC_SEC))
    {
      SC_REPORT_ERROR("Check Error", "Unexpected timestep");
    }

    if (get_time() != cnt2 * get_timestep())
    {
      SC_REPORT_ERROR("Check Error", "Unexpected time");
    }
    cnt2++;
    deout.write(in.read());
    cnt += 2;
  }

  SCA_CTOR(sink) : in("in"), deout("deout"), cnt(1000), cnt2(0)
  {}

 private:
  long cnt;
  long cnt2;
};

////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(decint)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  void set_attributes()
  {
    in.set_rate(in_rate);
    out.set_rate(out_rate);
  }

  void processing()
  {
    if (in_rate != in.get_rate())
    {
      SC_REPORT_ERROR("Check Error", "Unexpected input rate");
    }

    if (out_rate != out.get_rate())
    {
      std::cout << "out_rate: " << out_rate << "  get: " << out.get_rate() << std::endl;

      SC_REPORT_ERROR("Check Error", "Unexpected output rate");
    }

    if (get_timestep() != in.get_timestep() * in_rate)
    {
      in.get_timestep();

      std::ostringstream str;
      str << name() << " unexpected input timestep - port: " << in.get_timestep()
          << " module: " << get_timestep() << " in_rate: " << in_rate;
      SC_REPORT_ERROR("Check Error", str.str().c_str());
    }

    if (in.get_time() != in.get_timestep() * cnt * in_rate)
    {
      SC_REPORT_ERROR("Check Error", "Unexpected input time");
    }

    if (in.get_time(in_rate - 1) != cnt * in.get_timestep() * in_rate + (in_rate-1) * in.get_timestep())
    {
      SC_REPORT_ERROR("Check Error", "Unexpected input time");
    }

    if (get_timestep() != out.get_timestep() * out_rate)
    {
      SC_REPORT_ERROR("Check Error", "Unexpected output timestep");
    }

    if (out.get_time() != out.get_timestep() * cnt * out_rate)
    {
      SC_REPORT_ERROR("Check Error", "Unexpected output time");
    }

    if (out.get_time(out_rate-1) != out.get_timestep() * cnt * out_rate + out.get_timestep() * (out_rate - 1))
    {
      SC_REPORT_ERROR("Check Error", "Unexpected output time");
    }

    if (get_time() != cnt * get_timestep())
    {
      SC_REPORT_ERROR("Check Error", "Unexpected execution time");
    }

    double sum = 0;
    for (unsigned long i = 0; i < in_rate; i++)
    {
      sum += in.read(i);
    }

    for (unsigned long i = 0; i <out_rate; i++)
    {
      out.write(sum + double(i) / out_rate, i);
    }

    cnt++;
  }

  SCA_CTOR(decint) 
  : in("in"), out("out"), in_rate(1), out_rate(1), cnt(0)
  {}

 public:
  unsigned long in_rate;
  unsigned long out_rate;

 private:
  long cnt;
};

////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(delay)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  void set_attributes()
  {
    out.set_delay(delay_val);
  }

  void initialize()
  {
    for (unsigned long i = 0; i < delay_val; i++) 
      out.initialize(double(signed(-i) - 1), i);
  }

  void processing()
  {
    if (out.get_delay() != delay_val)
    {
      SC_REPORT_ERROR("Check Error", "Unexpected delay value");
    }

    if (get_time() != cnt * get_timestep())
    {
      SC_REPORT_ERROR("Check Error", "Unexpected execution time");
    }

    if (in.get_time() != cnt * in.get_timestep())
    {
      SC_REPORT_ERROR("Check Error", "Unexpected inport time");
    }

    if (out.get_time() != cnt * out.get_timestep())
    {
      SC_REPORT_ERROR("Check Error", "Unexpected inport time");
    }

    out.write(in.read());
    out = in;  // the same but not recommended
    cnt++;
  }

  SCA_CTOR(delay) : in("in"), out("out"), delay_val(0), cnt(0)
  {}

 public:
   unsigned long delay_val;

 private:
  long cnt;
};

///////////////////////////////////////

SC_MODULE(sc_test)
{
  sc_core::sc_in<double> in1;
  sc_core::sc_in<int>    flag_out_in;
  sc_core::sc_out<int>   outf;

  void fout_proc()
  {
    if (flag_out_in.read() != -200.0)
    {
      std::cout << flag_out_in.read() << std::endl;
      SC_REPORT_ERROR("Check Error", "Unexpected value");
    }

    while (true)
    {
      wait(sc_core::SC_ZERO_TIME);
      // now in delta cycle 1 the new value is available

      if (flag_out_in.read() != cnt3)
      {
        std::cout << flag_out_in.read() << std::endl;
        SC_REPORT_ERROR("Check Error", "Unexpected value");
      }

      wait(1.0, sc_core::SC_SEC);

      // in delta 0 yet the old value
      if (flag_out_in.read() != cnt3)
      {
        std::cout << flag_out_in.read() << std::endl;
        SC_REPORT_ERROR("Check Error", "Unexpected value");
      }
      cnt3++;
    }
  }

  void inf_proc()
  {
    if (in1.read() != -100.0)
    {
      std::cout << in1.read() << std::endl;
      SC_REPORT_ERROR("Check Error", "Unexpected value");
    }

    wait(sc_core::SC_ZERO_TIME);  // delta=1

    if (in1.read() != -1.0)
    {
      std::cout << in1.read() << std::endl;
      SC_REPORT_ERROR("Check Error", "Unexpected value");
    }

    wait(2.0, sc_core::SC_SEC);

    if (in1.read() != -1.0)
    {
      SC_REPORT_ERROR("Check Error", "Unexpected value");
    }

    while (true)
    {
      wait(sc_core::SC_ZERO_TIME);

      if (in1.read() != 2 * cnt2 + 1)
      {
        std::cout << in1.read() << std::endl;
        SC_REPORT_ERROR("Check Error", "Unexpected value");
      }

      wait(2.0, sc_core::SC_SEC);

      if (in1.read() != 2 * cnt2 + 1)
      {
        std::cout << in1.read() << std::endl;
        SC_REPORT_ERROR("Check Error", "Unexpected value");
      }

      cnt2 += 2;
    }
  }

  void outf_proc()
  {
    sc_core::sc_time period(1.0, sc_core::SC_SEC);

    outf = cnt;

    while (true)
    {
      cnt++;

      if (cnt%5)
      {
        wait(period);
        outf = cnt;
      }
      else
      {
        wait(period - sc_core::sc_get_time_resolution());
        outf = cnt;
        wait(sc_core::sc_get_time_resolution());
      }
    }
  }

  SC_CTOR(sc_test) 
  : in1("in1"), flag_out_in("flag_out_in"), outf("outf"), 
    cnt(5000), cnt2(1000), cnt3(3000)
  {
    SC_THREAD(outf_proc);
    SC_THREAD(inf_proc);
    SC_THREAD(fout_proc);
    outf.initialize(cnt);
  }

 private:
  int cnt;
  int cnt2;
  int cnt3;
};

SC_MODULE(delay_chain)
{
  sca_tdf::sca_in<double>  in;
  sca_tdf::sca_out<double> out;

  delay *del1, *del2;

  sca_tdf::sca_signal<double> internal_signal;

  SC_CTOR(delay_chain) : in("in"), out("out")
  {
    del1 = new delay("del1");
    del1->in(in);
    del1->out(internal_signal);
    del1->delay_val = 1;

    del2 = new delay("del2");
    del2->in(internal_signal);
    del2->out(out);
    del2->delay_val = 1;
  }

  ~delay_chain()
  {
    delete del1, del2;
  }
};

////////////////////////////////////////////
// TB and simulation control
////////////////////////////////////////////

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig0, sig1, sig2, sig3, sig0_delayed;
  sca_tdf::sca_signal<double> sig4, sig5, sig6, sig7;

  sc_core::sc_signal<int> fin, fout;
  sc_core::sc_signal<double> s_deout;

  source s1("s1");
  s1.out(sig0);
  s1.flag_in(fin);
  s1.flag_out(fout);

  delay_chain dc1("dc1");
  dc1.in(sig0);
  dc1.out(sig0_delayed);

  adder add1("add1");
  add1.in1(sig0_delayed);
  add1.in2(sig7);
  add1.out(sig1);
  add1.set_timestep(sc_core::sc_time(1.0, sc_core::SC_SEC));

  decint dum1("dum1");
  dum1.in(sig1);
  dum1.out(sig2);

  decint dez1("dez1");
  dez1.in(sig2);
  dez1.out(sig3);
  dez1.in_rate = 2;

  sink si1("si1");
  si1.in(sig3);
  si1.deout(s_deout);
  si1.set_timestep(sc_core::sc_time(2.0, sc_core::SC_SEC));

  decint int1("int1");
  int1.in(sig1);
  int1.out(sig5);
  int1.out_rate = 2;

  delay de1("de1");
  de1.delay_val = 2;
  de1.in(sig5);
  de1.out(sig6);
  de1.set_timestep(sc_core::sc_time(0.5, sc_core::SC_SEC));

  decint dez2("dez2");
  dez2.in(sig6);
  dez2.out(sig7);
  dez2.in_rate = 2;

  sc_core::sc_signal<double> scsig;

  sc_test scm("scm");
  scm.in1(s_deout);
  scm.outf(fin);
  scm.flag_out_in(fout);

  // signal tracing

  // analog signals for matlab, octave, gwave, ...
  sca_util::sca_trace_file* atf = sca_util::sca_create_tabular_trace_file("sdf_basic_trace.dat");
  sca_util::sca_trace(atf, sig5, "sig5");
  sca_util::sca_trace(atf, sig6, "sig6");
  sca_util::sca_trace(atf, sig7, "sig7");
  sca_util::sca_trace(atf, sig0, "sig0");
  sca_util::sca_trace(atf, dc1.in, "dc1.in");
  sca_util::sca_trace(atf, dc1.del1->in, "dc1.del1.in");

  // digital signals as vcd e.g. for gtkwave
  sc_core::sc_trace_file* tf = sc_core::sc_create_vcd_trace_file("dig_waves");
  sc_core::sc_trace(tf, fin, "fin");
  sc_core::sc_trace(tf, fout, "fout");
  sc_core::sc_trace(tf, s_deout, "s_deout");

  // simulation control

  s_deout.write(-100.0);  // initial value @ delta 0
  fout.write(-200);

  sc_core::sc_start(100.0, sc_core::SC_SEC);

  sc_core::sc_close_vcd_trace_file(tf);
  sca_util::sca_close_tabular_trace_file(atf);

  std::cout << "Test successfully finished" << std::endl;

  TEST_LABEL_END;

  return 0;
}
