/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

#include <systemc-ams>
#include "test_utilities.h"

//////////////////////////////////////////////////////////////////////////////

SC_MODULE(phone)
{
  sca_eln::sca_terminal tip;
  sca_eln::sca_terminal ring;
  sca_tdf::sca_in<double> v_voice_in;
  sc_core::sc_in<bool> hook;

  struct params
  {
    double rs;
    double rp;
    double cp;
    double rring;
    double cring;

    params()
      {
        rs = 220.0;
        rp = 820.0;
        cp = 115.0e-9;
        rring = 1.0e3;
        cring = 1.0e-6;
      }
  };

  // Instance
  sca_eln::sca_de_rswitch *hook_sw1;
  sca_eln::sca_de_rswitch *hook_sw2;

  sca_eln::sca_c *cring;
  sca_eln::sca_r *rring;
  sca_eln::sca_r *rs;
  sca_eln::sca_c *cp;
  sca_eln::sca_r *rp;
  sca_eln::sca_tdf_vsource *vin;

  // Nodes
  sca_eln::sca_node w_offhook;
  sca_eln::sca_node w_onhook;
  sca_eln::sca_node w1;
  sca_eln::sca_node w2;
  sca_eln::sca_node wring;

   phone(sc_core::sc_module_name nm, params pa = params()) 
   : tip("tip"), ring("ring"), v_voice_in("v_voice_in"),
     hook("hook"), p(pa)
   {
     architecture();
   }

   ~phone();

 private:
  params p;
  void architecture();
};

//////////////////////////////////////////////////////////////////////////////

void phone::architecture()
{
  // toggle switch for hook
  hook_sw1 = new sca_eln::sca_de_rswitch("hook_sw1");
     hook_sw1->p(tip);
     hook_sw1->n(w_onhook);
     hook_sw1->ctrl(hook);
     hook_sw1->off_state = true;

  hook_sw2 = new sca_eln::sca_de_rswitch("hook_sw2");
     hook_sw2->p(tip);
     hook_sw2->n(w_offhook);
     hook_sw2->ctrl(hook);

  // ringing load (onhook)
  cring = new sca_eln::sca_c("cring");
    cring -> p(wring);
    cring -> n(w_onhook);
    cring -> value = p.cring;

  rring = new sca_eln::sca_r("rring");
    rring -> p(wring);
    rring -> n(ring);
    rring -> value = p.rring;

  // active (subscriber) load (offhook)

  rs = new sca_eln::sca_r("rs");
    rs -> p(w1);
    rs -> n(w2);
    rs -> value = p.rs;

  cp = new sca_eln::sca_c("cp");
    cp -> p(w1);
    cp -> n(w_offhook);
    cp -> value = p.cp;

  rp = new sca_eln::sca_r("rp");
    rp -> p(w_offhook);
    rp -> n(w1);
    rp -> value = p.rp;

  // microphone
  vin = new sca_eln::sca_tdf_vsource("vin");
    vin -> p(w2);
    vin -> n(ring);
    vin -> inp(v_voice_in);
}

phone::~phone()
{
  delete hook_sw1, hook_sw2, cring, rring;
  delete rs, cp, rp, vin;
}

///////////////////////////////////////////////////////////////////////////////

SC_MODULE(slic)
{
  sca_eln::sca_terminal tip;
  sca_eln::sca_terminal ring;
  sca_tdf::sca_in<double> v_tip_ring;
  sca_tdf::sca_out<double> i_transversal;

  struct params
  {
    double scale_v_tr;
    double scale_i_tr;

    params()
    {
      scale_v_tr = 1.0;
      scale_i_tr = 1.0;
    }
  };

  sca_eln::sca_tdf_vsource* tip_driver;
  sca_eln::sca_tdf_vsource* ring_driver;

  sca_eln::sca_cccs*  i_tip_mirror;
  sca_eln::sca_cccs*  i_ring_mirror;

  sca_eln::sca_r*         r_tr_meas;
  sca_eln::sca_tdf_vsink* i_tr_measure;

  sca_eln::sca_node n_tr_i;
  sca_eln::sca_node n_tip_gnd, n_ring_gnd;
  sca_eln::sca_node_ref gnd;

  slic(sc_core::sc_module_name, params pa = params()) 
  : tip("tip"), ring("ring"), v_tip_ring("v_tip_ring"),
    i_transversal("i_transversal"), p(pa)
  {
    architecture();
  }

  ~slic();

private:
  params p;
  void architecture();
};

//////////////////////////////////////////////////////////////////////////////

void slic::architecture()
{
  tip_driver = new sca_eln::sca_tdf_vsource("tip_driver");
    tip_driver->inp(v_tip_ring);
    tip_driver->p(tip);
    tip_driver->n(n_tip_gnd);
    tip_driver->scale = p.scale_v_tr / 2.0;

  i_tip_mirror = new sca_eln::sca_cccs("i_tip_mirror");
    i_tip_mirror->ncp(n_tip_gnd);
    i_tip_mirror->ncn(gnd);
    i_tip_mirror->np(n_tr_i);
    i_tip_mirror->nn(gnd);
    i_tip_mirror->value = 0.5;

   ring_driver = new sca_eln::sca_tdf_vsource("ring_driver");
     ring_driver->inp(v_tip_ring);
     ring_driver->p(ring);
     ring_driver->n(n_ring_gnd);
     ring_driver->scale = -p.scale_v_tr / 2.0;

   i_ring_mirror = new sca_eln::sca_cccs("i_ring_mirror");
     i_ring_mirror->ncp(n_ring_gnd);
     i_ring_mirror->ncn(gnd);
     i_ring_mirror->np(n_tr_i);
     i_ring_mirror->nn(gnd);
     i_ring_mirror->value = -0.5;

   r_tr_meas = new sca_eln::sca_r("r_tr_meas");
     r_tr_meas->p(n_tr_i);
     r_tr_meas->n(gnd);
     r_tr_meas->value = 1.0;

   i_tr_measure = new sca_eln::sca_tdf_vsink("i_tr_measure");
     i_tr_measure->p(n_tr_i);
     i_tr_measure->n(gnd);
     i_tr_measure->outp(i_transversal);
     i_tr_measure->scale = p.scale_i_tr;
}

slic::~slic()
{
  delete tip_driver, i_tip_mirror, ring_driver;
  delete i_ring_mirror, r_tr_meas, i_tr_measure;
}

///////////////////////////////////////////////////////////////////////////////

SC_MODULE(protection_circuit)
{
  sca_eln::sca_terminal  tip_slic;
  sca_eln::sca_terminal  ring_slic;
  sca_eln::sca_terminal  tip;
  sca_eln::sca_terminal  ring;

  struct params
  {
    double rprot_tip_line;
    double rprot_ring_line;
    double rprot_tip_slic;
    double rprot_ring_slic;
    double cprot_tip;
    double cprot_ring;

    params()
    {
      rprot_tip_line  = 20.0;
      rprot_ring_line = 20.0;
      rprot_tip_slic  = 20.0;
      rprot_ring_slic = 20.0;
      cprot_tip  = 18.0e-9;
      cprot_ring = 18.0e-9;
    }
  };

  sca_eln::sca_c *cprot_tip;
  sca_eln::sca_c *cprot_ring;
  sca_eln::sca_r *rprot_tip_line;
  sca_eln::sca_r *rprot_ring_line;
  sca_eln::sca_r *rprot_tip_slic;
  sca_eln::sca_r *rprot_ring_slic;

  // Nodes
  sca_eln::sca_node     n_tip;
  sca_eln::sca_node     n_ring;
  sca_eln::sca_node_ref gnd;

  protection_circuit(sc_core::sc_module_name, params pa = params()) 
  : tip_slic("tip_slic"), ring_slic("ring_slic"), tip("tip"),
    ring("ring"), p(pa)
  {
     architecture();
  }

  ~protection_circuit();

private:
  params p;
  void architecture();
};

///////////////////////////////////////////////////////////////////////////////

void protection_circuit::architecture()
{
  rprot_tip_slic = new sca_eln::sca_r("rprot_tip_slic");
    rprot_tip_slic->p(tip_slic);
    rprot_tip_slic->n(n_tip);
    rprot_tip_slic->value = p.rprot_tip_slic;

  cprot_tip = new sca_eln::sca_c("cprot_tip");
    cprot_tip->p(n_tip);
    cprot_tip->n(gnd);
    cprot_tip->value = p.cprot_tip;

  rprot_tip_line = new sca_eln::sca_r("rprot_tip_line");
    rprot_tip_line->p(tip);
    rprot_tip_line->n(n_tip);
    rprot_tip_line->value = p.rprot_tip_line;

  rprot_ring_slic = new sca_eln::sca_r("rprot_ring_slic");
    rprot_ring_slic->p(ring_slic);
    rprot_ring_slic->n(n_ring);
    rprot_ring_slic->value = p.rprot_ring_slic;

  cprot_ring = new sca_eln::sca_c("cprot_ring");
    cprot_ring->p(n_ring);
    cprot_ring->n(gnd);
    cprot_ring->value = p.cprot_ring;

  rprot_ring_line = new sca_eln::sca_r("rprot_ring_line");
    rprot_ring_line->p(ring);
    rprot_ring_line->n(n_ring);
    rprot_ring_line->value = p.rprot_ring_line;
}

protection_circuit::~protection_circuit()
{
  delete rprot_tip_slic, cprot_tip, rprot_tip_line;
  delete rprot_ring_slic, cprot_ring, rprot_ring_line;
}

///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(sin_src)
{
  sca_tdf::sca_out<double> outp;

  struct params
  {
    double freq;
    double ampl;
    double offset;
    sca_core::sca_time sample_time;

    params()
    {
      ampl   = 1.0;
      offset = 0.0;
      freq   = 1e3;
      sample_time = sca_core::sca_time(10.0, sc_core::SC_US);
    }
  };

  void set_attributes()
  {
    outp.set_timestep(p.sample_time);
  }

  void ac_processing();
  void processing();

  sin_src(sc_core::sc_module_name nm, params pa = params()) 
  : outp("outp"), p(pa) {}

  params p;
};

// frequency domain implementation
void sin_src::ac_processing()
{
  // we use for ac-domain for all frequencies the same amplitude and a phase=0
  sca_ac_analysis::sca_ac(outp) = p.ampl;
}

// time domain implementation
void sin_src::processing()
{
  outp = p.ampl * std::sin(2.0 * M_PI * p.freq * get_time().to_seconds()) + p.offset;
}

////////////////////////////////////////////////////////

int sc_main(int argc, char* argv[])
{
  sca_eln::sca_node n_slic_tip, n_slic_ring;
  sca_eln::sca_node n_tip, n_ring, n_tip_ph, n_ring_ph;
  sca_tdf::sca_signal<double> s_v_tip_ring;
  sca_tdf::sca_signal<double> s_i_transversal;
  sca_tdf::sca_signal<double> s_voice;
  sc_core::sc_signal<bool>    s_hook;

  sin_src::params p_src_vtr;
    p_src_vtr.ampl = 0.0;
    p_src_vtr.offset = 10.0;
    p_src_vtr.freq = 1e3;

  sin_src* src_vtr = new sin_src("src_vtr", p_src_vtr);
    src_vtr->outp(s_v_tip_ring);

  slic* i_slic = new slic("i_slic");
    i_slic->tip(n_slic_tip);
    i_slic->ring(n_slic_ring);
    i_slic->v_tip_ring(s_v_tip_ring);
    i_slic->i_transversal(s_i_transversal);

  protection_circuit* i_protection_circuit = new protection_circuit("i_protection_circuit");
    i_protection_circuit->tip_slic(n_slic_tip);
    i_protection_circuit->ring_slic(n_slic_ring);
    i_protection_circuit->tip(n_tip);
    i_protection_circuit->ring(n_ring);

  sca_eln::sca_transmission_line* transmission_line = new sca_eln::sca_transmission_line("transmission_line");
    transmission_line->a1(n_tip);
    transmission_line->a2(n_ring);
    transmission_line->b1(n_tip_ph);
    transmission_line->b2(n_ring_ph);
    transmission_line->delay = sca_core::sca_time(1.0, sc_core::SC_US);

  phone* i_phone = new phone("i_phone");
    i_phone->tip(n_tip_ph);
    i_phone->ring(n_ring_ph);
    i_phone->v_voice_in(s_voice);
    i_phone->hook(s_hook);

  sin_src::params p_src_voice;
    p_src_voice.ampl = 1.0;
    p_src_voice.offset = 0.0;
    p_src_voice.freq = 1e3;

  sin_src* src_voice = new sin_src("src_voice", p_src_voice);
    src_voice->outp(s_voice);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("pots_fe.dat");
  sca_util::sca_trace(tf, n_tip, "n_tip");
  sca_util::sca_trace(tf, n_ring, "n_ring");
  sca_util::sca_trace(tf, n_slic_tip, "n_slic_tip");
  sca_util::sca_trace(tf, n_slic_ring, "n_slic_ring");
  sca_util::sca_trace(tf, s_i_transversal, "s_i_transversal");
  sca_util::sca_trace(tf, s_voice, "s_voice");
  sca_util::sca_trace(tf, s_v_tip_ring, "s_v_tip_ring");
  sca_util::sca_trace(tf, i_phone->rring, "i_phone.i_rring");
  sca_util::sca_trace(tf, i_phone->rs, "i_phone.i_rs");

  s_hook.write(false);  // subscriber onhook

  sc_core::sc_start(5.0, sc_core::SC_MS);

  s_hook.write(true);  // subscriber offhook

  sc_core::sc_start(5.0, sc_core::SC_MS);

  src_voice->p.ampl = 0.0;
  src_vtr->p.ampl = 1.0;

  sc_core::sc_start(5.0, sc_core::SC_MS);

  s_hook.write(false);  // subscriber onhook

  sc_core::sc_start(5.0, sc_core::SC_MS);

  src_voice->p.ampl = 1.0;

  sc_core::sc_start(5.0, sc_core::SC_MS);

  s_hook.write(true);  // subscriber offhook

  src_voice->p.ampl = 0.0;
  src_vtr->p.ampl = 10.0;
  src_vtr->p.freq = 50.0;

  sc_core::sc_start(100.0, sc_core::SC_MS);

  s_hook.write(false);  // subscriber onhook

  sc_core::sc_start(100.0, sc_core::SC_MS);

  tf->reopen("pots_fe_ac_on_vtr.dat");
  tf->set_mode(sca_util::sca_ac_format(sca_util::SCA_AC_DB_DEG));

  src_voice->p.ampl = 0.0;
  src_vtr->p.ampl = 1.0;

  sca_ac_analysis::sca_ac_start(10.0, 64e3, 1000, sca_ac_analysis::SCA_LOG);

  tf->reopen("pots_fe_ac_on_voice.dat");
  src_voice->p.ampl = 1.0;
  src_vtr->p.ampl = 0.0;

  sca_ac_analysis::sca_ac_start(10.0, 64e3, 1000, sca_ac_analysis::SCA_LOG);

  tf->disable();

  s_hook.write(true);  // subscriber offhook
  sc_core::sc_start(1.0, sc_core::SC_MS);  // signal must become valid

  src_voice->p.ampl = 0.0;
  src_vtr->p.ampl = 1.0;

  tf->reopen("pots_fe_ac_off_vtr.dat");
  tf->enable();

  sca_ac_analysis::sca_ac_start(10.0, 64e3, 1000, sca_ac_analysis::SCA_LOG);

  tf->reopen("pots_fe_ac_off_voice.dat");
  src_voice->p.ampl = 1.0;
  src_vtr->p.ampl = 0.0;

  sca_ac_analysis::sca_ac_start(10.0, 64e3, 1000, sca_ac_analysis::SCA_LOG);

  sca_util::sca_close_tabular_trace_file(tf);

  std::cout << sc_core::sc_time_stamp() << " simulation finished" << std::endl;

  TEST_LABEL_START;

  test_util::check_results("pots_fe", 9, "tdf_sc_ac_err_msg", 1e-12, 1e-8);

  test_util::check_results("pots_fe_ac_on_vtr", 9, "tdf_sc_ac_err_msg", 1e-12);

  test_util::check_results("pots_fe_ac_on_voice", 9, "tdf_sc_ac_err_msg", 1e-5, 1e-3);

  test_util::check_results("pots_fe_ac_off_vtr", 9, "tdf_sc_ac_err_msg", 1e-5, 1e-4);

  test_util::check_results("pots_fe_ac_off_voice", 9, "tdf_sc_ac_err_msg", 1e-5, 1e-4);

  TEST_LABEL_END;

  delete src_vtr, i_slic, i_protection_circuit, transmission_line, i_phone, src_voice;

  return 0;
}
