/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  disable_segfault.cpp - checks whether disable after trace creation
//                         causes a Segmentation fault if a sc_signal was added

#include<systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(delay)
{
  sca_tdf::sca_in<double> inp;
  sca_tdf::sca_out<double> outp;

  void set_attributes();
  void initialize();
  void processing();

  SCA_CTOR(delay)
  : inp("inp"),
    outp("outp") 
  {}

};

void delay::set_attributes()
{
  inp.set_delay(1);
}

void delay::initialize()
{
  inp.initialize(0.0);
}

void delay::processing()
{
  outp = inp;
}

/////////////////////////////

SCA_TDF_MODULE(adder)
{
  sca_tdf::sca_in<double> inp1;
  sca_tdf::sca_in<double> inp2;
  sca_tdf::sca_out<double> outp;

  void processing();

  SCA_CTOR(adder)
  : inp1("inp1"),
    inp2("inp2"),
    outp("outp")
  {}
};

void adder::processing()
{
  outp.write(inp1.read() + inp2.read());
}

///////////////////////////////////

SCA_TDF_MODULE(const_src)
{
  sca_tdf::sca_out<double> outp;
  sca_tdf::sca_de::sca_in<double> val_in;

  void set_attributes();
  void processing();

  SCA_CTOR(const_src)
  : outp("outp"),
    val_in("val_in")
  {}
};

void const_src::set_attributes()
{
  outp.set_timestep(1.0, sc_core::SC_MS);
}

void const_src::processing()
{
  outp = val_in.read();

  std::ostringstream str;
  str << get_time() << "  " << name()
      << " read from converter port: " << val_in.read();

  SC_REPORT_INFO("Simple_TDF", str.str().c_str());
}


SCA_TDF_MODULE(sink)
{
  sca_tdf::sca_in<double> inp;
  sca_tdf::sca_de::sca_out<double> out_sc;

  void processing();

  SCA_CTOR(sink)
  : inp("inp"),
  out_sc("out_sc")
  {}
};

void sink::processing()
{
  std::ostringstream str;
  str << get_time() << "  " << name()
      << " current value: " << inp.read();

  out_sc[0] = inp.read();

  SC_REPORT_INFO("Simple_TDF", str.str().c_str());
}

SC_MODULE(sc_monitor)
{
  sc_core::sc_in<double> inp;

  void print();

  SC_CTOR(sc_monitor) : inp("inp")
  {
    SC_METHOD(print);
    sensitive << inp;
  }
};

void sc_monitor::print()
{
  std::ostringstream str;
  str << sc_core::sc_time_stamp() << "  " << name()
      << " current value: " << inp.read();

  SC_REPORT_INFO("Simple_TDF", str.str().c_str());
}

////////////////////////////////

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1("sig1");
  sca_tdf::sca_signal<double> sig2;
  sca_tdf::sca_signal<double> sig3("sig3");
  sc_core::sc_signal<double> sc_sig1;
  sc_core::sc_signal<double> sc_sig2;

  adder add1("add1");
  add1.inp1(sig1);
  add1.inp2(sig2);
  add1.outp(sig3);

  delay del1("del1");
  del1(sig3, sig2); // positional binding

  const_src src1("src1");
  src1.outp(sig1);
  src1.val_in(sc_sig1);

  sink sink1("sink1");
  sink1.inp(sig3);
  sink1.out_sc(sc_sig2);

  sc_monitor sc_mon1("sc_mon1");
  sc_mon1.inp(sc_sig2);

  sca_util::sca_trace_file* tfa = sca_util::sca_create_tabular_trace_file("tdf_sync.dat");
  sca_util::sca_trace(tfa, sig1, "sig1");
  sca_util::sca_trace(tfa, sig2, "sig2");
  sca_util::sca_trace(tfa, sig3, "sig3");
  sca_util::sca_trace(tfa, sc_sig1, "sc_sig1");
  sca_util::sca_trace(tfa, sc_sig2, "sc_sig2");

  tfa->disable();

  sc_sig1.write(1.5);
  sc_core::sc_start(1.0, sc_core::SC_MS);

  sc_sig1.write(3.0);
  sc_core::sc_start(2.0, sc_core::SC_MS);

  tfa->enable();

  sc_sig1.write(4.0);

  sc_core::sc_start(2.0, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tfa);

  TEST_LABEL_END;

  return 0;

}
