/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/
// test sca_ac_f

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(mod_a)
{
  // port declarations
  sca_tdf::sca_in<double>  in;
  sca_tdf::sca_out<double> out;

  mod_a( sc_core::sc_module_name nm, double fc_, double h0_ = 1.0 )
    : in("in"), out("out"), fc(fc_), h0(h0_) {}

  // Implemented transfer function
  //
  //              1                num(0)
  // H(s) = -------------  = -----------------
  //               1          den(0) + den(1)s
  //        1 + ------- s
  //            2 PI fc

  void initialize()
  {
    num(0) = 1.0;
    den(0) = 1.0;
    den(1) = 1.0 /( 2.0 * M_PI * fc);
  }
  void processing()
  {
    out.write( ltf_nd( num, den, in.read(), h0) );
  }

  void ac_processing()
  {
    using namespace sca_ac_analysis;
    using namespace sca_util;

    sca_complex s;
    s = SCA_COMPLEX_J * 2.0 * M_PI * sca_ac_f(); // s=jw=j*2*pi*f

    // transfer function, reusing the numerator and denumerator
    sca_complex H_s = h0 * num(0) / ( den(0) + den(1) * s);
    sca_ac(out) = H_s * sca_ac(in);
  }

 private:
  sca_tdf::sca_ltf_nd ltf_nd;
  sca_util::sca_vector<double> num, den;
  double fc;
  double h0;
};


int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1, sig2;

  test_util::sin_src::params src_params;
  src_params.ampl = 1.0;
  src_params.freq = 1e3;
  src_params.offset = 0.0;
  src_params.sample_time = sca_core::sca_time(0.125, sc_core::SC_MS);

  test_util::sin_src src("src", src_params);
    src.outp(sig1);

  mod_a a("a", 1e3, 1.0);
    a.in(sig1);
    a.out(sig2);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("test04_ac");

  sca_util::sca_trace(tf, sig1, "sig1");
  sca_util::sca_trace(tf, sig2, "sig2");

  sc_core::sc_start(10, sc_core::SC_MS);

  tf->reopen("test04_ac"); // reopen to only store AC data

  tf->set_mode(sca_util::sca_ac_format(sca_util::SCA_AC_REAL_IMAG));

  sca_ac_analysis::sca_ac_start(10.0, 100.0e3, 200, sca_ac_analysis::SCA_LOG);

  sca_util::sca_close_tabular_trace_file(tf);

  std::cout << "Check frequency domain results..." << std::endl;

  test_util::check_results("test04_ac", 4, "test04");

  TEST_LABEL_END;

  return 0;
}
