/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

#ifndef TEST_CHECK_RESULTS_H_
#define TEST_CHECK_RESULTS_H_

#include <cmath>

// test utilities in dedicated namespace
// to avoid conflict with local definitions in tests
namespace test_util {

bool check_value(double val1, double val2, double abs_err = 1e-30, double rel_err = 1e-10)
{
  if ( (std::fabs(val1) < abs_err) || (std::fabs(val2) < abs_err) )
  {
    if (std::fabs(val1 - val2) > abs_err) return false;
  }
  else
  {
    if (std::fabs((val1 - val2) / (val1 + val2) / 2.0) > rel_err)
    {
      return false;
    }
  }

  if ( std::isnan(val1) && !std::isnan(val2) )
  {
    return false;
  }

  if ( std::isnan(val2) && !std::isnan(val1) )
  {
    return false;
  }

  return true;
}

void start_checker(std::string fname1, int cols = 1, std::string testname = "", double abs_err = 1e-30, double rel_err = 1e-10)
{
  std::string filein = fname1 + ".dat";

  if (testname == "") testname = fname1;
  std::string fileref = "./" + testname + "/golden/" + fname1 + ".ref";

  std::ifstream tdf_in(filein.c_str());
  std::ifstream tdf_ref(fileref.c_str());

  if (tdf_in.fail())
  {
    std::cerr << "Trace file " << filein << " not found!" << std::endl;
    return;
  }

  if (tdf_ref.fail())
  {
    std::cerr << "Trace file " << fileref << " not found!" << std::endl;
    return;
  }

  std::string line1, line2;

  unsigned long line_nr = 0;

  std::getline(tdf_in, line1);  // read (skip) header line of simulation data
  std::getline(tdf_ref, line2); // read (skip) header line of reference file
  line_nr++;

  while (true)
  {
    std::getline(tdf_in, line1);
    std::getline(tdf_ref, line2);

    if (tdf_ref.eof()) // if simulation longer than reference - dosent matter
    {
      break;
    }

    if (tdf_in.eof())
    {
      unsigned long lcnt = 0;
      while (std::getline(tdf_ref, line2))
      {
        lcnt++;
      }

      if (lcnt > (line_nr / 100)) // we tolerate 1% lines less
      {
        std::cout << "Reference file has: " << lcnt
                  << " more lines than trace file: " << line_nr << std::endl;
        return;
      }
      break;
    }

    line_nr++;

    std::istringstream str1(line1);
    std::istringstream str2(line2);

    std::istringstream str1_s(line1);
    std::istringstream str2_s(line2);

    double val1, val2;

    for (int i = 0; i < (cols + 1); i++)
    {
      std::string s1, s2;
      str1_s >> s1;
      str2_s >> s2;

      if (s1 == "*") val1 = NAN;
      else str1 >> val1;

      if (s2 == "*") val2 = NAN;
      else str2 >> val2;

      if (!check_value(val1, val2, abs_err, rel_err))
      {
        std::cerr << "Trace comparison check failed at line: " << line_nr << " column " << i+1 << std::endl;
        return;
      }
    }
  };

  std::cout << "Trace comparison check ok" << std::endl;
  return;
}

void check_results(std::string fname1, int cols = 1)
{
  start_checker(fname1, cols, "", 1e-14, 1e-10);
}

void check_results(std::string fname1, int cols, double abs_err)
{
  start_checker(fname1, cols, "", abs_err, 1e-10);
}

void check_results(std::string fname1, int cols, std::string testname,
                   double abs_err = 1e-14, double rel_err = 1e-10)
{
  start_checker(fname1, cols, testname, abs_err, rel_err);
}

} // namespace test_util

#endif // TEST_CHECK_RESULTS_H_
