/*****************************************************************************

  The following code is derived, directly or indirectly, from the SystemC
  source code Copyright (c) 1996-2001 by all Contributors.
  All Rights reserved.

  The contents of this file are subject to the restrictions and limitations
  set forth in the SystemC Open Source License Version 2.2 (the "License");
  You may not use this file except in compliance with such restrictions and
  limitations. You may obtain instructions on how to receive a copy of the
  License at http://www.systemc.org/. Software distributed by Contributors
  under the License is distributed on an "AS IS" basis, WITHOUT WARRANTY OF
  ANY KIND, either express or implied. See the License for the specific
  language governing rights and limitations under the License.

 *****************************************************************************/

/*****************************************************************************

  scfx_mant.h - 

  Original Author: Robert Graulich, Synopsys, Inc.
                   Martin Janssen,  Synopsys, Inc.

 *****************************************************************************/

/*****************************************************************************

  MODIFICATION LOG - modifiers, enter your name, affiliation, date and
  changes you are making here.

      Name, Affiliation, Date:
  Description of Modification:

 *****************************************************************************/


#ifndef SCFX_MANT_H
#define SCFX_MANT_H


#include "systemc/datatypes/fx/scfx_ieee.h"
#include "systemc/datatypes/fx/scfx_utils.h"
#include "systemc/kernel/sc_macros.h"


typedef unsigned long  word;
typedef unsigned short half_word;


// ----------------------------------------------------------------------------
//  CLASS : scfx_mant
//
//  Mantissa class.
// ----------------------------------------------------------------------------

class scfx_mant
{

    word* m_array;
    int   m_size;

public:

    explicit scfx_mant( size_t );
             scfx_mant( const scfx_mant& );

    scfx_mant& operator = ( const scfx_mant& );

    ~scfx_mant();

    void clear();

    void resize_to( int, int = 0 );

    int size() const;

    word  operator [] ( int ) const;
    word& operator [] ( int );

    half_word  half_at( int ) const;
    half_word& half_at( int );

    half_word* half_addr( int = 0 ) const;

private:

    static word* alloc( size_t );
    static void free( word*, size_t );

    static word* alloc_word( size_t size );
    static void free_word( word* array, size_t size );

};


// IIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIII

inline
int
scfx_mant::size() const
{
    return m_size;
}


inline
word*
scfx_mant::alloc( size_t size )
{
#if defined( SCFX_BIG_ENDIAN )
    return alloc_word( size ) + ( size - 1 );
#elif defined( SCFX_LITTLE_ENDIAN )
    return alloc_word( size );
#endif
}

inline
void
scfx_mant::free( word* mant, size_t size )
{
#if defined( SCFX_BIG_ENDIAN )
    free_word( mant - ( size - 1 ), size );
#elif defined( SCFX_LITTLE_ENDIAN )
    free_word( mant, size );
#endif
}

inline
word
scfx_mant::operator[]( int i ) const
{
    SC_ASSERT_( i >= 0 && i < m_size, "mantissa index out of range" );
#if defined( SCFX_BIG_ENDIAN )
    return m_array[-i];
#elif defined( SCFX_LITTLE_ENDIAN )
    return m_array[i];
#endif
}

inline
word&
scfx_mant::operator[]( int i )
{
    SC_ASSERT_( i >= 0 && i < m_size, "mantissa index out of range" );
#if defined( SCFX_BIG_ENDIAN )
    return m_array[-i];
#elif defined( SCFX_LITTLE_ENDIAN )
    return m_array[i];
#endif
}

inline
scfx_mant::scfx_mant( size_t size )
{
    m_array = alloc( m_size = size );
}

inline
scfx_mant::scfx_mant( const scfx_mant& rhs )
{
    m_array = alloc( m_size = rhs.m_size );
    for( int i = 0; i < m_size; i ++ )
    {
        (*this)[i] = rhs[i];
    }
}

inline
scfx_mant&
scfx_mant::operator = ( const scfx_mant& rhs )
{
    if( &rhs != this )
    {
        if( m_size != rhs.m_size )
	{
	    free( m_array, m_size );
	    m_array = alloc( m_size = rhs.m_size );
	}

	for( int i = 0; i < m_size; i ++ )
	{
	    (*this)[i] = rhs[i];
	}
    }
    return *this;
}

inline
scfx_mant::~scfx_mant()
{
    if( m_array != 0 )
    {
        free( m_array, m_size );
    }
}

inline
void
scfx_mant::clear()
{
    for( int i = 0; i < m_size; i ++ )
    {
        (*this)[i] = 0;
    }
}

inline
void
scfx_mant::resize_to( int size, int restore )
{
    if( size == m_size )
    {
        return;
    }

    if( ! m_array )
    {
        m_array = alloc( m_size = size );
    }
    else
    {
        word* p = alloc( size );

	if( restore )
	{
	    int end = sc_min( size, m_size );
	    if( restore == 1 )		// msb resized -> align at 0
	    {
	        for( int i = 0; i < size; i ++ )
		{
		    if( i < end )
		    {
#if defined( SCFX_BIG_ENDIAN )
		        p[-i] = m_array[-i];
#elif defined( SCFX_LITTLE_ENDIAN )
			p[i] = m_array[i];
#endif
		    }
		    else
		    {
#if defined( SCFX_BIG_ENDIAN )
		        p[-i] = 0;
#elif defined( SCFX_LITTLE_ENDIAN )
			p[i] = 0;
#endif
		    }
		}
	    }
	    else			// lsb resized -> align at size-1
	    {
	        for( int i = 0; i < size; i ++ )
		{
		    if( i < end )
		    {
#if defined( SCFX_BIG_ENDIAN )
		        p[-size+1+i] = m_array[-m_size+1+i];
#elif defined( SCFX_LITTLE_ENDIAN )
			p[size-1-i] = m_array[m_size-1-i];
#endif
		    }
		    else
		    {
#if defined( SCFX_BIG_ENDIAN )
		        p[-size+1+i] = 0;
#elif defined( SCFX_LITTLE_ENDIAN )
			p[size-1-i] = 0;
#endif
		    }
		}
	    }
	}

	free( m_array, m_size );
	m_array = p;
	m_size = size;
    }
}

inline
half_word
scfx_mant::half_at( int i ) const
{
    SC_ASSERT_( ( i >> 1 ) >= 0 && ( i >> 1 ) < m_size,
		"mantissa index out of range" );
#if defined( SCFX_BIG_ENDIAN )
    return reinterpret_cast<half_word*>( m_array )[-i];
#elif defined( SCFX_LITTLE_ENDIAN )
    return reinterpret_cast<half_word*>( m_array )[i];
#endif
}

inline
half_word&
scfx_mant::half_at( int i )
{
    SC_ASSERT_( ( i >> 1 ) >= 0 && ( i >> 1 ) < m_size,
		"mantissa index out of range" );
#if defined( SCFX_BIG_ENDIAN )
    return reinterpret_cast<half_word*>( m_array )[-i];
#elif defined( SCFX_LITTLE_ENDIAN )
    return reinterpret_cast<half_word*>( m_array )[i];
#endif
}

inline
half_word*
scfx_mant::half_addr( int i ) const
{
    SC_ASSERT_( i >= 0 && i < m_size, "mantissa index out of range" );
#if defined( SCFX_BIG_ENDIAN )
    return reinterpret_cast<half_word*>( m_array - i ) + 1;
#elif defined( SCFX_LITTLE_ENDIAN )
    return reinterpret_cast<half_word*>( m_array + i );
#endif
}


// ----------------------------------------------------------------------------
//  one's complement of a mantissa
// ----------------------------------------------------------------------------

inline
void
complement( scfx_mant& target, const scfx_mant& source, int size )
{
    for( int i = 0; i < size; i ++ )
    {
        target[i] = ~source[i];
    }
}


// ----------------------------------------------------------------------------
//  increment mantissa
// ----------------------------------------------------------------------------

inline
void
inc( scfx_mant& mant )
{
    for( int i = 0; i < mant.size(); i ++ )
    {
        if( ++ mant[i] )
	{
	    break;
	}
    }
}


// ----------------------------------------------------------------------------
//  CLASS : scfx_mant_ref
//
//  Mantissa reference class.
// ----------------------------------------------------------------------------

class scfx_mant_ref
{

    scfx_mant* m_mant;
    bool       m_not_const;

public:

    scfx_mant_ref();
    scfx_mant_ref( const scfx_mant& );
    scfx_mant_ref( scfx_mant* );

    scfx_mant_ref& operator = ( const scfx_mant& );
    scfx_mant_ref& operator = ( scfx_mant* );

    ~scfx_mant_ref();

    operator scfx_mant&();

    word operator [] ( int );

private:

    void remove_it();

    scfx_mant_ref( const scfx_mant_ref& );
    scfx_mant_ref& operator = ( const scfx_mant_ref& );

    void* operator new( size_t sz ) { return ::operator new( sz ); }

};


// IIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIII

inline
void
scfx_mant_ref::remove_it()
{
    if( m_not_const )
    {
        delete m_mant;
    }
}

inline
scfx_mant_ref::scfx_mant_ref()
: m_mant( 0 ), m_not_const( false )
{}

inline
scfx_mant_ref::scfx_mant_ref( const scfx_mant& mant )
: m_mant( const_cast<scfx_mant*>( &mant ) ), m_not_const( false )
{}

inline
scfx_mant_ref::scfx_mant_ref( scfx_mant* mant )
: m_mant( mant ), m_not_const( true )
{}

inline
scfx_mant_ref&
scfx_mant_ref::operator = ( const scfx_mant& mant )
{
    remove_it();

    m_mant = const_cast<scfx_mant*>( &mant );
    m_not_const = false;

    return *this;
}

inline
scfx_mant_ref&
scfx_mant_ref::operator = ( scfx_mant* mant )
{
    remove_it();

    m_mant = mant;
    m_not_const = true;

    return *this;
}

inline
scfx_mant_ref::~scfx_mant_ref()
{
    remove_it();
}

inline
scfx_mant_ref::operator scfx_mant&()
{
    // SC_ASSERT_( m_not_const, "not allowed to modify mant" );
    return *m_mant;
}

inline
word
scfx_mant_ref::operator [] ( int i )
{
    return (*m_mant)[i];
}


#endif

// Taf!
