/* Copyright
* ========================================================================================
* Project:       Reference DRAM Model
* Author:        Lei Liang, KTH
* ID:            sampdram, v 1.1, 2010-12-06
*
* Description:	 A sample DRAM Model with certain delay (5 cycle) to read and write operation
* 				 This program follows the OCP-IP protocol which supports following operation
* 				 A: MRMD and SRMD burst read and write operation
* 				 B: Posted write and read operation
*
* ========================================================================================
* Version History
* ========================================================================================
* Version 1.1: Fully support all the burst read and write operation
* Version 1.0: First executable version
* ========================================================================================
*/

#ifndef SAMPDRAM_H_
#define SAMPDRAM_H_


#include <iostream>
#include "systemc.h"
#include "Memorymem.h"
#include "ocpip.h"
#include <deque>

template <unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
class sampdram : public sc_core::sc_module {

public:

  /*Define the Data and Address type in OCP-IP communication protocol*/
  typedef typename ocpip::ocp_data_class_unsigned<BUSWIDTH,ADDRWIDTH>::DataType Td;
  typedef typename ocpip::ocp_data_class_unsigned<BUSWIDTH,ADDRWIDTH>::AddrType Ta;

  /*Define the OCP-IP transaction*/
  ocpip::ocp_slave_socket_tl1<sizeof(Td)*8> ocpInitPort;

  sc_core::sc_in_clk clk;

  SC_HAS_PROCESS(sampdram);


  sampdram(sc_core::sc_module_name name_, int delay);		// constructor

  ~sampdram();									// destructor

  void proc();									// The main method

  /*Functions used in the main thread*/

  void queuePop();								// pop the transaction pointer out of the queue
  void burstwrite();							// write the data into the Memory
  void push();									// push the transaction pointer into the queue
  void extraendreq();							// This is used to send extra END_REQ

  tlm::tlm_sync_enum nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim);
    											// OCP request group
  sc_core::sc_time time;

private:

  tlm::tlm_generic_payload* req;				//the ongoing REQ transaction pointer
  tlm::tlm_generic_payload* reqTemp;			//to store the blocked transaction
  tlm::tlm_generic_payload* reqTempburst;		//to store the blocked burst transaction pointer
  tlm::tlm_generic_payload* response;			//the ongoing RESPONSE transaction pointer
  tlm::tlm_phase phase;

  /*
   * The queuedata is the data that need to be put into the FIFO
   * Empty: True means the pointer is valid and vice versa
   * rsp: Store the pointer of the transaction
   */

  struct queuedata{
	 queuedata():empty(true),rsp(NULL){}
	 queuedata(tlm::tlm_generic_payload* rsp):empty(false),rsp(rsp){}

 	bool empty;
 	tlm::tlm_generic_payload* rsp;
  };

  /*
   * instantiation of a FIFO which used to generate certain for read operation
   */
  std::deque<queuedata> queue;

  /*
   * instantiation of a memory
   */
  Memorymem<ocpip::ocp_data_class_unsigned<BUSWIDTH,ADDRWIDTH> > m_Memory;


  int m_delayCycles;								//cycles of delay
  int m_burstLength;								//To store the length of burst
  bool m_respOngoing;								//To verify if END_RESP is sent
  bool m_burstOn;									//To verify if burst is on
  bool m_srmdOn;									//To verify SRMD and MRMD
  bool m_posted;									//To verify if posted write is needed
  bool m_needSrmdpost;								//To verify if Srmd write ACK is needed
  bool m_needMrmdpost;								//To verify if Mrmd write ACK is needed
  bool m_queueStopped;								//To verify if queue has been stopped
  bool m_needendsrmd;								//To verify is END_REQ for blocked srmd is needed

  };


#endif
