/*
 *  Copyright
 * ========================================================================================
 * Project:		Accurate DRAM Model
 * Author:		Yi Wang, KTH
 * ID:			adm_request_decoder.h, v1.2, 2010/12/08
 *
 * Description:	A command decoder that has 3 major functions
 * 				1. Realize interface access for the DRAM controller sending requests
 * 				2. Translate requests from the controller to a series of DRAM commands
 * 				3. Issue these commands through a internal interface between the decoder
 * 					and the DRAM core
 *
 * ========================================================================================
 * Version History
 * ========================================================================================
 * Version 1.2: Change the external delay policy by rejecting the request before the current
 * 				operation finishes when detecting a hop row in the comming transmission.
 * Version 1.1: 1. fix the bug of overlapped refresh time and burst time
 * 				2. fix the runtime synchronization mechanism with the controller
 * Version 1.0: First Executable Version
 * Version 0.1: Draft version
 * ========================================================================================
 */

#ifndef ADA_REQUEST_DECODER_H_
#define ADA_REQUEST_DECODER_H_

#define DEBUG 0
#define REFRESH 0

#include "systemc.h"
#include "adm_dram_ctrl_ifs.h"
#include "adm_basic_types.h"
#include "adm_configuration.h"


class adm_request_decoder : public sc_module, public adm_dram_if
{
public:
	/* IO ports */
	sc_in<bool> clk;
	sc_export<adm_dram_if> decoder_target_port;
	sc_port<adm_core_if> decoder_initiator_port;

	/* Constructors */
	SC_HAS_PROCESS(adm_request_decoder);
	adm_request_decoder(sc_module_name nm, adm_configuration admConfig)
			: sc_module(nm)
			, m_admConfig(admConfig)
			, m_noRead(false)
			, m_noWrite(false)
			, m_isRefreshing(false)
			, m_refreshWait(0)
			, m_noReadCountDown(0)
			, m_noWriteCountDown(0)
			, m_tPreparation(0)
			, decoder_initiator_port("Decoder_Initiator_Port")
			, decoder_target_port("Decoder_Target_Port")
	{
		/* Initiate IO ports */
		decoder_target_port(*this);

		/* Initiate Activate Information Recorder */
		int bankCount = 1 << m_admConfig.m_bankAddressWidth;
		m_activeRow = new int[bankCount];
		for (int i = 0; i < bankCount; ++i) {
			m_activeRow[i] = -1;
		}


		SC_THREAD(timeCountDown);
		sensitive<<clk.pos();
		dont_initialize();
	}

	void timeCountDown()
	{

		while (true) {
			wait();
			/* Count Down for refresh */
			m_refreshWait = (++m_refreshWait) % (m_admConfig.m_refreshPeriod);
			if (m_refreshWait == 0) {												/* Start Refresh */
				m_isRefreshing = true;
				m_noReadCountDown += m_admConfig.m_refreshDuration;
				m_noWriteCountDown += m_admConfig.m_refreshDuration;				/* further delay the next operation */
				if(REFRESH)
					cout << "@" << sc_time_stamp() << "Decoder: Refresh start" << endl;
			}
			else if ((m_refreshWait == m_admConfig.m_refreshDuration)
						&& (m_isRefreshing == true)){
				if(REFRESH)
					cout << "@" << sc_time_stamp() << "Decoder: Refresh Ends" << endl;
				m_isRefreshing = false;												/* Notify the controller */
				int bankCount = 1 << m_admConfig.m_bankAddressWidth;
				m_activeRow = new int[bankCount];
				for (int i = 0; i < bankCount; ++i) {
					m_activeRow[i] = -1;
				}																	/* No bank is active after a refresh */
			}

			/* Count Down for noRead */
			if (m_noReadCountDown > 1) {
				m_noReadCountDown--;
			}
			else{
				m_noRead = false;
			}

			/* Count Down for noWrite */
			if (m_noWriteCountDown > 1) {
				m_noWriteCountDown--;
			}
			else{
				m_noWrite = false;
			}
			wait(SC_ZERO_TIME);														/* Make sure that read permission can be received by the controller in time */

		}
	}

	bool putWriteRequest(uint64 address, const adm_data & data)
	{
		/* If current operation is not finished, reject the coming request */
		if(m_noWrite) {
			return false;
		}
		/*
		 * If preparation delay is applied, record the duration for the delay when accept the command
		 * then reject the request after current burst completes
		 */
		int tPreparation = getPreparetionDelay(address);
		if (tPreparation != 0) {
			m_tPreparation = tPreparation;
			m_noRead = true;
			m_noWrite = true;
			if (DEBUG) {
				cout << endl << "@" << sc_time_stamp() << " Decoder: Remaining cycles: " << decoder_initiator_port->getRemainCycles() << endl;
			}
			int remainCycles = decoder_initiator_port->getRemainCycles();
			m_noReadCountDown = remainCycles;
			m_noWriteCountDown = remainCycles;
			if (remainCycles != 0) {
				return false;
			}
		}
		/* If no preparation delay is applied, perform both remaining external delay */
		if(DEBUG)
			cout << endl << "@" << sc_time_stamp() << " Decoder: Write request accepted" << endl;
		m_noRead = true;
		m_noWrite = true;

		decoder_initiator_port->virtualDelay(m_tPreparation);
		int burstCycle = ceil((float)data.getLength() / m_admConfig.m_dataWidth / m_admConfig.m_burstLength) * m_admConfig.m_burstLength;

		m_noReadCountDown = m_tPreparation + m_admConfig.m_tDQSS + burstCycle + m_admConfig.m_tWTR;
		m_noWriteCountDown = m_tPreparation + burstCycle;

		decoder_initiator_port->coreWrite(address, data, m_tPreparation, burstCycle);
		m_tPreparation = 0;
		return true;
	}

	bool putReadRequest(uint64 address, int length)
	{
		/* If current operation is not finished, reject the coming request */
		if(m_noRead){
			return false;
		}

		/*
		 * If preparation delay is applied, record the duration for the delay when accept the command
		 * then reject the request after current burst completes
		 */
		int tPreparation = getPreparetionDelay(address);
		if (tPreparation != 0) {
			m_tPreparation = tPreparation;
			m_noRead = true;
			m_noWrite = true;
			if (DEBUG) {
				cout << endl << "@" << sc_time_stamp() << " Decoder: Remaining cycles: " << decoder_initiator_port->getRemainCycles() << endl;
			}
			int remainCycles = decoder_initiator_port->getRemainCycles();
			m_noReadCountDown = remainCycles;
			m_noWriteCountDown = remainCycles;
			if (remainCycles != 0) {
				return false;
			}
		}
		/* If no preparation delay is applied, perform both remaining external and internal delay */
		if(DEBUG)
			cout << endl << "@" << sc_time_stamp() << " Decoder: Read request accepted" << endl;
		m_noRead = true;
		m_noWrite = true;

		decoder_initiator_port->virtualDelay(m_tPreparation);
		int burstCycle = ceil((float)length / m_admConfig.m_dataWidth / m_admConfig.m_burstLength) * m_admConfig.m_burstLength;

		m_noReadCountDown = m_tPreparation + burstCycle;
		m_noWriteCountDown = m_tPreparation + m_admConfig.m_tCAS + burstCycle;

		decoder_initiator_port->coreRead(address, length, m_tPreparation, burstCycle);
		m_tPreparation = 0;
		return true;
	}



	int getPreparetionDelay(uint64 address)
	{
		address >>= (m_admConfig.m_mbAddressWidth + m_admConfig.m_columnAddressWidth);		/* Skip don't care bits */
		int bankRequested = (1 << m_admConfig.m_bankAddressWidth) - 1;					 	/* Mask for getting requested Bank number */
		bankRequested = address & bankRequested;											/* Get requested bank number */
		int rowRequested = address >> (m_admConfig.m_bankAddressWidth);						/* Get requested row number */
		if(DEBUG){
			cout << "@" << sc_time_stamp() << " Decoder: Request to access bank " << bankRequested << endl;
			cout << "@" << sc_time_stamp() << " Decoder: Request to access row " << rowRequested << endl;
		}
		if (m_activeRow[bankRequested] == -1) {												/* If after a refresh */
			m_activeRow[bankRequested] = rowRequested;
			if(DEBUG)
				cout << "@" << sc_time_stamp() << " Decoder: Delay " << m_admConfig.m_tRCD << " cycle due to row reactivation" << endl;
			return m_admConfig.m_tRCD;
		}
		else if ((m_activeRow[bankRequested] != -1)
				&& (m_activeRow[bankRequested] != rowRequested)){							/* If hopping row needed */
			m_activeRow[bankRequested] =  rowRequested;
			if(DEBUG)
				cout << "@" << sc_time_stamp() << " Decoder: Delay " << m_admConfig.m_tHopRow << " cycle due to row hopping" << endl;
			return m_admConfig.m_tHopRow;
		}
		else{
			return 0;
		}
	}

	~adm_request_decoder()
	{

	}
private:
	adm_configuration m_admConfig;
	int m_refreshWait;
	bool m_noRead;
	int m_noReadCountDown;
	bool m_noWrite;
	int m_noWriteCountDown;
	bool m_isRefreshing;
	int *m_activeRow;
	int m_tPreparation;
};

#endif /* ADA_REQUEST_DECODER_H_ */
