/*
 *  Copyright
 * ========================================================================================
 * Project:		Accurate DRAM Model
 * Author:		Yi Wang, KTH
 * ID:			adm_configuration.h, v1.2, 2010/11/27
 *
 * Description:	A set of DRAM parameters that will affect the DRAM operation
 *
 * ========================================================================================
 * Version History
 * ========================================================================================
 * Version 1.2: Remove all DRAM type specific parameters: prefetch buffer length, column width
 * 				Add IO parameters: IO bit width, Mini burst length.
 * 				Correct mbAddressWidth calculation algorithm.
 * Version 1.1: Add member m_clockPeriod
 *				Add new constructor that configure all parameters according to the OCP
 *				Parameter file given
 * 				Add new method parseConfig() to help the parameterized constructor set parameters
 * 				specified in the file
 * Version 1.0: First Executable Version
 * Version 0.1: Draft version
 * ========================================================================================
 */

#ifndef ADM_CONFIGURATION_H_
#define ADM_CONFIGURATION_H_
#include <iostream>
#include <fstream>
#include <cmath>
#include <sstream>
#include <string>
#include "adm_basic_types.h"
#include <cstdlib>

class adm_configuration {
public:
	/* Constructors */
	adm_configuration()
			: m_refreshDuration(2)
			, m_refreshPeriod(16)
			, m_addressBusWidth(32)
			, m_bankAddressWidth(2)
			, m_rowAddressWidth(13)
			, m_columnAddressWidth(15)
			, m_tHopRow(10)
			, m_tDQSS(1)
			, m_tRCD(3)
			, m_dataWidth(8)
			, m_dataRate(DDR1)
	{
		m_mbAddressWidth = m_addressBusWidth - m_bankAddressWidth - m_rowAddressWidth
				- m_columnAddressWidth;
	}

	adm_configuration(const char* admConfigFileName)
	{
		std::string nameField;
		std::string valueField;
		std::string oneLine;

		/* Get parameters from the admParams file */
		ifstream inputfile(admConfigFileName);
		inputfile.setf(std::ios::skipws);
		while (inputfile) {
			getline(inputfile,oneLine);
			if (oneLine[0] == '#' || oneLine.empty()) {
	//			cout << "Comments" << endl;
			}
			else {
	//			cout << endl << "Data Field" << endl;
				std::stringstream ss_stream;
				ss_stream << oneLine;
				ss_stream >> nameField;
				ss_stream >> valueField;
				valueField = valueField.substr(2);
				phaseConfig(nameField, valueField);
			}
		}
		inputfile.close();

		/* Generate the rest parameters that dependent on the parameters specified in the file */
		int temp = m_dataWidth;
		m_mbAddressWidth = 0;
		while (temp > 1) {
			m_mbAddressWidth++;
			temp = temp >> 1;
		}
		cout << "Mini Byte width = " << m_mbAddressWidth << " bits" << endl;

		cout << "......Configuration Complete" << endl << endl;
	}


	void phaseConfig(std::string name, std::string value)
	{
		//cout << "name = " << name << " ;value = " << value << endl;
		if (name.find("clockPeriod") != std::string::npos) {
			m_clockPeriod = atoi(value.c_str());
			cout << "clock period = " << m_clockPeriod << " ns" << endl;
		}
		else if (name.find("dataRate") != std::string::npos) {
			m_dataRate = (DataRate)atoi(value.c_str());
			switch (m_dataRate) {
				case 0:
					cout << "data rate = SDR" << endl;
					break;
				case 1:
					cout << "data rate = DDR1" << endl;
					break;
				case 2:
					cout << "data rate = DDR2" << endl;
					break;
				case 3:
					cout << "data rate = DDR3" << endl;
					break;
				default:
					SC_REPORT_ERROR(0,"Data Rate Parameter Error!");
					break;
			}

		}
		else if (name.find("refreshPeriod") != std::string::npos) {
			m_refreshPeriod = ceil((float)atoi(value.c_str()) / m_clockPeriod);
			cout << "refresh Period = " << m_refreshPeriod << " cycles" << endl;
		}
		else if (name.find("refreshDuration") != std::string::npos) {
			m_refreshDuration = ceil((float)atoi(value.c_str()) / m_clockPeriod);
			cout << "refresh Duration = " << m_refreshDuration << " cycles" << endl;
		}
		else if (name.find("addressBusWidth") != std::string::npos) {
			m_addressBusWidth = atoi(value.c_str());
			cout << "address bus width = " << m_addressBusWidth << " bits" << endl;
		}
		else if (name.find("bankAddressWidth") != std::string::npos) {
			m_bankAddressWidth = atoi(value.c_str());
			cout << "bank address width = " << m_bankAddressWidth << " bits" << endl;
		}
		else if (name.find("rowAddressWidth") != std::string::npos) {
			m_rowAddressWidth = atoi(value.c_str());
			cout << "row address width = " << m_rowAddressWidth << " bits" << endl;
		}
		else if (name.find("columnAddressWidth") != std::string::npos) {
			m_columnAddressWidth = atoi(value.c_str());
			cout << "address bus width = " << m_columnAddressWidth << " bits" << endl;
		}
		else if (name.find("dataWidth") != std::string::npos) {
			m_dataWidth = atoi(value.c_str());
			cout << "Data Width = " << m_dataWidth << " bits" << endl;
		}
		else if (name.find("burstLength") != std::string::npos) {
			m_burstLength = atoi(value.c_str());
			cout << "Mini Burst Length = " << m_burstLength << endl;
		}
		else if (name.find("tHopRow") != std::string::npos) {
			m_tHopRow = ceil((float)atoi(value.c_str()) / m_clockPeriod);
			cout << "tHopRow = " << m_tHopRow << " cycles" << endl;
		}
		else if (name.find("tRCD") != std::string::npos) {
			m_tRCD = ceil((float)atoi(value.c_str()) / m_clockPeriod);
			cout << "tRCD = " << m_tRCD << " cycles" << endl;
		}
		else if (name.find("tCAS") != std::string::npos) {
			m_tCAS = ceil((float)atoi(value.c_str()) / m_clockPeriod);
			cout << "tCAS = " << m_tCAS << " cycles" << endl;
		}
		else if (name.find("tDQSS") != std::string::npos) {
			m_tDQSS = ceil((float)atoi(value.c_str()) / m_clockPeriod);
			cout << "tDQSS = " << m_tDQSS << " cycles" << endl;
		}
		else if (name.find("tWTR") != std::string::npos) {
			m_tWTR = ceil((float)atoi(value.c_str()) / m_clockPeriod);
			cout << "tWTR = " << m_tWTR << " cycles" << endl;
		}
	}

	~adm_configuration() {
	}
    /*
     * Parameters about Refresh
     */
    int m_clockPeriod;			/* I/O Clock Cycle */
	int m_refreshPeriod;        /* Time between 2 DRAM Refresh */
	int m_refreshDuration; 		/* Time of one Refresh Operation */

	/*
	 * Parameters about Addressing
	 */
	int m_addressBusWidth;			/* Width of address bus */
	int m_bankAddressWidth;			/* Band address bits */
	int m_rowAddressWidth;			/* Row address bits */
	int m_columnAddressWidth;		/* Column address bits */
	int m_mbAddressWidth;			/* Min Byte address bits, which should be skipped during decoding */

	/*
	 * Parameters about data accessing
	 */
	enum DataRate {SDR,DDR1,DDR2,DDR3} m_dataRate;
	int m_dataWidth;				/* Bit-Width of the data bus, in byte*/
	int m_burstLength;				/* minimum burst length, in byte */
	int m_tHopRow;					/* Delay cycles caused by hop rows */
	int m_tRCD;						/* Cycles between a Activate and a RD/WR command */
	int m_tCAS;						/* Cycles between Read and the first response */
	int m_tDQSS;					/* Cycles between Write and the first data gets registered */
	int m_tWTR;						/* Cycles between a Read and the last write data gets registered */
};

#endif /* DRAM_CONFIGURATION_H_ */
