/**
 *
 * @file event.cc
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: event.cc 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#include "event.hh"

#include <iostream>
#include <iomanip>

#include <systemc>


namespace sctg
{
   Event::Event(sc_core::sc_module_name name, 
		Configuration& config,
		const boost::property_tree::ptree& pt)
      :
      sc_core::sc_module(name),
      _name(name),
      _config(config)
   {
      SC_THREAD(thread);

      _port   = pt.get<unsigned long int>("<xmlattr>.out_port_id");
      _amount = pt.get<unsigned long int>("<xmlattr>.amount");    
      _id     = pt.get<unsigned long int>("<xmlattr>.id");
      _offset = pt.get_optional<double>("<xmlattr>.offset");
      _period = pt.get<double>("<xmlattr>.period", 0.0);
      _count  = pt.get<unsigned long int>("<xmlattr>.count", 0);
      _prob   = pt.get<double>("<xmlattr>.prob", 1.0);
      _timesHappened = 0;
   }

   Event::~Event()
   {
      if(_config.getSummaryStream())
      {
	 **_config.getSummaryStream()
	    << "- Event " << std::setw(20) << std::left << _name 
	    << " happened " << std::setw(6) << _timesHappened 
	    << " times" << std::endl;
      }
   }
  
   void Event::thread()
   {
      // Wait offset if present
      if(_offset)
      {
	 sc_core::wait(*_offset, sc_core::SC_SEC);
      }
    
      // Generate _count amount of events or infinite amount if it's zero
      if(_count != 0)
      {
	 for(;_count > 0; --_count)
	 {
	    if(_config.random() < _prob)
	    {
	       _timesHappened++;
	       /*std::cout << "Event (" << _name << ") happened at "
		 << sc_core::sc_time_stamp() << std::endl;*/
	       send();
	    }
	    sc_core::wait(_period, sc_core::SC_SEC);
	 }
      }
      else
      {
	 while(true)
	 {
	    if(_config.random() < _prob)
	    {
	       _timesHappened++;
	       /*std::cout << "Event (" << _name << ") happened at "
		 << sc_core::sc_time_stamp() << std::endl;*/
	       send();	  
	    }  
	    sc_core::wait(_period, sc_core::SC_SEC);
	 }
      }
   }

   void Event::mapPe(ProcessingElement* pe)
   {
      std::cout << "PE " << pe->getName() << " mapped to event "
		<< _id << std::endl;
      _owners.push_back(pe);
   }

   void Event::send()
   {
      tgToken token;
      token.source = 0;
      token.destination = _config.getDestination(_port);
      token.size = _amount;
      token.isEvent = true;
      token.timeSent = sc_core::sc_time_stamp();
      for(unsigned int i = 0; i < _owners.size(); ++i)
      {
	 _owners.at(i)->receiveEvent(token);
      }
   }

   unsigned long int Event::getOutPort()
   { return _port; }

   unsigned long int Event::getTimesHappened()
   { return _timesHappened; }

   unsigned long int Event::getId()
   { return _id; }

   std::string& Event::getName()
   { return _name; }
}


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
