/**
 *
 * @file processing_element.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: processing_element.hh 1408 2010-09-01 14:26:53Z lehton87 $
 *
 */


#ifndef SCTG_PROCESSING_ELEMENT_HH
#define SCTG_PROCESSING_ELEMENT_HH

#include "resource.hh"
#include "amount.hh"
#include "task.hh"
#include "buffer.hh"
#include "common.hh"

#include <systemc>
#include <boost/property_tree/ptree.hpp>
#include <memory>
#include <map>

namespace sctg
{
   class ProcessingElement : public Resource
   {
   public:

      SC_HAS_PROCESS(ProcessingElement);

      //* Constructor
      ProcessingElement(sc_core::sc_module_name name, 
			const boost::property_tree::ptree& pt,
			const boost::property_tree::ptree& peLib,
			sctg::Configuration& config);
    
      //* Destructor
      virtual ~ProcessingElement();

      //* Main execution thread which consumes simulation time
      void thread();

      // Returns performance factors
      double getIntOps();
      double getFloatOps();
      double getMemOps();

      // Return communication costs (clock cycles)
      unsigned long int getIntraGroupRxCost();
      unsigned long int getIntraGroupTxCost();
      unsigned long int getInterGroupRxCost();
      unsigned long int getInterGroupTxCost();
      unsigned long int getInterPeRxCost();
      unsigned long int getInterPeTxCost();
    
      //* Maps task to this PE
      void mapTask(Task* task);

      //* True if PE has task with this in_port
      bool hasInPort(unsigned long int port);

      //* PE receives a token from event
      void receiveEvent(tgToken token);

      //* Returns measurement information
      const sctg::PeMeasurements& getMeasurements();

      std::queue<tgToken>& getTokenQueue(); 
      
      //* Returns average utilization from the beginning to this point
      double getAvgUtilization();

      void updateUnfinishedTokensLatency();

   private:

      //* Returns next task to be executed
      sctg::Task* schedule();

      //* updates measurements
      void updateMeasurements();

      //* TX DMA handler
      void txDma();

      //* RX DMA handler
      void rxDma();

      // Performance factors (operations per clock cycle)
      double _intOps;
      double _floatOps;
      double _memOps;    

      // Communication costs
      Amount<unsigned long int>* _intraGroupRxCost;
      Amount<unsigned long int>* _intraGroupTxCost;
      Amount<unsigned long int>* _interGroupRxCost;
      Amount<unsigned long int>* _interGroupTxCost;
      Amount<unsigned long int>* _interPeRxCost;
      Amount<unsigned long int>* _interPeTxCost;

      // Whether DMA is enabled or not
      bool _dmaEnabled;

      // List of tasks on this PE
      std::vector<Task*> _tasks;
      std::queue<Task*> _fifoScheduler;

      Configuration& _config;

      std::auto_ptr<Buffer> _buffer;
      unsigned long int _packetSize;
      std::map<unsigned long int, tgToken> _sendTokens;

      sc_core::sc_time  _cycleLength;

      PeMeasurements _measurements;

      sc_core::sc_event _eventEvent;

      Task* _currentTask;

      enum PeState {IDLE, EXECUTING, SENDING, INTRA_TX_WAIT,
		    TX_WAIT, RX_WAIT};
      PeState _currentState;
    
      sc_core::sc_time _measureStart;
      sc_core::sc_time _measureEnd;

      std::queue<tgPacket*> _txDmaQueue;
      sc_core::sc_event     _txDmaEvent;

      std::queue<tgPacket*> _rxDmaQueue;
      sc_core::sc_event     _rxDmaEvent;

      std::queue<tgToken>   _tokenQueue;

   };
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
