/**
 *
 * @file simple_bus.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: simple_bus.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#ifndef ASEBT_SIMPLE_BUS_HH
#define ASEBT_SIMPLE_BUS_HH

#include "noc_conf_if.hh"

#include "tlm.h"
#ifdef MTI_SYSTEMC
#include "simple_initiator_socket.h"
#include "simple_target_socket.h"
#include "peq_with_get.h"
#else
#include "tlm_utils/simple_initiator_socket.h"
#include "tlm_utils/simple_target_socket.h"
#include "tlm_utils/peq_with_get.h"
#endif
#include <systemc>

#include <stdexcept>
#include <iostream>
#include <queue>


namespace asebt
{

   template<unsigned int N_AGENTS,
	    unsigned int DATA_WIDTH>
   class SimpleBus : public sc_core::sc_module
   {
   public:
      
      SC_HAS_PROCESS(SimpleBus);
      
      tlm_utils::simple_initiator_socket_tagged
      <SimpleBus, DATA_WIDTH> initSockets[N_AGENTS];
      tlm_utils::simple_target_socket_tagged
      <SimpleBus, DATA_WIDTH> targetSockets[N_AGENTS];

      //* Constructor
      SimpleBus(sc_core::sc_module_name name,
		sctg::NocConfIf* confIf)
	 : sc_module(name),
	   _confIf(confIf),
	   _rxPeq(0)
      {
	 for(unsigned int i = 0; i < N_AGENTS; ++i)
	 {
	    initSockets[i].
	       register_nb_transport_bw(this, &SimpleBus::nb_transport_bw, i);
	    targetSockets[i].
	       register_nb_transport_fw(this, &SimpleBus::nb_transport_fw, i);
	    
	 }
	 
	 _rxPeq = new 
	    tlm_utils::peq_with_get<tlm::tlm_generic_payload>("rxpeq");	 

	 SC_THREAD(thread);
      }

      //* Destructor
      ~SimpleBus()
      {
	 if(_rxPeq) {delete _rxPeq; _rxPeq = 0;}
      }

   private:

      // Disable operator= and copy constructor
      SimpleBus& operator=(const SimpleBus&);
      SimpleBus(const SimpleBus&);

      void thread()
      {	 
	 tlm::tlm_generic_payload* trans;
	 tlm::tlm_phase            phase;
	 sc_core::sc_time          delay;
	 tlm::tlm_sync_enum        retval;
	 unsigned long int         destination = 0;
	 int                       source      = 0;
	 int                       size        = 0;
	 int                       flits       = 0;

	 //std::cout << "SimpleBus::thread() ready" << std::endl;

	 while(true)
	 {
	    if((trans = _rxPeq->get_next_transaction()) == 0)
	    {
	       //std::cout << "SimpleBus::thread : waiting for transaction" << std::endl;
	       wait(_rxPeq->get_event());
	       trans = _rxPeq->get_next_transaction();
	    }	    
	    
	    destination = trans->get_address();
	    size        = trans->get_data_length();
	    flits       = size / trans->get_streaming_width();
	    source      = _rxSourceQueue.front();
	    _rxSourceQueue.pop();

	    //std::cout << "SimpleBus::thread : got transaction from " << source 
	    //<< " to " << destination << std::endl;

	    phase = tlm::BEGIN_REQ;
	    delay = sc_core::SC_ZERO_TIME;

	    if(destination >= N_AGENTS)
	    {
	       std::ostringstream oss;
	       oss << "SimpleBus::thread : destination (NoC address) "
		   << destination << " should be less than "
		   << N_AGENTS << ".";
	       throw std::runtime_error(oss.str().c_str());
	    }
	    
	    retval = initSockets[destination]
	       ->nb_transport_fw(*trans, phase, delay);

	    if(retval == tlm::TLM_ACCEPTED || retval == tlm::TLM_UPDATED)
	    {
	       if(phase == tlm::BEGIN_REQ)
	       {
		  //std::cout << "SimpleBus::thread : waiting completion" << std::endl;
		  wait(_txCompleteEvent);
		  //std::cout << "SimpleBus::thread : completed" << std::endl;
	       }
	       else if(phase == tlm::END_REQ)
	       {
		  std::ostringstream oss;
		  oss << "SimpleBus::thread : END_REQ not supported";
		  throw std::runtime_error(oss.str().c_str());
	       }
	       else if(phase == tlm::BEGIN_RESP)
	       {
		  std::ostringstream oss;
		  oss << "SimpleBus::thread : BEGIN_RESP not supported";
		  throw std::runtime_error(oss.str().c_str());
	       }
	       else
	       {
		  std::ostringstream oss;
		  oss << "SimpleBus::thread : invalid PHASE";
		  throw std::runtime_error(oss.str().c_str());
	       }	       
	    }
	    else if(retval == tlm::TLM_COMPLETED)
	    {
	       //std::cout << "SimpleBus::thread : got TLM_COMPLETED" << std::endl;
	       if(delay != sc_core::SC_ZERO_TIME)
	       {
		  wait(delay);
	       }
	    }
	    else
	    {
	       std::ostringstream oss;
	       oss << "SimpleBus::thread : invalid SYNC_ENUM";
	       throw std::runtime_error(oss.str().c_str());
	    }	       

	    phase = tlm::END_REQ;
	    delay = sc_core::SC_ZERO_TIME;

	    retval = targetSockets[source]->nb_transport_bw(*trans, phase, delay);
	    
	    if(retval != tlm::TLM_COMPLETED)
	    {
	       std::ostringstream oss;
	       oss << "SimpleBus::thread : Not supporting responses";
	       throw std::runtime_error(oss.str().c_str());
	    }
	    	    
	 } // End of while(true)
      }

      tlm::tlm_sync_enum nb_transport_fw(int id,
					 tlm::tlm_generic_payload &trans,
					 tlm::tlm_phase           &phase,
					 sc_core::sc_time         &delay)
      {
	 // Only write command is supported
	 if(trans.get_command() != tlm::TLM_WRITE_COMMAND)
	 {
	    std::ostringstream oss;
	    oss << "SimpleBus::nb_tranport_fw " << id 
		<< ": only write command is supported";
	    throw std::runtime_error(oss.str().c_str());
	 }
	 
	 if(phase == tlm::BEGIN_REQ)
	 {
	    //std::cout << "SimpleBus::nb_fw BEGIN_REQ : " << id << std::endl;
	    _rxSourceQueue.push(id);
	    _rxPeq->notify(trans, delay);
	 }
	 else if(phase == tlm::END_RESP)
	 {
	    //std::cout << "SimpleBus::nb_fw END_RESP : " << id << std::endl;
	    trans.set_response_status( tlm::TLM_OK_RESPONSE );
	    return tlm::TLM_COMPLETED;
	 }
	 else
	 {
	    std::ostringstream oss;
	    oss << "SimpleBus::nb_tranport_fw " << id 
		<< ": got invalid PHASE";
	    throw std::runtime_error(oss.str().c_str());
	 }
	 trans.set_response_status( tlm::TLM_OK_RESPONSE );
	 return tlm::TLM_ACCEPTED;
      }
      
      tlm::tlm_sync_enum nb_transport_bw(int id,
					 tlm::tlm_generic_payload &trans,
					 tlm::tlm_phase           &phase,
					 sc_core::sc_time         &delay)
      {
	 if(phase == tlm::BEGIN_REQ || phase == tlm::END_RESP)
	 {
	    std::ostringstream oss;
	    oss << "nb_tranport_bw " << id << " got wrong phase";
	    throw std::runtime_error(oss.str().c_str());
	 }
	 
	 //std::cout << "SimpleBus::nb_bw CALLED : " << id << std::endl;

	 _txCompleteEvent.notify(delay);
	 
	 trans.set_response_status( tlm::TLM_OK_RESPONSE );
	 return tlm::TLM_COMPLETED;
      }

      // Pointer to configuration interface
      sctg::NocConfIf* _confIf;

      sc_core::sc_event _txCompleteEvent;
      tlm_utils::peq_with_get<tlm::tlm_generic_payload>* _rxPeq;
      std::queue<int>                                   _rxSourceQueue;



   };
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:

