/**
 *
 * @file packet_encoder_ctrl.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: packet_encoder_ctrl.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#ifndef ASEBT_SC_RTL_2_PACKET_ENCODER_CTRL_HH
#define ASEBT_SC_RTL_2_PACKET_ENCODER_CTRL_HH

#include "addr_lut.hh"
#include "pkt_counter.hh"

#include <systemc>

namespace asebt
{
namespace sc_rtl_2
{

   template<int wait_empty_fifo_g = 0,
	    int data_width_g      = 36,
	    int addr_width_g      = 32,
	    int tx_len_width_g    = 8,
	    int packet_length_g   = 0, 
	    int timeout_g         = 0,
	    int fill_packet_g     = 0,
	    int lut_en_g          = 0,
	    int net_type_g        = 0,
	    int len_flit_en_g     = 1,
	    int oaddr_flit_en_g   = 1,
	    int status_en_g       = 0>
   class packet_encoder_ctrl : public sc_core::sc_module
   {
   public:
      sc_core::sc_in_clk                              clk;
      sc_core::sc_in<bool>               rst_n;

      sc_core::sc_in<bool>               ip_av_in;
      sc_core::sc_in<sc_dt::sc_bv<data_width_g> >   ip_data_in;
      sc_core::sc_in<bool>               ip_we_in;
      sc_core::sc_in<sc_dt::sc_bv<tx_len_width_g> > ip_tx_len_in;

      sc_core::sc_in<bool>               fifo_av_in;
      sc_core::sc_in<sc_dt::sc_bv<data_width_g> >   fifo_data_in;
      sc_core::sc_in<bool>               fifo_full_in;
      sc_core::sc_in<bool>               fifo_empty_in;
      sc_core::sc_out<bool>              fifo_re_out;
   
      sc_core::sc_in<bool>               net_full_in;
      sc_core::sc_in<bool>               net_empty_in;

      sc_core::sc_out<bool>              ip_stall_out;
      sc_core::sc_out<bool>              net_av_out;
      sc_core::sc_out<sc_dt::sc_bv<data_width_g> >  net_data_out;
      sc_core::sc_out<bool>              net_we_out;

      SC_HAS_PROCESS(packet_encoder_ctrl);

      packet_encoder_ctrl(sc_core::sc_module_name name)
	 : sc_core::sc_module(name),
	   clk("clk"),
	   rst_n("rst_n"),
	   ip_av_in("ip_av_in"),
	   ip_data_in("ip_data_in"),
	   ip_we_in("ip_we_in"),
	   ip_tx_len_in("ip_tx_len_in"),
	   fifo_av_in("fifo_av_in"),
	   fifo_data_in("fifo_data_in"),
	   fifo_full_in("fifo_full_in"),
	   fifo_empty_in("fifo_empty_in"),
	   fifo_re_out("fifo_re_out"),
	   net_full_in("net_full_in"),
	   net_empty_in("net_empty_in"),
	   ip_stall_out("ip_stall_out"),
	   net_av_out("net_av_out"),
	   net_data_out("net_data_out"),
	   net_we_out("net_we_out"),
	   wr_state_r("wr_state_r"),
	   hdr_words_c(1 + len_flit_en_g + oaddr_flit_en_g),
	   payload_words_c(packet_length_g - hdr_words_c),
	   amount_r("amount_r"),
	   timeout_counter_r("timeout_counter_r"),
	   write_counter_r("write_counter_r"),
	   rd_state_r("rd_state_r"),
	   amount_rd_r("amount_rd_r"),
	   prev_addr_r("prev_addr_r"),
	   timeout_counter_rd_r("timeout_counter_rd_r"),
	   amount_to_wr_r("amount_to_wr_r"),
	   len_from_ip("len_from_ip"),
	   len_r("len_r"),
	   len_valid_r("len_valid_r"),
	   next_len_valid_r("next_len_valid_r"),
	   next_addr_r("next_addr_r"),
	   next_addr_valid_r("next_addr_valid_r"),
	   ack_wr_rd("ack_wr_rd"),
	   ack_tmp_r("ack_tmp_r"),
	   ack_from_wr_r("ack_from_wr_r"),
	   req_rd_wr("req_rd_wr"),
	   req_writing_r("req_writing_r"),
	   data_in_fifo_r("data_in_fifo_r"),
	   addr_lut_(0),
	   pkt_counter_(0),
	   addr_to_lut("addr_to_lut"),
	   addr_from_lut("addr_from_lut"),
	   in_addr_r("in_addr_r"),
	   ip_stall("ip_stall"),
	   len_to_status("len_to_status"),
	   len_width_c(8)
      {
	 if(lut_en_g == 1)
	 {
	    addr_lut_ = new addr_lut<addr_width_g,
	       data_width_g,
	       addr_width_g-1,
	       0,
	       net_type_g,
	       lut_en_g>("addr_lut");
	 
	    addr_lut_->addr_in(addr_to_lut);
	    addr_lut_->addr_out(addr_from_lut);      
	 }
	 else
	 {
	    addr_lut_ = 0;

	    SC_METHOD(addrlut);
	    sensitive << addr_to_lut;
	 }

	 pkt_counter_ = new pkt_counter<tx_len_width_g>("pkt_counter");
      
	 pkt_counter_->clk(clk);
	 pkt_counter_->rst_n(rst_n);
	 pkt_counter_->len_in(len_to_status);
	 pkt_counter_->new_tx_in(ip_av_in);
	 pkt_counter_->new_pkt_in(req_rd_wr);
	 pkt_counter_->idle_in(fifo_empty_in);
      
	 SC_METHOD(reqrdwr);
	 sensitive << req_writing_r;

	 SC_METHOD(ipstall);
	 sensitive << ip_stall;

	 SC_METHOD(iplen);
	 sensitive << ip_tx_len_in;

	 SC_METHOD(lenstat);
	 sensitive << amount_to_wr_r;

	 SC_METHOD(read_data);
	 sensitive << clk.pos() << rst_n;

	 SC_METHOD(full_tmp);
	 sensitive << rd_state_r << ip_av_in << amount_rd_r
		   << timeout_counter_rd_r << data_in_fifo_r
		   << len_valid_r << len_r << req_writing_r
		   << amount_to_wr_r;

	 SC_METHOD(sync);
	 sensitive << clk.pos() << rst_n;

	 SC_METHOD(async);
	 sensitive << fifo_data_in << fifo_av_in << fifo_empty_in
		   << net_full_in << write_counter_r << amount_r
		   << prev_addr_r << ack_tmp_r << wr_state_r
		   << addr_from_lut << in_addr_r;
      }

      virtual ~packet_encoder_ctrl()
      {
	 if(addr_lut_) {delete addr_lut_; addr_lut_ = 0;}
	 if(pkt_counter_) {delete pkt_counter_; pkt_counter_ = 0;}
      }

      void reqrdwr()
      {
	 req_rd_wr.write(req_writing_r.read());
      }

      void ipstall()
      {
	 ip_stall_out.write(ip_stall.read());
      }

      void iplen()
      {
	 len_from_ip.write(ip_tx_len_in.read().to_ulong());
      }

      void lenstat()
      {
	 len_to_status.write(amount_to_wr_r.read());
      }

      void addrlut()
      {
	 addr_from_lut.write(addr_to_lut.read());
      }

      void read_data()
      {
	 if(rst_n.read() == false)
	 {
	    amount_rd_r          = 0;
	    rd_state_r           = WAIT_TRANS;
	    prev_addr_r          = "0";
	    timeout_counter_rd_r = 0;
	    ack_from_wr_r        = false;
	    amount_to_wr_r       = 0;
	    len_valid_r          = false;
	    next_len_valid_r     = false;
	    req_writing_r        = false;
	    len_r                = 0;

	    next_addr_r          = "0";
	    next_addr_valid_r    = false;
	    data_in_fifo_r       = false;
	 }
	 else
	 {
	    if(ack_wr_rd.read() == true &&
	       rd_state_r.read() != WAIT_TRANS)
	    {
	       ack_from_wr_r = true;
	    }

	    switch(rd_state_r.read())
	    {
	       case WAIT_TRANS:
		  timeout_counter_rd_r = 0;

		  if(ip_we_in.read() == true &&
		     fifo_full_in.read() == false)
		  {
		     if(ip_av_in.read() == true)
		     {
			amount_rd_r    = 0;
			rd_state_r     = READ_IN;
			prev_addr_r    = ip_data_in.read();
			data_in_fifo_r = false;

			if(len_from_ip.read() != 0)
			{
			   if(len_from_ip.read() > payload_words_c)
			   {
			      amount_to_wr_r   = payload_words_c;
			      next_len_valid_r = true;
			      len_valid_r      = false;
			   }
			   else
			   {
			      amount_to_wr_r   = len_from_ip.read();
			      next_len_valid_r = false;
			      len_valid_r      = true;
			   }
			
			   len_r         = len_from_ip.read();
			   req_writing_r = true;
			}
			else
			{
			   len_valid_r = false;
			}
		     }
		     else
		     {
			amount_rd_r    = 1;
			rd_state_r     = READ_IN;
			prev_addr_r    = prev_addr_r.read();
			data_in_fifo_r = true;

			if(next_len_valid_r.read() == true)
			{
			   if(len_r.read() > payload_words_c)
			   {
			      amount_to_wr_r   = payload_words_c;
			      next_len_valid_r = true;
			      len_valid_r      = false;
			   }
			   else
			   {
			      amount_to_wr_r   = len_r.read();
			      next_len_valid_r = false;
			      len_valid_r      = true;
			   }
			
			   req_writing_r = true;
			}
		     }
		  }
		  else
		  {
		     amount_rd_r = 0;
		     rd_state_r  = WAIT_TRANS;
		     prev_addr_r = prev_addr_r.read();
		  }

		  if(next_addr_valid_r.read() == true)
		  {
		     prev_addr_r       = next_addr_r.read();
		     next_addr_valid_r = false;
		     next_addr_r       = "0";
		  }

		  break;

	       case READ_IN:
		  prev_addr_r = prev_addr_r.read();

		  if(ip_we_in.read() == true &&
		     fifo_full_in.read() == false &&
		     ip_stall.read() == false)
		  {
		     timeout_counter_rd_r = 0;
		  
		     if(ip_av_in.read() == true)
		     {
			amount_to_wr_r    = amount_rd_r.read();
			req_writing_r     = true;
			rd_state_r        = WAIT_ACK;
			next_addr_r       = ip_data_in.read();
			next_addr_valid_r = true;
		     }
		     else
		     {
			data_in_fifo_r = true;
		     
			if(amount_rd_r.read() == payload_words_c ||
			   (len_valid_r.read() == true &&
			    amount_rd_r.read() == len_r.read()) ||
			   (req_writing_r.read() == true &&
			    amount_rd_r.read() == amount_to_wr_r.read()))
			{
			   if(req_writing_r.read() == false)
			   {
			      amount_to_wr_r = amount_rd_r.read();
			   }

			   req_writing_r = true;
			   rd_state_r    = WAIT_ACK;
			}
			else
			{
			   amount_rd_r = amount_rd_r.read() + 1;
			   rd_state_r  = READ_IN;
			}
		     }
		  }
		  else
		  {
		     if(data_in_fifo_r.read() == true)
		     {
			if(amount_rd_r.read() == payload_words_c ||
			   (len_valid_r.read() == true &&
			    amount_rd_r.read() == len_r.read()) ||
			   req_writing_r.read() == true &&
			   amount_rd_r.read() == amount_to_wr_r.read())
			{
			   if(req_writing_r.read() == false)
			   {
			      amount_to_wr_r = amount_rd_r.read();
			   }
			
			   req_writing_r = true;
			   rd_state_r    = WAIT_ACK;

			}
			else if(timeout_counter_rd_r.read() == timeout_g)
			{
			   timeout_counter_rd_r = timeout_counter_rd_r.read();
			   req_writing_r        = true;
			   rd_state_r           = WAIT_ACK;
			   amount_to_wr_r       = amount_rd_r.read();
			}
			else
			{
			   if(net_full_in.read() == false)
			   {
			      timeout_counter_rd_r = 
				 timeout_counter_rd_r.read() + 1;
			   }
			   else
			   {
			      timeout_counter_rd_r = timeout_counter_rd_r.read();
			   }

			   rd_state_r  = READ_IN;
			   amount_rd_r = amount_rd_r.read();
			}
		     }
		  }
		  break;

	       default: // WAIT_ACK
		  if(ack_from_wr_r.read() == true)
		  {
		     rd_state_r    = WAIT_TRANS;
		     ack_from_wr_r = false;
		     req_writing_r = false;

		     if(next_len_valid_r.read() == true)
		     {
			len_r = len_r.read() - payload_words_c;
		     }
		  }
		  else
		  {
		     rd_state_r = WAIT_ACK;
		  }
		  break;
	    }	 
	 }
      }

      void full_tmp()
      {
	 ip_stall = false;
	 if(rd_state_r.read() == WAIT_ACK)
	 {
	    ip_stall = true;
	 }
	 else if(rd_state_r.read() == READ_IN)
	 {
	    if(amount_rd_r.read() == payload_words_c ||
	       (timeout_counter_r.read() == timeout_g &&
		data_in_fifo_r.read() == true) ||
	       ip_av_in.read() == true ||
	       (len_valid_r.read() == true &&
		amount_rd_r.read() == len_r.read()) ||
	       (req_writing_r.read() == true &&
		amount_rd_r.read() == amount_to_wr_r.read()))
	    {
	       ip_stall = true;
	    }
	 }
      }

      void sync()
      {
	 if(rst_n.read() == false)
	 {
	    wr_state_r        = START;
	    timeout_counter_r = 0;
	    write_counter_r   = 0;
	    amount_r          = 1;
	    in_addr_r         = "0";
	    ack_tmp_r         = false;
	 }
	 else
	 {
	    ack_tmp_r = false;

	    switch(wr_state_r.read())
	    {
	       case START:
		  timeout_counter_r = 0;
		  write_counter_r   = 0;
		  amount_r          = 1;
		  wr_state_r        = WAIT_REQ;
		  break;

	       case WAIT_REQ:
		  if(req_rd_wr.read() == true &&
		     ack_tmp_r.read() == false)
		  {
		     amount_r = amount_to_wr_r.read() + oaddr_flit_en_g;
		  
		     if(net_empty_in.read() == false &&
			wait_empty_fifo_g == 1)
		     {
			wr_state_r = WAIT_EMPTY;
		     }
		     else
		     {
			wr_state_r = WRITE_OUT_ADDR;
		     }
		  }
		  else
		  {
		     amount_r   = 1;
		     wr_state_r = WAIT_REQ;
		  }
		  break;

	       case WAIT_EMPTY:
		  if(net_empty_in.read() == true)
		  {
		     wr_state_r = WRITE_OUT_ADDR;
		  }
		  else
		  {
		     wr_state_r = WAIT_EMPTY;
		  }
		  break;

	       case WRITE_OUT_ADDR:
		  timeout_counter_r = 0;
		  write_counter_r   = 0;
		  amount_r          = amount_r.read();

		  if(net_full_in.read() == false)
		  {
		     if(len_flit_en_g == 1)
		     {
			wr_state_r = WRITE_OUT_AMOUNT;
		     }
		     else
		     {
			if(oaddr_flit_en_g == 1)
			{
			   wr_state_r = WRITE_OUT_IN_ADDR;
			}
			else
			{
			   wr_state_r = WRITE_OUT_DATA;
			}
		     }
		  }
		  else
		  {
		     wr_state_r = WRITE_OUT_ADDR;
		  }

		  if(fifo_av_in.read() == true)
		  {
		     in_addr_r = fifo_data_in.read();
		  }
		  else
		  {
		     in_addr_r = prev_addr_r.read();
		  }
		  break;

	       case WRITE_OUT_AMOUNT:
		  timeout_counter_r = 0;
		  write_counter_r   = 0;
		  amount_r          = amount_r.read();
	       
		  if(net_full_in.read() == false)
		  {
		     if(oaddr_flit_en_g == 1)
		     {
			wr_state_r = WRITE_OUT_IN_ADDR;
		     }
		     else
		     {
			wr_state_r = WRITE_OUT_DATA;
		     }
		  }
		  else
		  {
		     wr_state_r = WRITE_OUT_AMOUNT;
		  }
		  break;

	       case WRITE_OUT_IN_ADDR:
		  timeout_counter_r = 0;
		  write_counter_r   = 0;
		  amount_r          = amount_r.read();
		  if(net_full_in.read() == false)
		  {
		     wr_state_r = WRITE_OUT_DATA;
		  }
		  else
		  {
		     wr_state_r = WRITE_OUT_IN_ADDR;
		  }
		  break;

	       case WRITE_OUT_DATA:
		  timeout_counter_r = 0;
		  ack_tmp_r         = true;

		  if(net_full_in.read() == false &&
		     (fifo_empty_in.read() == false ||
		      (write_counter_r.read() > 
		       (amount_r.read() - 1 - oaddr_flit_en_g) &&
		       fill_packet_g == 1)))
		  {
		     if(write_counter_r.read() == packet_length_g)
		     {
			write_counter_r = write_counter_r.read();
		     }
		     else
		     {
			write_counter_r = write_counter_r.read() + 1;
		     }
		  }
		  else
		  {
		     write_counter_r = write_counter_r.read();
		  }

		  amount_r = amount_r.read();

		  if(net_full_in.read() == false &&
		     ((write_counter_r.read() == 
		       (amount_r.read() - 1 - oaddr_flit_en_g) &&
		       fill_packet_g == 0) ||
		      (write_counter_r.read() == payload_words_c - 1 &&
		       fill_packet_g == 1)))
		  {
		     wr_state_r = WAIT_REQ;
		  }
		  else
		  {
		     wr_state_r = WRITE_OUT_DATA;
		  }	       
		  break;
	    }	 
	 }
      }


      void async()
      {
	 addr_to_lut = "0";

	 switch(wr_state_r.read())
	 {
	    case START:
	       fifo_re_out  = false;
	       net_data_out = "0";
	       net_av_out   = false;
	       net_we_out   = false;
	       ack_wr_rd    = true;
	       break;

	    case WAIT_REQ:
	    case WAIT_EMPTY:
	       net_we_out   = false;
	       fifo_re_out  = false;
	       net_data_out = "0";
	       net_av_out   = false;
	       ack_wr_rd    = false;
	       break;
	    
	    case WRITE_OUT_ADDR:
	       ack_wr_rd    = false;
	       net_av_out   = true;

	       if(net_full_in.read() == false)
	       {
		  net_we_out = true;
	       }
	       else
	       {
		  net_we_out = false;
	       }

	       if(fifo_av_in.read() == true)
	       {
		  fifo_re_out = true;
		  addr_to_lut = fifo_data_in.read().range(addr_width_g-1, 0);
	       }
	       else
	       {
		  fifo_re_out = false;
		  addr_to_lut = prev_addr_r.read().range(addr_width_g-1, 0);
	       }

	       if(len_flit_en_g == 1)
	       {
		  net_data_out = addr_from_lut.read();
	       }
	       else
	       {
		  sc_dt::sc_bv<data_width_g> tmp;
		  if(fill_packet_g == 0)
		  {		  
		     //tmp = net_data_out.get_new_value();
		     tmp.range(data_width_g-1, data_width_g - len_width_c)
			= amount_r.read();
		     //net_data_out = tmp.get_new_value();
		  }
		  else
		  {		  
		     //tmp = net_data_out.get_new_value();
		     tmp.range(data_width_g-1, data_width_g - len_width_c)
			= payload_words_c;
		     //net_data_out = tmp.get_new_value();
		  }
		  //tmp = net_data_out.get_new_value();
		  tmp.range(data_width_g - len_width_c - 1, 0)
		     = addr_from_lut.read().range(data_width_g-len_width_c - 1, 0);
		  net_data_out = tmp;
	       }
	       break;

	    case WRITE_OUT_AMOUNT:
	       ack_wr_rd   = false;
	       fifo_re_out = false;

	       if(fill_packet_g == 0)
	       {
		  net_data_out = amount_r.read();
	       }
	       else
	       {
		  net_data_out = payload_words_c;
	       }

	       net_av_out = false;

	       if(net_full_in.read() == false)
	       {
		  net_we_out = true;
	       }
	       else
	       {
		  net_we_out = false;
	       }	    
	       break;

	    case WRITE_OUT_IN_ADDR:
	       fifo_re_out  = false;
	       net_data_out = in_addr_r.read();
	       net_av_out   = false;

	       if(net_full_in.read() == false)
	       {
		  net_we_out = true;	       
	       }
	       else
	       {
		  net_we_out = false;
	       }

	       ack_wr_rd = false;
	       break;

	    case WRITE_OUT_DATA:
	       net_av_out = false;
	    
	       if(ack_tmp_r.read() == false)
	       {
		  ack_wr_rd = true;
	       }
	       else
	       {
		  ack_wr_rd = false;
	       }

	       if(net_full_in.read() == false &&
		  (fifo_empty_in.read() == false ||
		   (write_counter_r.read() > amount_r.read() - 1 - oaddr_flit_en_g
		    && fill_packet_g == 1)))
	       {
		  net_we_out = true;
	       }
	       else
	       {
		  net_we_out = false;
	       }

	       if(write_counter_r.read() < amount_r.read() - oaddr_flit_en_g &&
		  net_full_in.read() == false)
	       {
		  if(fifo_empty_in.read() == false)
		  {
		     net_data_out = fifo_data_in.read();
		     fifo_re_out  = true;
		  }
		  else
		  {
		     net_data_out = fifo_data_in.read();
		     fifo_re_out  = false;
		  }
	       }
	       else
	       {
		  fifo_re_out  = false;
		  net_data_out = "0";
	       }
	       break;
	 }
      }
   
   private:
   
      enum wr_state_type {START, WAIT_REQ, WAIT_EMPTY, WRITE_OUT_ADDR,
			  WRITE_OUT_IN_ADDR, WRITE_OUT_AMOUNT,
			  WRITE_OUT_DATA};

      sc_core::sc_signal<wr_state_type> wr_state_r;
   
      const int hdr_words_c;
      const int payload_words_c;

      sc_core::sc_signal<int> amount_r;
      sc_core::sc_signal<int> timeout_counter_r;
      sc_core::sc_signal<int> write_counter_r;

      enum rd_state_type {WAIT_TRANS, READ_IN, WAIT_ACK};
   
      sc_core::sc_signal<rd_state_type> rd_state_r;
      sc_core::sc_signal<int>           amount_rd_r;
      sc_core::sc_signal<sc_dt::sc_bv<data_width_g> > prev_addr_r;

      sc_core::sc_signal<int> timeout_counter_rd_r;
      sc_core::sc_signal<int> amount_to_wr_r;
      sc_core::sc_signal<int> len_from_ip;
      sc_core::sc_signal<int> len_r;

      sc_core::sc_signal<bool> len_valid_r;
      sc_core::sc_signal<bool> next_len_valid_r;

      sc_core::sc_signal<sc_dt::sc_bv<data_width_g> > next_addr_r;
      sc_core::sc_signal<bool>             next_addr_valid_r;

      sc_core::sc_signal<bool> ack_wr_rd;
      sc_core::sc_signal<bool> ack_tmp_r;
      sc_core::sc_signal<bool> ack_from_wr_r;
      sc_core::sc_signal<bool> req_rd_wr;
      sc_core::sc_signal<bool> req_writing_r;
      sc_core::sc_signal<bool> data_in_fifo_r;

      addr_lut<addr_width_g,
	       data_width_g,
	       addr_width_g-1,
	       0,
	       net_type_g,
	       lut_en_g>*          addr_lut_;
      pkt_counter<tx_len_width_g>* pkt_counter_;
   
      sc_core::sc_signal<sc_dt::sc_bv<addr_width_g> > addr_to_lut;
      sc_core::sc_signal<sc_dt::sc_bv<data_width_g> > addr_from_lut;

      sc_core::sc_signal<sc_dt::sc_bv<data_width_g> > in_addr_r;
      sc_core::sc_signal<bool> ip_stall;

      sc_core::sc_signal<sc_dt::sc_bv<tx_len_width_g> > len_to_status;

      const int len_width_c;

   };

}
}

#endif

// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
