/**
 *
 * @file tlm_mesh_router.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: tlm_mesh_router.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#ifndef ASEBT_TLM_MESH_ROUTER_HH
#define ASEBT_TLM_MESH_ROUTER_HH

#include "tlm.h"
#ifdef MTI_SYSTEMC
#include "simple_initiator_socket.h"
#include "simple_target_socket.h"
#include "peq_with_get.h"
#else
#include "tlm_utils/simple_initiator_socket.h"
#include "tlm_utils/simple_target_socket.h"
#include "tlm_utils/peq_with_get.h"
#endif

#include <systemc>
#include <stdexcept>
#include <iostream>

namespace asebt
{
   namespace mesh_2d_sc_tlm_1
   {

      template<unsigned int data_width_g = 32>
      class TlmMeshRouter : public sc_core::sc_module
      {
      public:
      
	 SC_HAS_PROCESS(TlmMeshRouter);
      
	 // Sockets: N, W, S, E, IP
	 tlm_utils::simple_initiator_socket_tagged<TlmMeshRouter,
						   data_width_g>* initSockets[5];
	 tlm_utils::simple_target_socket_tagged<TlmMeshRouter,
						data_width_g>* targetSockets[5];

	 //* Constructor
	 TlmMeshRouter(sc_core::sc_module_name name,
		       unsigned int row,
		       unsigned int col,
		       unsigned int n_rows,
		       unsigned int n_cols)
	    : sc_module(name),
	      _row(row),
	      _col(col),
	      _cycleTime(20, sc_core::SC_NS)
	 {
	    for(unsigned int i = 0; i < 5; ++i)
	    {
	       for(unsigned int j = 0; j < 5; ++j)
	       {
		  std::ostringstream oss;
		  oss << "tlm_router_peq_" << i << "_" << j;
		  _peq[i][j] = 
		     new tlm_utils::peq_with_get<tlm::tlm_generic_payload>
		     (oss.str().c_str());
	       }
	    }

	    initSockets[4] = new tlm_utils::simple_initiator_socket_tagged
	       <TlmMeshRouter, data_width_g>;
	    targetSockets[4] = new tlm_utils::simple_target_socket_tagged
	       <TlmMeshRouter, data_width_g>;
	    if(row != 0)
	    {
	       initSockets[0] = new tlm_utils::simple_initiator_socket_tagged
		  <TlmMeshRouter, data_width_g>;
	       targetSockets[0] = new tlm_utils::simple_target_socket_tagged
		  <TlmMeshRouter, data_width_g>;
	    }
	    else
	    {
	       initSockets[0] = 0;
	       targetSockets[0] = 0;
	    }
	    
	    if(row != (n_rows - 1))
	    {
	       initSockets[2] = new tlm_utils::simple_initiator_socket_tagged
		  <TlmMeshRouter, data_width_g>;
	       targetSockets[2] = new tlm_utils::simple_target_socket_tagged
		  <TlmMeshRouter, data_width_g>;
	    }
	    else
	    {
	       initSockets[2] = 0;
	       targetSockets[2] = 0;
	    }
	    
	    if(col != 0)
	    {
	       initSockets[1] = new tlm_utils::simple_initiator_socket_tagged
		  <TlmMeshRouter, data_width_g>;
	       targetSockets[1] = new tlm_utils::simple_target_socket_tagged
		  <TlmMeshRouter, data_width_g>;
	    }
	    else
	    {
	       initSockets[1] = 0;
	       targetSockets[1] = 0;
	    }

	    if(col != (n_cols - 1))
	    {
	       initSockets[3] = new tlm_utils::simple_initiator_socket_tagged
		  <TlmMeshRouter, data_width_g>;
	       targetSockets[3] = new tlm_utils::simple_target_socket_tagged
		  <TlmMeshRouter, data_width_g>;
	    }
	    else
	    {
	       initSockets[3] = 0;
	       targetSockets[3] = 0;
	    }

	    for(unsigned int i = 0; i < 5; ++i)
	    {
	       if(initSockets[i])
	       {
		  initSockets[i]->register_nb_transport_bw
		     (this, &TlmMeshRouter::nb_transport_bw, i);
		  targetSockets[i]->register_nb_transport_fw
		     (this, &TlmMeshRouter::nb_transport_fw, i);
		  sc_spawn(sc_bind(&TlmMeshRouter::thread, this, i));
	       }
	    }
	 }
      
	 //* Destructor
	 ~TlmMeshRouter()
	 {
	    for(unsigned int i = 0; i < 5; ++i)
	    {
	       if(initSockets[i]) {delete initSockets[i]; initSockets[i] = 0;}
	       if(targetSockets[i]) {delete targetSockets[i]; targetSockets[i] = 0;}

	       for(unsigned int j = 0; j < 5; ++j)
	       {
		  delete _peq[i][j]; _peq[i][j] = 0;
	       }
	    }
	 }
      

	 void thread(unsigned int dir)
	 {
	    tlm::tlm_generic_payload* trans = 0;
	    tlm::tlm_phase            phase;
	    sc_core::sc_time          delay;
	    tlm::tlm_sync_enum        retval;
	    unsigned int              source = 0;

	    // std::ostringstream oss;
	    // oss << "at_" << _row << "_" << _col << "_"
	    // 	<< dir << "_router.txt";
	    // std::ofstream ofs(oss.str().c_str());

	    while(true)
	    {
	       // Check for pending transactions
	       for(unsigned int i = 0; i < 5; ++i)
	       {
		  if((trans = _peq[dir][i]->get_next_transaction()) != 0)
		  { source = i; break; }
	       }

	       if(trans == 0)
	       {
		  // Wait for transactions
		  wait(_peq[dir][0]->get_event() |
		       _peq[dir][1]->get_event() |
		       _peq[dir][2]->get_event() |
		       _peq[dir][3]->get_event() |
		       _peq[dir][4]->get_event());
		  continue; // jump back to beginning of while(true)
	       }

	       // ofs << "Router " << _row << "_" << _col << "_" 
	       // 	   << dir << " begins send  "
	       // 	   << sc_core::sc_time_stamp().value() << std::endl;

	       phase = tlm::END_REQ;
	       if(source == 4)
	       {
		  delay = sc_core::SC_ZERO_TIME;
	       }
	       else
	       {
		  delay = _cycleTime * 
		     ((trans->get_data_length() / trans->get_streaming_width())
		      + 3);
	       }

	       retval = (*targetSockets[source])->nb_transport_bw(*trans, 
							       phase, delay);
	    
	       if(retval != tlm::TLM_COMPLETED)
	       {
		  std::ostringstream oss;
		  oss << "TlmMeshRouter::thread : Not supporting responses";
		  throw std::runtime_error(oss.str().c_str());
	       }
	    
	       // Forward transaction	       

	       phase = tlm::BEGIN_REQ;
	       delay = _cycleTime * 3;
	    
	       retval = (*initSockets[dir])->nb_transport_fw(*trans, phase, delay);
	    
	       if(retval == tlm::TLM_ACCEPTED || retval == tlm::TLM_UPDATED)
	       {
		  if(phase == tlm::BEGIN_REQ)
		  {		  
		     wait(_txCompleteEvent[dir]);		
		  }
		  else if(phase == tlm::END_REQ)
		  {
		     std::ostringstream oss;
		     oss << "TlmMeshRouter::thread : END_REQ not supported";
		     throw std::runtime_error(oss.str().c_str());
		  }
		  else if(phase == tlm::BEGIN_RESP)
		  {
		     std::ostringstream oss;
		     oss << "TlmMeshRouter::thread : BEGIN_RESP not supported";
		     throw std::runtime_error(oss.str().c_str());
		  }
		  else
		  {
		     std::ostringstream oss;
		     oss << "TlmMeshRouter::thread : invalid PHASE";
		     throw std::runtime_error(oss.str().c_str());
		  }	       
	       }
	       else if(retval == tlm::TLM_COMPLETED)
	       {
		  if(delay != sc_core::SC_ZERO_TIME)
		  {
		     wait(delay);
		  }
	       }
	       else
	       {
		  std::ostringstream oss;
		  oss << "TlmMeshRouter::thread : invalid SYNC_ENUM";
		  throw std::runtime_error(oss.str().c_str());
	       }	       
	    	    
	       trans->release();

	       //wait(_cycleTime * 1);

	    } // end of while(true)
	 }      

      private:

	 tlm::tlm_sync_enum nb_transport_fw(int id,
					    tlm::tlm_generic_payload &trans,
					    tlm::tlm_phase           &phase,
					    sc_core::sc_time         &delay)
	 {
	    // Only write command is supported
	    if(trans.get_command() != tlm::TLM_WRITE_COMMAND)
	    {
	       std::ostringstream oss;
	       oss << "TlmMeshRouter::nb_tranport_fw " << id 
		   << ": only write command is supported";
	       throw std::runtime_error(oss.str().c_str());
	    }
	 
	    if(phase == tlm::BEGIN_REQ)
	    {
	       trans.acquire();
	       unsigned int destination = parseDest(trans.get_address());
	       _peq[destination][id]->notify(trans, delay);
	       //_receiveEvent[destination].notify(delay);
	    }
	    else if(phase == tlm::END_RESP)
	    {
	       trans.set_response_status(tlm::TLM_OK_RESPONSE);
	       return tlm::TLM_COMPLETED;
	    }
	    else
	    {
	       std::ostringstream oss;
	       oss << "TlmMeshRouter::nb_tranport_fw " << id 
		   << ": got invalid PHASE";
	       throw std::runtime_error(oss.str().c_str());
	    }
	    trans.set_response_status( tlm::TLM_OK_RESPONSE );
	    return tlm::TLM_ACCEPTED;
	 }


	 tlm::tlm_sync_enum nb_transport_bw(int id,
					    tlm::tlm_generic_payload &trans,
					    tlm::tlm_phase           &phase,
					    sc_core::sc_time         &delay)
	 {
	    if(phase == tlm::BEGIN_REQ || phase == tlm::END_RESP)
	    {
	       std::ostringstream oss;
	       oss << "TlmMeshRouter::nb_tranport_bw " << id << " got wrong phase";
	       throw std::runtime_error(oss.str().c_str());
	    }

	    _txCompleteEvent[id].notify(delay);
	 
	    trans.set_response_status( tlm::TLM_OK_RESPONSE );
	    return tlm::TLM_COMPLETED;
	 }


	 unsigned int parseDest(unsigned long int addr)
	 {
	    unsigned long int rowPart = (addr & 0xFFFF0000) >> 16;
	    unsigned long int colPart = addr & 0x0000FFFF;

	    if(_row == rowPart && _col == colPart)
	    {
	       return 4; // IP
	    }
	    else if(_row < rowPart)
	    {
	       return 2; // S
	    }
	    else if(_row > rowPart)
	    {
	       return 0; // N
	    }
	    else if(_col < colPart)
	    {
	       return 3; // E
	    }
	    else
	    {
	       return 1; // W
	    }
	 }

	 unsigned int _row;
	 unsigned int _col;

	 sc_core::sc_event _txCompleteEvent[5];
	 tlm_utils::peq_with_get<tlm::tlm_generic_payload>* _peq[5][5];
	 sc_core::sc_time _cycleTime;

      };

   }
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:

