/**
 * File:    ServiceGraphPanel.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 22.2.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui.service;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ScrollPaneConstants;
import javax.swing.Timer;

import org.jgraph.JGraph;
import org.jgraph.event.GraphModelEvent;
import org.jgraph.event.GraphModelListener;
import org.jgraph.event.GraphModelEvent.GraphModelChange;
import org.jgraph.graph.DefaultCellViewFactory;
import org.jgraph.graph.DefaultEdge;
import org.jgraph.graph.DefaultGraphCell;
import org.jgraph.graph.DefaultGraphModel;
import org.jgraph.graph.DefaultPort;
import org.jgraph.graph.GraphConstants;
import org.jgraph.graph.GraphLayoutCache;

import fi.cpu.Settings;
import fi.cpu.data.Configuration;
import fi.cpu.data.Connection;
import fi.cpu.data.Model;
import fi.cpu.data.ModelNode;
import fi.cpu.data.Service;
import fi.cpu.data.ServiceActivationInfo;
import fi.cpu.event.ModelEvent;
import fi.cpu.event.ModelListener;
import fi.cpu.ui.SettingsDialog;


/**
 * Class for implementing the panel that shows services, connetions
 * between them and their activity.
 */
public class ServiceGraphPanel extends JPanel {
	private static final Color SERVICE_ACTIVE;
	private static final Color SERVICE_PASSIVE;
	private static final Color MESSAGE_ACTIVE_COLOR;
	private static final Color MESSAGE_PASSIVE_COLOR;
	private static final float MESSAGE_ACTIVE_WIDTH;
	private static final float MESSAGE_PASSIVE_WIDTH;
	private static final Color TEXT_COLOR;
	private static final Color BORDER_COLOR = Color.BLACK;
	private static final int TEXT_FONT;	
	private static final int SERVICE_WIDTH;
	private static final int SERVICE_HEIGHT;
	private static final double DEFAULT_ACTIVE_TIME;
	private static final int BORDER_WIDTH = 2;
	private static final int LINE_STYLE = GraphConstants.STYLE_SPLINE;
	private static final int ARROW_STYLE = GraphConstants.ARROW_TECHNICAL;
	private static final boolean SERVICE_SHAPE_SQUARE;
	
	protected Configuration config;
	protected JGraph graph;
	protected DefaultGraphModel graphModel;
	protected GraphLayoutCache graphView;
	protected HashMap<Integer, DefaultGraphCell> graphCells;
    protected HashMap<Integer, DefaultEdge> edges;
    protected HashMap<Integer, Timer> serviceTimers;
    protected HashMap<Integer, Timer> connectionTimers;
    protected HashMap<Integer, Integer> serviceValues;
    protected HashMap<Integer, Integer> connectionValues;
	protected HashMap<DefaultGraphCell, Service> services;
	protected HashMap<DefaultEdge, Connection> connections;
	protected double activationTime;
	
	
	static {
		int red = 0;
		int green = 0;
		int blue = 0;
		
		// Get configuration values
		red = Integer.parseInt( Settings.getAttributeValue("SERVICE_ACTIVE_COLOR_RED") );
		green = Integer.parseInt( Settings.getAttributeValue("SERVICE_ACTIVE_COLOR_GREEN") );
		blue = Integer.parseInt( Settings.getAttributeValue("SERVICE_ACTIVE_COLOR_BLUE") );
		SERVICE_ACTIVE = new Color(red, green, blue);
		
		red = Integer.parseInt( Settings.getAttributeValue("SERVICE_PASSIVE_COLOR_RED") );
		green = Integer.parseInt( Settings.getAttributeValue("SERVICE_PASSIVE_COLOR_GREEN") );
		blue = Integer.parseInt( Settings.getAttributeValue("SERVICE_PASSIVE_COLOR_BLUE") );
		SERVICE_PASSIVE = new Color(red, green, blue);
		
		red = Integer.parseInt( Settings.getAttributeValue("SERVICE_ACTIVE_MSG_COLOR_RED") );
		green = Integer.parseInt( Settings.getAttributeValue("SERVICE_ACTIVE_MSG_COLOR_GREEN") );
		blue = Integer.parseInt( Settings.getAttributeValue("SERVICE_ACTIVE_MSG_COLOR_BLUE") );
		MESSAGE_ACTIVE_COLOR = new Color(red, green, blue);
		
		red = Integer.parseInt( Settings.getAttributeValue("SERVICE_PASSIVE_MSG_COLOR_RED") );
		green = Integer.parseInt( Settings.getAttributeValue("SERVICE_PASSIVE_MSG_COLOR_GREEN") );
		blue = Integer.parseInt( Settings.getAttributeValue("SERVICE_PASSIVE_MSG_COLOR_BLUE") );
		MESSAGE_PASSIVE_COLOR = new Color(red, green, blue);

		red = Integer.parseInt( Settings.getAttributeValue("SERVICE_TEXT_COLOR_RED") );
		green = Integer.parseInt( Settings.getAttributeValue("SERVICE_TEXT_COLOR_GREEN") );
		blue = Integer.parseInt( Settings.getAttributeValue("SERVICE_TEXT_COLOR_BLUE") );
		TEXT_COLOR = new Color(red, green, blue);

		TEXT_FONT = Integer.parseInt( Settings.getAttributeValue("SERVICE_TEXT_FONT"));
		SERVICE_WIDTH = Integer.parseInt( Settings.getAttributeValue("SERVICE_WIDTH"));
		SERVICE_HEIGHT = Integer.parseInt( Settings.getAttributeValue("SERVICE_HEIGHT"));
		MESSAGE_ACTIVE_WIDTH = Float.parseFloat( Settings.getAttributeValue("SERVICE_ACTIVE_MSG_WIDTH"));
		MESSAGE_PASSIVE_WIDTH = Float.parseFloat( Settings.getAttributeValue("SERVICE_PASSIVE_MSG_WIDTH"));
		DEFAULT_ACTIVE_TIME = Double.parseDouble( Settings.getAttributeValue("SERVICE_ACTIVE_TIME_DEFAULT"));
		SERVICE_SHAPE_SQUARE = Boolean.parseBoolean( Settings.getAttributeValue("SERVICE_SHAPE_SQUARE"));
	}
	
	
	/**
	 * Creates a new ServiceGraphPanel.
	 */
	public ServiceGraphPanel(Configuration config) {
		super();
		this.config = config;

		config.getServiceModel().addModelListener(new MyModelListener());
        graphModel = new DefaultGraphModel();

        graphCells = new HashMap<Integer, DefaultGraphCell>();
		edges = new HashMap<Integer, DefaultEdge>();
        services = new HashMap<DefaultGraphCell, Service>();
		connections = new HashMap<DefaultEdge, Connection>();
        serviceTimers = new HashMap<Integer, Timer>();
        connectionTimers = new HashMap<Integer, Timer>();
        serviceValues = new HashMap<Integer, Integer>();
        connectionValues = new HashMap<Integer, Integer>();
		
		activationTime = DEFAULT_ACTIVE_TIME;
		
		setLayout(new BorderLayout());
		// use default factory if squares are enough
		if (SERVICE_SHAPE_SQUARE) {
			graphView = new GraphLayoutCache(graphModel, new DefaultCellViewFactory());			
		} else {
			graphView = new GraphLayoutCache(graphModel, new ServiceCellViewFactory());
		}
		
		graphModel.addGraphModelListener(new MyGraphModelListener());
		graph = new JGraph(graphModel, graphView);
		graph.setBackground(getBackground());
		
		JScrollPane sp = new JScrollPane(graph,
				ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
				ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		add(sp, BorderLayout.CENTER);	
		
		// Set graph attributes
		graph.setEditable(false);
		graph.setConnectable(false);
		graph.setDisconnectable(false);
//	    graph.setMoveable(false);
//		graph.setSelectionEnabled(false);
		graph.setSizeable(false);
//		graph.setPortsVisible(true);
		graph.setAntiAliased(true);
		
		initPanel();
	}

	
	/**
	 * Initializes the panel.
	 */
	public void initPanel() {
		graphView.remove(graphCells.values().toArray());
		graphView.remove(edges.values().toArray());
		graphCells.clear();
		services.clear();
		edges.clear();
		connections.clear();
        clearActives();
		
		/*** start ***/
		
		layoutServiceGraph();
		
		/***  end  ***/
        
		Model sModel = config.getServiceModel();
		Iterator<ModelNode> iter = sModel.getChildren(sModel.getModelRoot()).iterator();
		
		while (iter.hasNext()) {
			ModelNode node = iter.next();
			
			if (node instanceof Service) {
				Service s = (Service) node;
				DefaultGraphCell serviceCell = createServiceCell(s);
				graphCells.put(new Integer(s.getId()), serviceCell);
				services.put(serviceCell, s);
				
			} else if (node instanceof Connection) {
				Connection c = (Connection) node;
			
				int src = c.getSrcService();
				int dest = c.getDestService();
				DefaultGraphCell srcCell = graphCells.get(new Integer(src));
				DefaultGraphCell destCell = graphCells.get(new Integer(dest));
				
				DefaultEdge edge = createConnectionEdge(c, srcCell, destCell);
				
				if (edge != null) {
					edges.put(new Integer(c.getId()), edge);
					connections.put(edge, c);
				}
			}
		}
		
		graphView.insert(graphCells.values().toArray());
		graphView.insert(edges.values().toArray());

		revalidate();
		repaint();
	}
	
	
	/**
	 * Creates a new service cell.
	 */
	protected DefaultGraphCell createServiceCell(Service s) {
		if (s == null) {
			return null;
		}
		DefaultGraphCell serviceCell = new DefaultGraphCell(s.getName());

		// set cell attributes
		GraphConstants.setForeground(serviceCell.getAttributes(), TEXT_COLOR);
		GraphConstants.setBackground(serviceCell.getAttributes(), SERVICE_PASSIVE);
		GraphConstants.setOpaque(serviceCell.getAttributes(), true);
		GraphConstants.setBorderColor(serviceCell.getAttributes(), BORDER_COLOR);
		GraphConstants.setBorder(serviceCell.getAttributes(),
				BorderFactory.createLineBorder(BORDER_COLOR, BORDER_WIDTH));

		Font f = GraphConstants.getFont(serviceCell.getAttributes());
		f = f.deriveFont((float)TEXT_FONT);
		GraphConstants.setFont(serviceCell.getAttributes(), f);

		int x = (int)s.getLocation().getX();
		int y = (int)s.getLocation().getY();
		
		GraphConstants.setBounds(serviceCell.getAttributes(), new
				Rectangle2D.Double(x, y, SERVICE_WIDTH, SERVICE_HEIGHT));

		return serviceCell;
	}
	
	
	/**
	 * Creates a new connection edge between cells.
	 */
	protected DefaultEdge createConnectionEdge(
			Connection c, DefaultGraphCell src, DefaultGraphCell dest) {
		if (c == null || src == null || dest == null) {
			// connection can't be made
			return null;
		}
		
		DefaultPort srcPort = new DefaultPort();
		DefaultPort destPort = new DefaultPort();
		src.add(srcPort);
		dest.add(destPort);
		
		DefaultEdge edge = new DefaultEdge();
		
		// set edge attributes
		edge.setSource(srcPort);
		edge.setTarget(destPort);
		GraphConstants.setLineWidth(edge.getAttributes(), MESSAGE_PASSIVE_WIDTH);
		GraphConstants.setLineColor(edge.getAttributes(), MESSAGE_PASSIVE_COLOR);
		GraphConstants.setLineStyle(edge.getAttributes(), LINE_STYLE);
		GraphConstants.setLineEnd(edge.getAttributes(), ARROW_STYLE);
		GraphConstants.setLabelAlongEdge(edge.getAttributes(), true);
		
		Font f = GraphConstants.getFont(edge.getAttributes());
		f = f.deriveFont((float)TEXT_FONT);
		GraphConstants.setFont(edge.getAttributes(), f);		
		
		List<Point2D> points = c.getPoints();
		if (points.size() > 0) {
			GraphConstants.setPoints(edge.getAttributes(), points);
		}
		
		return edge;
	}
	
	
	/**
	 * Makes defaut layout of the service graph using externat dot tool.
	 */
	protected void layoutServiceGraph() {
		HashMap<Integer, Service> serviceMap = new HashMap<Integer, Service>();

		Model sModel = config.getServiceModel();
		Iterator<ModelNode> iter = sModel.getChildren(sModel.getModelRoot()).iterator();
		
		// Do noting if there is no service nodes
		if (!iter.hasNext()) {
			return;
		}
		
		try {
			Runtime env = Runtime.getRuntime();
			String command[] = {SettingsDialog.getInstance().getDotDirectoryPath() + "\\dot.exe", "-Tplain"};
			Process dot = env.exec(command);
			
			StringBuffer outBuffer = new StringBuffer();
			OutputStream outStream = dot.getOutputStream();
			StringBuffer inBuffer = new StringBuffer();
			InputStream inStream = dot.getInputStream();
			
			outBuffer.append("digraph{");
			
			outBuffer.append("ratio=\"fill\";");
			outBuffer.append("size=\"");
			outBuffer.append(String.valueOf(this.getWidth()));
			outBuffer.append(",");
			outBuffer.append(String.valueOf(this.getHeight()));
			outBuffer.append("\";");
			
			while (iter.hasNext()) {
				ModelNode node = iter.next();
				
				if (node instanceof Connection) {
					Connection c = (Connection) node;
					
					outBuffer.append(String.valueOf(c.getSrcService()));
					outBuffer.append("->");
					outBuffer.append(String.valueOf(c.getDestService()));
					outBuffer.append(";");
				
				} else if (node instanceof Service) {
					Service service = (Service)node;
					serviceMap.put(new Integer(service.getId()), service);
					
					outBuffer.append(Integer.valueOf(service.getId()));
					outBuffer.append("[shape=ellipse,width=" + SERVICE_WIDTH + ",height=" + SERVICE_HEIGHT + "];");
				}
			}
			
			outBuffer.append("}");
			
			//System.out.println("DOT input:");
			//System.out.println(outBuffer);
			
			// Write input to dot
			outStream.write(outBuffer.toString().getBytes());
			outStream.close();
			
			// Read output from dot
			int ch;
			while ((ch = inStream.read()) != -1) {
				inBuffer.append((char)ch);
			}
			inStream.close();
			
			//System.out.println("DOT output:");
			//System.out.println(inBuffer);
			
			StringTokenizer st = new StringTokenizer(inBuffer.toString());
			
			while (st.hasMoreTokens()) {
				if (st.nextToken().equals("node")) { // Read keyword 'node'
					int node = Integer.valueOf(st.nextToken()); // Read node number
					double x = Double.valueOf(st.nextToken()); // Read x coord of the node
					double y = Double.valueOf(st.nextToken()); // Read y coord of the node
					
					Service service = serviceMap.get(node);
							
					if (service != null)  {
						service.setLocation(new Point2D.Double(x - SERVICE_WIDTH/2, y - SERVICE_HEIGHT/2));
					} else {
						System.out.println("Node not found!");
					}
				}
			}
			
		} catch (Exception e) {
			System.out.println("Dot failed!");
		}
	}

	
	/**
	 * Returns the time between activation steps in milliseconds.
	 */
	public double getActivationTime() {
		return activationTime;
	}

	
	/**
	 * Sets the time between activation steps in seconds.
	 */
	public void setActivationTime(double secs) {
		activationTime = secs;
	}

	
	/**
	 * Sets services and connections to be (de)activated.
	 * @param info The activation info.
	 */
	public void setActivationStates(ServiceActivationInfo info) {
        // Create object for containing the items that
        // really need to be activated
        ServiceActivationInfo activationInfo = new ServiceActivationInfo();
        
		if (info != null) {
            // Set timers to do deactivation
            for (int i=0; i<info.getItemCount(); ++i) {
                int type = info.getItemType(i);
                int id = info.getItemId(i);
                int value = info.getItemValue(i);
                
                // Check that the service/connection exists
                if ((type == ServiceActivationInfo.SERVICE
                        && !graphCells.containsKey(new Integer(id)))
                        || (type == ServiceActivationInfo.CONNECTION
                                && !edges.containsKey(new Integer(id)))) {
                    continue;
                }

                Integer idInt = new Integer(id);
                Integer valueInt = new Integer(value);
                
                // Replace the old value with new
                Integer oldValueInt = null;
                if (type == ServiceActivationInfo.SERVICE) {
                    oldValueInt = serviceValues.get(idInt);
                    serviceValues.put(idInt, valueInt);
                } else {
                    oldValueInt = connectionValues.get(idInt);
                    connectionValues.put(idInt, valueInt);                    
                }

                int oldValue = 0;
                if (oldValueInt != null) {
                    oldValue = oldValueInt.intValue();
                }
                
                if (oldValue <= 0 && value > 0) {
                    // Remove the timer so that it won't fire
                    Timer t = null;
                    if (type == ServiceActivationInfo.SERVICE) {
                        t = serviceTimers.remove(idInt);
                        if (t != null) {
                            t.stop();
                        }
                        
                    } else {
                        t = connectionTimers.remove(idInt);
                        if (t != null) {
                            t.stop();
                        }
                    }
                    
                    // Schedule the item to be activated
                    activationInfo.addActivatedItem(type, id, value);
                    
                } else if (oldValue > 0 && value <= 0) {
                    // The value changed to zero. Start the timer to handle
                    // the deactivation.
                    Timer t = null;
                    if (type == ServiceActivationInfo.SERVICE) {
                        t = serviceTimers.get(new Integer(id));
                    } else {
                        t = connectionTimers.get(new Integer(id));
                    }
                    
                    if (t == null) {
                        // Create new timer
                        ServiceActivationInfo deactivationInfo =
                            new ServiceActivationInfo();
                        deactivationInfo.addActivatedItem(
                                type, id, value);
                        Timer newTimer = new Timer(0,
                                new MyTimerListener(deactivationInfo));
                        
                        if (type == ServiceActivationInfo.SERVICE) {
                            serviceTimers.put(new Integer(id), newTimer);
                        } else {
                            connectionTimers.put(new Integer(id), newTimer);
                        }
                        
                        newTimer.setInitialDelay((int)activationTime*1000);
                        newTimer.setRepeats(false);
                        newTimer.start();
                        
                    } else {
                        // Update the firing delay and restart
                        t.setInitialDelay((int)activationTime*1000);
                        t.restart();
                    }
                    
                } else if (value > 0 && value != oldValue) {
                    // The item needs to be updated
                    activationInfo.addActivatedItem(type, id, value);
                    
                } else {
                    continue;
                }
            }
		}
        
        // Activate items
        setActive(activationInfo, true);
	}

	
	/**
	 * Sets the active state of items.
	 */
	protected void setActive(ServiceActivationInfo info, boolean active) {
		if (info == null) {
			return;
		}
		
		for (int i=0; i<info.getItemCount(); ++i) {
			int id = info.getItemId(i);
			int type = info.getItemType(i);
			int value = info.getItemValue(i);
			
			if (type == ServiceActivationInfo.SERVICE) {
				DefaultGraphCell cell = graphCells.get(new Integer(id));
				if (cell == null) {
					continue;
				}
				
				if (active) {
					GraphConstants.setBackground(cell.getAttributes(), SERVICE_ACTIVE);
				} else {
					GraphConstants.setBackground(cell.getAttributes(), SERVICE_PASSIVE);					
				}
				
			} else if (type == ServiceActivationInfo.CONNECTION) {
				DefaultEdge edge = edges.get(new Integer(id));
				if (edge == null) {
					continue;
				}
				
				DefaultGraphCell srcCell = (DefaultGraphCell)((DefaultPort)edge.getSource()).getParent();
				DefaultGraphCell targetCell = (DefaultGraphCell)((DefaultPort)edge.getTarget()).getParent();
				
				Rectangle2D srcBounds = GraphConstants.getBounds(srcCell.getAttributes());
				Rectangle2D targetBounds = GraphConstants.getBounds(targetCell.getAttributes());
				
				int yOffset = 10;
				if (srcBounds.getX() < targetBounds.getX()) {
					yOffset = -10;
				}
				
				if (active) {
					edge.setUserObject(""+value);
					GraphConstants.setLineColor(edge.getAttributes(), MESSAGE_ACTIVE_COLOR);
					GraphConstants.setLineWidth(edge.getAttributes(), MESSAGE_ACTIVE_WIDTH);
					GraphConstants.setLabelPosition(edge.getAttributes(), new Point2D.Double(500, yOffset));
					
				} else {
					edge.setUserObject("");
					GraphConstants.setLineColor(edge.getAttributes(), MESSAGE_PASSIVE_COLOR);
					GraphConstants.setLineWidth(edge.getAttributes(), MESSAGE_PASSIVE_WIDTH);
				}
			}
		}
		graphView.reload();
		graph.repaint();
	}
	
	
	/**
	 * Clears the activation information
	 */
	public void clearActives() {
        ServiceActivationInfo deactivationInfo = new ServiceActivationInfo();
        
        // Handle service timers
        Iterator<Timer> tIter = serviceTimers.values().iterator();
        while (tIter.hasNext()) {
            Timer t = tIter.next();
            t.stop();
            
            ActionListener[] listeners = t.getActionListeners();
            for (int i=0; i<listeners.length; ++i) {
                ActionListener l = listeners[i];
                if (l instanceof MyTimerListener) {
                    ServiceActivationInfo info =
                        ((MyTimerListener)l).getActivationInfo();
                    
                    for (int j=0; j<info.getItemCount(); ++j) {
                        deactivationInfo.addActivatedItem(
                                info.getItemType(j),
                                info.getItemId(j),
                                info.getItemValue(j));
                    }
                }
            }
        }
        
        // Handle connection timers
        tIter = connectionTimers.values().iterator();
        while (tIter.hasNext()) {
            Timer t = tIter.next();
            t.stop();
            
            ActionListener[] listeners = t.getActionListeners();
            for (int i=0; i<listeners.length; ++i) {
                ActionListener l = listeners[i];
                if (l instanceof MyTimerListener) {
                    ServiceActivationInfo info =
                        ((MyTimerListener)l).getActivationInfo();
                    
                    for (int j=0; j<info.getItemCount(); ++j) {
                        deactivationInfo.addActivatedItem(
                                info.getItemType(j),
                                info.getItemId(j),
                                info.getItemValue(j));
                    }
                }
            }
        }

        serviceTimers.clear();
        connectionTimers.clear();
		setActive(deactivationInfo, false);
	}

	
	/**
	 * Listener for action events from timer.
     * When the timer fires, related items are deactivated.
	 */
	private class MyTimerListener implements ActionListener {
        private ServiceActivationInfo info;
        
        public MyTimerListener(ServiceActivationInfo info) {
            this.info = info;
        }

        public ServiceActivationInfo getActivationInfo() {
            return info;
        }
        
		public void actionPerformed(ActionEvent e) {
            // deactivate items
            setActive(info, false);
            
            // remove timers
            for (int i=0; i<info.getItemCount(); ++i) {
                int id = info.getItemId(i);
                int type = info.getItemType(i);
                
                if (type == ServiceActivationInfo.SERVICE) {
                    serviceTimers.remove(new Integer(id));
                } else {
                    connectionTimers.remove(new Integer(id));
                }
            }            
		}
	}
	
		
	/**
	 * Listener for GraphModelEvents
	 */
	private class MyGraphModelListener implements GraphModelListener {
		public void graphChanged(GraphModelEvent e) {
			GraphModelChange change = e.getChange();
			Object[] changedObjects = change.getChanged();
			
			for (int i=0; i<changedObjects.length; ++i) {
				Object o = changedObjects[i];
				if (o instanceof DefaultEdge) {
					DefaultEdge edge = (DefaultEdge) o;
					Connection connection = connections.get(edge);
					
					if (connection != null) {
						List points = GraphConstants.getPoints(edge.getAttributes());
						if (points != null) {
							connection.setPoints(points);
						}
					}
					
				} else if (o instanceof DefaultGraphCell) {
					DefaultGraphCell cell = (DefaultGraphCell) o;
					Service service = services.get(cell);
					
					if (service != null) {
						Rectangle2D bounds = GraphConstants.getBounds(cell.getAttributes());
						service.setLocation(new Point2D.Double(bounds.getX(), bounds.getY()));
					}	
				}
			}
		}
	}
	
	
    /**
     * Listener for ModelEvents.
     */
    private class MyModelListener implements ModelListener {
    	public void nodeInserted(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		if (parent != null && parent.equals(config.getServiceModel().getModelRoot())) {
    			// Services/connections changed.
    			initPanel();
    		}
    	}
    	public void nodeRemoved(ModelEvent e) {
    		ModelNode oldParent = e.getOldParent();
    		
    		if (oldParent != null && oldParent.equals(config.getServiceModel().getModelRoot())) {
    			// Services/connections changed.
    			initPanel();
    		}
    	}
    	public void nodeMoved(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		ModelNode oldParent = e.getOldParent();
    		
    		if ((parent != null && parent.equals(config.getServiceModel().getModelRoot()))
    				|| (oldParent != null && oldParent.equals(config.getServiceModel().getModelRoot()))) {
    			// Services/connections changed.
    			initPanel();
    		}
    	}
    	public void structureChanged(ModelEvent e) {
    		ModelNode node = e.getNode();
    		
    		if (node != null && node.equals(config.getServiceModel().getModelRoot())) {
    			// Structure changed starting from the root.
    			initPanel();
    		}
    	}
    }    
}
