/**
 * CVS:
 * $Id: ThreadPanel.java 1399 2010-08-26 13:56:45Z lehton87 $
 * 
 * File:    ThreadPanel.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 1.12.2006
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui.report;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.Rectangle2D;
import java.util.Collections;
import java.util.Iterator;
import java.util.ResourceBundle;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.ScrollPaneConstants;
import javax.swing.border.TitledBorder;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumn;

import fi.cpu.Settings;
import fi.cpu.data.Chart;
import fi.cpu.data.Configuration;
import fi.cpu.data.ModelNode;
import fi.cpu.data.Process;
import fi.cpu.data.Thread;
import fi.cpu.event.ModelEvent;
import fi.cpu.event.ModelListener;
import fi.cpu.event.ModelNodeEvent;
import fi.cpu.event.ModelNodeListener;
import fi.cpu.table.ReportTableCellRenderer;
import fi.cpu.table.ReportTableHeaderRenderer;
import fi.cpu.ui.MainWindow;
import fi.cpu.ui.graph.GraphPanel;
import fi.cpu.ui.graph.SimpleChartFactory;


/**
 * A panel fro showing report data of a thread.
 */
public class ThreadPanel extends JPanel implements Comparable<ThreadPanel> {
    protected static final Color BACKGROUND;
    protected static final Color BORDER_COLOR;
    
    protected static Vector<String> COLUMN_NAMES = new Vector<String>();
    protected static final String THREAD_TOTAL_LABEL;
    protected static final String THREAD_CHART_TITLE;
    protected static final String CHART_ID_PREFIX = "thread";
    protected static final int FRACTION_DIGITS;

    protected Thread thread;
    protected Configuration config;
	protected Vector<Process> processes;
	protected Process changedProcess;
	protected DefaultTableModel tableModel;
	protected JTable table;
    protected JPopupMenu popupMenu;
	protected int popupRow = 0;
	protected int popupColumn = 0;
    protected Font defaultFont;
    protected Font sumRowFont;
    protected MyProcessListener processListener;
    
    
	static {
		ResourceBundle bundle = MainWindow.bundle;
		COLUMN_NAMES.add(" ");
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_ECOUNT"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_AVG_ETIME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_TOT_ETIME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_THREAD_REL_ETIME"));
		THREAD_TOTAL_LABEL = bundle.getString("REPORT_LABEL_THREAD_TOTAL");
		THREAD_CHART_TITLE = bundle.getString("CHART_TITLE_THREAD");
		
		FRACTION_DIGITS = Integer.parseInt(Settings.getAttributeValue("REPORT_FRACTION_DIGITS"));
		
        int red = 0;
        int green = 0;
        int blue = 0;
        
        red = Integer.parseInt( Settings.getAttributeValue("THREAD_BACKGROUD_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("THREAD_BACKGROUD_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("THREAD_BACKGROUD_COLOR_BLUE") );
        BACKGROUND = new Color(red, green, blue);

        red = Integer.parseInt( Settings.getAttributeValue("THREAD_TEXT_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("THREAD_TEXT_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("THREAD_TEXT_COLOR_BLUE") );
        BORDER_COLOR = new Color(red, green, blue);
	}
	

	/**
	 * Creates a new ThreadPanel.
	 */
	public ThreadPanel(Thread thread) {
		super();
		this.thread = thread;
		processes = new Vector<Process>();
		processListener = new MyProcessListener();
		config = MainWindow.getInstance().getConfiguration();
		
		thread.addModelNodeListener(new MyThreadListener());
		config.getPeModel().addModelListener(new MyModelListener());
		
		defaultFont = getFont();
		sumRowFont = defaultFont.deriveFont(Font.BOLD);
		tableModel = new DefaultTableModel(COLUMN_NAMES, 0);

        setBackground( BACKGROUND );
		setLayout(new GridBagLayout());
		initBorder();
		
		// Create the table view
		table = new JTable(tableModel);
		table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		table.getTableHeader().setReorderingAllowed(false);
		table.setDefaultRenderer(Object.class, new ReportTableCellRenderer(defaultFont, sumRowFont));
		table.setDefaultEditor(Object.class, null);
		table.addMouseListener(new MyMouseListener());
		
		// Set column widths to be large enough to show full titles
		TableColumn column = null;
		FontMetrics fm = null;
		Graphics g = getGraphics();
		for (int i=0; i<COLUMN_NAMES.size(); ++i)
		{
			String columnTitle = null;
			if (i == 0)
			{
				columnTitle = THREAD_TOTAL_LABEL;
				fm = getFontMetrics(sumRowFont);
			}
			else 
			{
				columnTitle = COLUMN_NAMES.get(i);
				fm = getFontMetrics(defaultFont);
			}
			Rectangle2D bounds = fm.getStringBounds(columnTitle, g);
			int width = (int)bounds.getWidth();

			column = table.getColumnModel().getColumn(i);
			column.setPreferredWidth(width+8); // a little bit extra for borders, empty space etc.
		}
		
		// Configure the table header
		DefaultTableCellRenderer headerRenderer = new ReportTableHeaderRenderer();
		table.getTableHeader().setDefaultRenderer(headerRenderer);
		table.getTableHeader().setBackground(BACKGROUND);

		// Add items to panel
		JScrollPane scrollPane = new JScrollPane(table, ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
		ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        
        scrollPane.getViewport().setBackground(BACKGROUND);
        scrollPane.setBorder(null);
        scrollPane.getViewport().setBorder(null);
        this.add(scrollPane, new GridBagConstraints(
        		0, 0, GridBagConstraints.REMAINDER, GridBagConstraints.REMAINDER,
        		1.0, 1.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH,
        		new Insets(0, 0, 0, 0), 0, 0));
        
		popupMenu = new JPopupMenu();
		JMenuItem showChartItem = new JMenuItem(MainWindow.bundle.getString("SHOW_GRAPH"));
		showChartItem.addActionListener(new MyActionListener());
		popupMenu.add(showChartItem);

		initPanel();
	}
	
	
	/**
	 * Initializes the panel.
	 */
	public void initPanel() {
		processes.clear();
		
        Iterator<ModelNode> pIter = config.getPeModel().getChildren(thread).iterator();

        while (pIter.hasNext()) {
        	ModelNode node = pIter.next();
        	if (!(node instanceof Process)) {
        		continue;
        	}
        	Process p = (Process) node;
        	p.addModelNodeListener(new MyProcessListener());
        	processes.add(p);
        }
        
        // Sort processes
		Collections.sort(processes);
		
		// clear the table
		tableModel.setRowCount(0);
		
		// add process rows
		for (int i=0; i<processes.size(); ++i) {			
			Process p = processes.get(i);
			Object[] rowData = {p.getName(), new Integer(p.getExecCount()),
					new Double(p.getAvgExecTime()), new Double(p.getTotExecTime()), new Double(0)};
			tableModel.addRow(rowData);
		}
		
		// add thread total row
		Object[] sumRow = {THREAD_TOTAL_LABEL,
				new Integer(thread.getExecCount()),
				new Double(thread.getAvgExecTime()),
				new Double(thread.getTotExecTime()),
				new Double(0)};
		tableModel.addRow(sumRow);

		updateDataTable();
	}
	

	/**
	 * Initializes the border.
	 */
	protected void initBorder() {
        StringBuilder title = new StringBuilder(" ");
        title.append(MainWindow.bundle.getString("THREAD_TITLE_THREAD"));
        title.append(" ");
        title.append(thread.getName());
        title.append(" - ");
        title.append(MainWindow.bundle.getString("THREAD_TITLE_PRIORITY"));
        title.append(" ");
        title.append(thread.getPriority());
 
        TitledBorder border = BorderFactory.createTitledBorder(
        		BorderFactory.createLineBorder( BORDER_COLOR ), title.toString());
        setBorder( border );
	}

	
	/**
	 * @return The thread model object.
	 */
	public Thread getThread() {
		return thread;
	}
	


	/**
	 * Updates the data in the table.
	 */
	protected void updateDataTable() {	
		// update process rows
		for (int i=0; i<processes.size(); ++i) {			
			Process p = processes.get(i);
			
			Object[] rowData = {p.getName(), new Integer(p.getExecCount()),
					new Double(p.getAvgExecTime()), new Double(p.getTotExecTime()), new Double(0)};
			
			for (int j=0; j<rowData.length; ++j) {
				tableModel.setValueAt(rowData[j], i, j);
			}
		}
		
		// update thread total row
		int threadRowIndex = table.getRowCount()-1;

		Object[] sumRow = {THREAD_TOTAL_LABEL,
				new Integer(thread.getExecCount()),
				new Double(thread.getAvgExecTime()),
				new Double(thread.getTotExecTime()),
				new Double(0)};
		
		for (int j=0; j<sumRow.length; ++j) {
			tableModel.setValueAt(sumRow[j], threadRowIndex, j);
		}
		
		updateRelativeValues();
		
		// update process charts
		if (changedProcess != null && processes.indexOf(changedProcess) != -1) {
			int pIndex = processes.indexOf(changedProcess);
			
			for (int i=1; i<table.getColumnCount(); ++i) {
				StringBuilder chartID = new StringBuilder();
				chartID.append(changedProcess.getName());
				chartID.append(":");
				chartID.append(table.getColumnName(i));
				
				Number value = (Number)table.getValueAt(pIndex, i);
				MainWindow.getInstance().addValueToChart(
						null, chartID.toString(), value.doubleValue());
			}
			changedProcess = null;
		}
		
		// update thread charts
		for (int i=1; i<table.getColumnCount(); ++i) {
			StringBuilder chartID = new StringBuilder();
			chartID.append(CHART_ID_PREFIX);
			chartID.append(thread.getId());
			chartID.append(":");
			chartID.append(table.getColumnName(i));
			
			Number value = (Number)table.getValueAt(threadRowIndex, i);
			MainWindow.getInstance().addValueToChart(
					null, chartID.toString(), value.doubleValue());
		}

		// refresh the view
		revalidate();
		repaint();
	}
	
	
	/**
	 * Updates the column with thread relative values.
	 */
	private void updateRelativeValues() {
		Double threadTotETime = (Double)tableModel.getValueAt(tableModel.getRowCount()-1, 3);
		Double sum = 0.0;
		
		for (int i=0; i<processes.size(); ++i) {
			Process p = processes.get(i);
			Double tRel = new Double(0);
			
			// do not divide by 0
			if (threadTotETime.doubleValue() != 0) {
				tRel = new Double((p.getTotExecTime() / threadTotETime.doubleValue()) * 100);
			}
			tableModel.setValueAt(new Double(tRel), i, 4);
			sum += tRel;
		}
		tableModel.setValueAt(new Double(sum), tableModel.getRowCount()-1, 4);
	}
	
	
	/**
	 * Implements the Compararable-interface
	 */
    public int compareTo(ThreadPanel o) {
    	return thread.compareTo(o.thread);
    }
	
	
	/**
	 * Listener for MouseEvents that may trigger the popup menu.
	 */
	private class MyMouseListener extends MouseAdapter {
		public void mousePressed(MouseEvent e) {
			showPopupIfTriggered(e);
		}
		public void mouseReleased(MouseEvent e) {
			showPopupIfTriggered(e);
		}
		private void showPopupIfTriggered(MouseEvent e) {
			if (e.isPopupTrigger()) {
				Point p = e.getPoint();
				popupRow = table.rowAtPoint(p);
				popupColumn = table.columnAtPoint(p);
				if (popupColumn > 0 && popupRow >= 0) {
					popupMenu.show(e.getComponent(), e.getX(), e.getY());
				}
			}			
		}
	}

	
	/**
	 * Listener for actions from the popup menu.
	 */
	private class MyActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			StringBuilder title = new StringBuilder();
			StringBuilder chartID = new StringBuilder();
			
			if (popupRow == table.getRowCount()-1) {
				title.append(THREAD_CHART_TITLE);
				title.append(" ");
				title.append(thread.getName());
				
				chartID.append(CHART_ID_PREFIX);
				chartID.append(thread.getId());
				
			} else {
				String pName = (String)table.getValueAt(popupRow, 0);
				title.append(pName);
				chartID.append(pName);
			}
			
			title.append(": ");
			title.append(table.getColumnName(popupColumn));
			int historySize = 0;
			chartID.append(":");
			chartID.append(table.getColumnName(popupColumn));
			
			try {
				String history = System.getProperty("history");
				if (history != null) {
					historySize = Integer.parseInt(history);
				}
			} catch (NumberFormatException nfe) {
				// do nothing
			}
			
			// Create the chart and add it to main window chart panel
			GraphPanel panel = new GraphPanel();
			Chart chart = new Chart(chartID.toString(), SimpleChartFactory.LINECHART_TYPE,
					title.toString(), MainWindow.bundle.getString("XAXIS"), "", 0.0, 0.0,
					historySize, true, false);
			panel.setChart(chart);
			MainWindow.getInstance().addChart(panel, 10, 10, 200, 300);
		}
	}
	
	
	/**
	 * Listener for events from the represented thread.
	 */
	private class MyThreadListener implements ModelNodeListener {
		public void modelNodeChanged(ModelNodeEvent e) {
			if (e.getEventId() == Thread.PRIORITY_CHANGED_EVENT) {
				initBorder();	
			} else if (e.getEventId() == Thread.DATA_CHANGED_EVENT) {
				updateDataTable();
			}
		}
	}
	
	
	/**
	 * Listener for events from contained processes.
	 */
	private class MyProcessListener implements ModelNodeListener {
		public void modelNodeChanged(ModelNodeEvent e) {
			if (e.getEventId() == Process.DATA_CHANGED_EVENT) {
				Object source = e.getSource();
				
				if (source instanceof Process) {
					Process p = (Process) source;
					changedProcess = p;
				}
			}
		}
	}
	
	
    /**
     * Listener for ModelEvents.
     */
    private class MyModelListener implements ModelListener {
    	public void nodeInserted(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		if (parent != null && parent.equals(thread)) {
    			// Thread's children changed.
    			initPanel();
    		}
    	}
    	public void nodeRemoved(ModelEvent e) {
    		ModelNode node = e.getNode();
    		ModelNode oldParent = e.getOldParent();
    		
    		if (node != null && node.equals(thread)) {
    			// Thread was removed.
    			// Stop listening events, because this panel
    			// will be removed.
    			config.getPeModel().removeModelListener(this);
    		} else if (oldParent != null && oldParent.equals(thread)) {
    			// Thread's children changed.
    			initPanel();
    		}
    	}
    	public void nodeMoved(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		ModelNode oldParent = e.getOldParent();
    		
    		if ((parent != null && parent.equals(thread))
    				|| (oldParent != null && oldParent.equals(thread))) {
    			// Thread's children changed.
    			initPanel();    			
    		}
    	}
    	public void structureChanged(ModelEvent e) {
    		ModelNode node = e.getNode();
    		
    		if (node != null && node.equals(thread)) {
    			// Structure changed starting from this node.
    			initPanel();
    		} else if (config.getPeModel().isChildOf(thread, node)) {
    			// Thread is part of the changed structure.
    			// Stop listening events, because this panel
    			// will be removed.
    			config.getPeModel().removeModelListener(this);
    		}
    	}
    }
}
