/**
 * CVS: $Id: ScrollablePanel.java 1399 2010-08-26 13:56:45Z lehton87 $
 * 
 * File:    ScrollablePanel.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 6.11.2006
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;

import javax.swing.JPanel;
import javax.swing.Scrollable;
import javax.swing.SwingConstants;


/**
 * ScrollablePanel is a panel that can be scrolled
 * e.g. inside a JScrollPane. ScrollablePanel resizes
 * itself when its child components change sizes.
 */
public class ScrollablePanel extends JPanel implements Scrollable {
	protected ScrollablePanel self_;
	protected final int DEFAULT_SCROLLABLE_UNIT_INCREMENT = 5;
	
	/**
	 * Create a new ScrollablePanel.
	 */
	public ScrollablePanel() {
		super();
		self_ = this;
		setPreferredSize(new Dimension(200,200));
		addContainerListener(new MyContainerListener());
	}
	
	
	/* Methods that implement the Scrollable interface. */
	public Dimension getPreferredScrollableViewportSize() {
		return getPreferredSize();
	}
	
	public boolean getScrollableTracksViewportHeight() {
		return false;
	}

	public boolean getScrollableTracksViewportWidth() {
		return false;
	}
	
	public int getScrollableUnitIncrement(Rectangle visibleRect, int orientation, int direction) {
		return DEFAULT_SCROLLABLE_UNIT_INCREMENT;
	}
	
	public int getScrollableBlockIncrement(Rectangle visibleRect, int orientation, int direction) {
		if (orientation == SwingConstants.HORIZONTAL) {
			return visibleRect.width;
		} else {
			return visibleRect.height;
		}
	}

	
	/**
	 * Resizes the panel to contain all the space child
	 * components need.
	 */
	protected void resize() {
		int maxX = 0;
		int maxY = 0;
		for (int i=0; i<self_.getComponentCount(); ++i) {
			Component comp = self_.getComponent(i);
			Rectangle bounds = comp.getBounds();
			int x = bounds.x + bounds.width;
			int y = bounds.y + bounds.height;
			
			if (maxX < x) {
				maxX = x;
			}
			if (maxY < y) {
				maxY = y;
			}
		}
		setPreferredSize(new Dimension(maxX,maxY));
		repaint();
	}

	
	/**
	 * Class for listening ContainerEvents.
	 */
	private class MyContainerListener implements ContainerListener {
		public void componentAdded(ContainerEvent e) {
			e.getChild().addComponentListener(new MyComponentListener());
			resize();
		}
		
		public void componentRemoved(ContainerEvent e) {
			resize();
		}
	}
	
	
	/**
	 * Class for listening ComponentEvents from child components. 
	 */
	protected class MyComponentListener implements ComponentListener {
		public void componentHidden(ComponentEvent e) {
			resize();
		}
		public void componentMoved(ComponentEvent e) {
			resize();
		}
		public void componentResized(ComponentEvent e) {
			resize();
		}
		public void componentShown(ComponentEvent e) {
			resize();
		}
	}
}
