/*
 * CVS-tietoja:
 * $Author: perttul5 $ 
 * $Date: 2008-05-20 17:23:56 +0300 (ti, 20 touko 2008) $ 
 * $Revision: 3514 $
 *
 * Created on 19.7.2005
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *
 */
package fi.cpu.ui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.beans.PropertyVetoException;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JDesktopPane;
import javax.swing.JFrame;
import javax.swing.JInternalFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTabbedPane;
import javax.swing.JTextArea;
import javax.swing.JToggleButton;
import javax.swing.KeyStroke;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;
import javax.swing.event.MouseInputAdapter;

import fi.cpu.Settings;
import fi.cpu.data.ChartDataLogger;
import fi.cpu.data.Configuration;
import fi.cpu.data.Model;
import fi.cpu.data.TraceDataLogger;
import fi.cpu.event.ModelEvent;
import fi.cpu.event.ModelListener;
import fi.cpu.handler.Handler;
import fi.cpu.serial.LinuxSerial;
import fi.cpu.serial.WindowsSerial;
import fi.cpu.tcp.TCPClient;
import fi.cpu.ui.ctrl.ControlPanel;
import fi.cpu.ui.ctrl.PePanel;
import fi.cpu.ui.ctrl.ProcessIcon;
import fi.cpu.ui.graph.GraphPanel;
import fi.cpu.ui.report.ReportPanel;
import fi.cpu.ui.service.ServiceGraphPanel;
import fi.cpu.ui.service.ServiceReportPanel;
import fi.cpu.ui.xml.UIConfigIO;


/**
 * The main window for Execution Monitor
 */
public class MainWindow extends JFrame {
    public static final String VERSION = Settings.getAttributeValue("VERSION");

    public static final String EXIT_NAME = "EXIT";
    public static final String OPEN_NAME = "OPEN";
    public static final String SAVE_NAME = "SAVE";
    public static final String SAVE_AS_NAME = "SAVE_AS";
    public static final String SHOW_DEBUG_WINDOW = "SHOW_DEBUG_WINDOW";
    public static final String ADD_GRAPH = "ADD_GRAPH";
    public static final String ABOUT_NAME = "ABOUT";
    public static final String EDIT_SETTINGS_NAME = "EDIT_SETTINGS";
    public static final String CONNECTION = "CONNECTION";

    private static final String UITEXT_BUNDLE_NAME = "uitexts";
    private static final String SERIAL_BUNDLE_NAME = "serial";
    protected static final String RECONNECT_CMD = "reconnect";
    protected static final String ACTIVATE_PROCESSES_CMD = "activate";
    protected static final String RUN_USECASE_CMD = "runusecase";

    public static ResourceBundle bundle;
    private ResourceBundle serialProperties;

    protected static MainWindow self;
    protected Configuration configuration;
    protected Handler connectionHandler;
    protected File currentLogDir;
    protected File currentTraceDir;
    protected boolean activatingProcesses;
    protected Map<String, GraphPanel> chartPanels;

    protected ProcessIcon draggedProcess;
    private MyProcessDragListener dragListener;
    private Point dragPoint;
    private JTextArea debugTextArea;
    private JInternalFrame debugWindow;

    protected ImageIcon logoIcon;
    protected JMenuItem connectionMenuItem;
    protected JPopupMenu popupMenu;
    protected JTabbedPane tabbedPane;
    protected ControlPanel controlPanel;
    protected ReportPanel peReportPanel;
    protected JDesktopPane chartPanel;
    protected ProcessesPanel processesPanel;
    protected SignalsPanel signalsPanel;
    protected ServiceGraphPanel servicePanel;
    protected ServiceReportPanel serviceReportPanel;
    protected JButton reconnectButton;
    protected JButton activateButton;
    protected JButton runUseCaseButton;
    protected JToggleButton doLoggingButton;
    protected JToggleButton doTracingButton;
    protected JCheckBox autoActiveCheckBox;
    protected JPanel serviceActiveTimePanel;
    protected JSpinner serviceActiveTimeSpinner;
    protected JLabel serviceActiveTimeLabel;

    
    
    /**
     * @return An instance of MainWindow.
     */
    public static MainWindow getInstance() {
        if (self == null) {
            self = new MainWindow();
        }
        return self;
    }

    
    /**
     * Creates new MainWindow.
     */
    private MainWindow() {
    	configuration = new Configuration();
    	dragListener = new MyProcessDragListener();
    	chartPanels = new HashMap<String, GraphPanel>();
    	activatingProcesses = false;
    	
		// Create icon
		String iconLocation = "/fi/cpu/resources/koski_logo_16x16.gif";
		URL iconURL = MainWindow.class.getResource(iconLocation);
		
		if (iconURL != null) {
			logoIcon = new ImageIcon(iconURL);
			setIconImage(logoIcon.getImage());
			
		} else {
			System.err.println("Couldn't load Koski logo.");
		}
    	
    	// Create resource bundles
        bundle = ResourceBundle.getBundle(UITEXT_BUNDLE_NAME);
        serialProperties = ResourceBundle.getBundle(SERIAL_BUNDLE_NAME);

        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        setTitle(bundle.getString("MAINTITLE"));
        setPreferredSize(new Dimension(800, 600));
        addWindowListener(new MyWindowListener());

        // Create chart tab
        chartPanel = new JDesktopPane();
        chartPanel.addMouseListener(new MyMouseListener());
        chartPanel.setBackground(getBackground());

        // Create other tabs
        controlPanel = new ControlPanel(configuration);
        processesPanel = new ProcessesPanel(configuration);
        signalsPanel = new SignalsPanel(configuration);
        peReportPanel = new ReportPanel(configuration);
        servicePanel = new ServiceGraphPanel(configuration);
        serviceReportPanel = new ServiceReportPanel(configuration);

        // Create tabbed pane
        tabbedPane = new JTabbedPane();
        tabbedPane.addChangeListener(new MyTabChangeListener());
        
        tabbedPane.addTab(bundle.getString("CONTROL_TAB"), controlPanel);
        tabbedPane.addTab(bundle.getString("PE_REPORT_TAB"), peReportPanel);
        tabbedPane.addTab(bundle.getString("PROCESSES_TAB"), processesPanel);
        tabbedPane.addTab(bundle.getString("SIGNALS_TAB"), signalsPanel);
        tabbedPane.addTab(bundle.getString("SERVICES_TAB"), servicePanel);
        tabbedPane.addTab(bundle.getString("SERVICE_REPORT_TAB"), serviceReportPanel);
        tabbedPane.addTab(bundle.getString("CHARTS_TAB"), chartPanel);

        // Add main components to view
        setLayout(new BorderLayout());
        add(tabbedPane, BorderLayout.CENTER);
        add(createBottomPanel(), BorderLayout.SOUTH);
        
        configuration.getServiceModel().addModelListener(new MyServiceModelListener());
        
        initMenus();
        initConnectionHandler();
    }

    
    /**
     * Creates the panel at the bottom of the main window
     */
    private JPanel createBottomPanel() {
    	MyActionListener actionListener = new MyActionListener();
    	
    	// Create the panel
        JPanel bottomPanel = new JPanel();
        bottomPanel.setBorder(BorderFactory.createEtchedBorder());
        bottomPanel.setLayout(new BoxLayout(bottomPanel, BoxLayout.X_AXIS));
        
        // Create button for reconnection
        reconnectButton = new JButton(bundle.getString("RECONNECT_BUTTON"));
        reconnectButton.setMnemonic(KeyEvent.VK_R);
        reconnectButton.setActionCommand(RECONNECT_CMD);
        reconnectButton.addActionListener(actionListener);
        
        // Create button for run a use case
        runUseCaseButton = new JButton(bundle.getString("RUN_USECASE_BUTTON"));
        runUseCaseButton.setMnemonic(KeyEvent.VK_U);
        runUseCaseButton.setActionCommand(RUN_USECASE_CMD);
        runUseCaseButton.addActionListener(actionListener);
        
        // Create components for process activation
        autoActiveCheckBox = new JCheckBox( bundle.getString( "AUTO_ACTIVE" ) );
        autoActiveCheckBox.setSelected( true );

        activateButton = new JButton( bundle.getString( "ACTIVE_BUTTON" ) );
        activateButton.setMnemonic(KeyEvent.VK_A);
        activateButton.setActionCommand(ACTIVATE_PROCESSES_CMD);
        activateButton.addActionListener(actionListener);

        // Create button for starting and stopping chart data logging
        doLoggingButton = new JToggleButton(bundle.getString("LOGGING_BUTTON"));
        doLoggingButton.setMnemonic(KeyEvent.VK_L);
        doLoggingButton.addItemListener(new MyLoggingButtonListener());
        
        // Create button for starting and stopping execution tracing
        doTracingButton = new JToggleButton(bundle.getString("TRACING_BUTTON"));
        doTracingButton.setMnemonic(KeyEvent.VK_T);
        doTracingButton.addItemListener(new MyTracingButtonListener());
        
        // Create components for service activation time selection
		double activeTimeDefault = Double.parseDouble( Settings.getAttributeValue("SERVICE_ACTIVE_TIME_DEFAULT"));
		double activeTimeMin = Double.parseDouble( Settings.getAttributeValue("SERVICE_ACTIVE_TIME_MIN"));
		double activeTimeMax = Double.parseDouble( Settings.getAttributeValue("SERVICE_ACTIVE_TIME_MAX"));
		double activeTimeStep = Double.parseDouble( Settings.getAttributeValue("SERVICE_ACTIVE_TIME_STEP"));
		
		serviceActiveTimePanel = new JPanel();
		serviceActiveTimePanel.setLayout(new GridBagLayout());
        serviceActiveTimeLabel = new JLabel(bundle.getString("ACTIVATION_TIME_LENGTH"));
        
        // create spinner for selecting activation duration
        SpinnerNumberModel spinnerModel = new SpinnerNumberModel(
        		activeTimeDefault, activeTimeMin, activeTimeMax, activeTimeStep);
        serviceActiveTimeSpinner = new JSpinner(spinnerModel);
        serviceActiveTimeSpinner.addChangeListener(new ChangeListener() {
        	public void stateChanged(ChangeEvent e) {
        		Double value = (Double)serviceActiveTimeSpinner.getValue();
        		servicePanel.setActivationTime(value);
        	}
        });

        serviceActiveTimePanel.add(serviceActiveTimeLabel, new GridBagConstraints(
        		0, 0, 1, GridBagConstraints.REMAINDER, 1.0, 0.0,
        		GridBagConstraints.EAST, GridBagConstraints.NONE,
        		new Insets(0, 0, 0, 0), 0, 0));
        serviceActiveTimePanel.add(serviceActiveTimeSpinner, new GridBagConstraints(
        		1, 0, GridBagConstraints.REMAINDER, GridBagConstraints.REMAINDER, 0.0, 0.0,
        		GridBagConstraints.EAST, GridBagConstraints.NONE,
        		new Insets(0, 5, 0, 0), 0, 0));

        // initially not visible, because starting tab is not service tab
        serviceActiveTimePanel.setVisible(false);
        
        // Add components to the panel
        bottomPanel.add( Box.createHorizontalStrut( 30 ) );
        bottomPanel.add( reconnectButton );
        bottomPanel.add( Box.createHorizontalStrut( 30 ) );
        bottomPanel.add( doLoggingButton );
        bottomPanel.add( Box.createHorizontalStrut( 30 ) );
        bottomPanel.add( doTracingButton );
        bottomPanel.add( Box.createHorizontalStrut( 30 ) );
        bottomPanel.add( runUseCaseButton );
        bottomPanel.add( Box.createVerticalStrut( 40 ) );
        bottomPanel.add( Box.createHorizontalGlue() );
        bottomPanel.add( autoActiveCheckBox );
        bottomPanel.add( Box.createHorizontalStrut( 30 ) );
        bottomPanel.add( activateButton );
        bottomPanel.add( Box.createHorizontalStrut( 30 ) );
        bottomPanel.add( serviceActiveTimePanel );
        bottomPanel.add( Box.createHorizontalStrut( 30 ) );

        return bottomPanel;
    }

    
    /**
     * Initializes main window menus
     */
    private void initMenus() {
        JMenuBar menuBar = null;
        JMenu menu = null;
        JMenuItem menuItem = null;
        MenuItemListener actionListener = new MenuItemListener();
        
        //Create the menu bar.
        menuBar = new JMenuBar();

        //Build the FILE menu.
        menu = new JMenu( bundle.getString( "MENU_FILE" ) );
        menu.setMnemonic( KeyEvent.VK_F );
        menu.getAccessibleContext().setAccessibleDescription(
                "MENU_FILE_DESCRIPTION" );
        menuBar.add( menu );

        menuItem = new JMenuItem( bundle.getString( "MENU_OPEN" ) );
        menuItem.setMnemonic( KeyEvent.VK_O );
        menuItem.setAccelerator( KeyStroke.getKeyStroke( KeyEvent.VK_O,
                ActionEvent.CTRL_MASK ) );
        menuItem.setName( OPEN_NAME );
        menuItem.addActionListener( actionListener );
        menu.add( menuItem );

        menuItem = new JMenuItem( bundle.getString( "MENU_SAVE" ) );
        menuItem.setMnemonic( KeyEvent.VK_S );
        menuItem.setAccelerator( KeyStroke.getKeyStroke( KeyEvent.VK_S,
                ActionEvent.CTRL_MASK ) );
        menuItem.setName( SAVE_NAME );
        menuItem.addActionListener( actionListener );
        menu.add( menuItem );

        menuItem = new JMenuItem( bundle.getString( "MENU_SAVE_AS" ) );
        menuItem.setName( SAVE_AS_NAME );
        menuItem.addActionListener( actionListener );
        menu.add( menuItem );

        menu.addSeparator();
        
        menuItem = new JCheckBoxMenuItem( bundle.getString( "CONNECTION" ) );
        menuItem.setName( CONNECTION );
        menuItem.addActionListener( actionListener );
        menuItem.setSelected( true );
        menu.add( menuItem );
        connectionMenuItem = menuItem;
        
        menu.addSeparator();
        
        menuItem = new JMenuItem( bundle.getString( "MENU_EXIT" ) );
        menuItem.setMnemonic( KeyEvent.VK_X );
        menuItem.setName( EXIT_NAME );
        menuItem.addActionListener( actionListener );
        menu.add( menuItem );

        //Build the EDIT menu.
        menu = new JMenu( bundle.getString( "MENU_EDIT" ) );
        menu.setMnemonic( KeyEvent.VK_E );
        menu.getAccessibleContext().setAccessibleDescription(
                "MENU_EDIT_DESCRIPTION" );
        menuBar.add( menu );

        menuItem = new JMenuItem( bundle.getString( "MENU_EDIT_SETTINGS" ) );
        menuItem.setMnemonic( KeyEvent.VK_S );
        menuItem.setName( EDIT_SETTINGS_NAME );
        menuItem.addActionListener( actionListener );
        menu.add( menuItem );
    
        //Build the HELP menu.
        menu = new JMenu( bundle.getString( "MENU_HELP" ) );
        menu.setMnemonic( KeyEvent.VK_H );
        menu.getAccessibleContext().setAccessibleDescription(
                "MENU_HELP_DESCRIPTION" );
        menuBar.add( menu );

        menuItem = new JMenuItem( bundle.getString( "MENU_ABOUT" ) );
        menuItem.setMnemonic( KeyEvent.VK_A );
        menuItem.setName( ABOUT_NAME );
        menuItem.addActionListener( actionListener );
        menu.add( menuItem );

        this.setJMenuBar( menuBar );

        // Create the popup menu for the chart panel.
        popupMenu = new JPopupMenu();

        menuItem = new JMenuItem( bundle.getString( ADD_GRAPH ) );
        menuItem.setName( ADD_GRAPH );
        menuItem.addActionListener( actionListener );

        popupMenu.add( menuItem );
        popupMenu.addSeparator();

        menuItem = new JMenuItem( bundle.getString( SHOW_DEBUG_WINDOW ) );
        menuItem.setName( SHOW_DEBUG_WINDOW );
        menuItem.addActionListener( actionListener );

        popupMenu.add( menuItem );
    }
    
    
    /**
     * Initializes the handler for the board connection.
     */
    private void initConnectionHandler() {
        try {
            if ((System.getProperty("os.name").toString().equalsIgnoreCase("linux"))
            		&& System.getProperty("serial") != null) {
            	// linux + serial
                connectionHandler = new LinuxSerial(serialProperties.getString("PORT_LINUX"),
                		Integer.parseInt(serialProperties.getString("SPEED")), true);
                                
            } else if (System.getProperty("serial") != null) {
            	// windows + serial
                connectionHandler = new WindowsSerial(serialProperties.getString("PORT_LINUX"),
                		Integer.parseInt(serialProperties.getString("SPEED")), true);

            } else {
            	// tcp/ip
            	String tcpIp = null;
            	String tcpPort = null;
            	Integer tcpPortInt = null;
            	
				tcpIp = System.getProperty("ip");
				if (tcpIp == null) {
            		tcpIp = serialProperties.getString("TCP_IP");					
				}

				tcpPort = System.getProperty("port");
				if (tcpPort != null) {
					tcpPortInt = Integer.parseInt(tcpPort);
				} else {
            		tcpPortInt = Integer.parseInt(serialProperties.getString("TCP_PORT"));
				}
            	
                connectionHandler = new TCPClient(tcpIp, tcpPortInt);
            }
            
        } catch (NumberFormatException nfe) {
            nfe.printStackTrace();
        }
    }
    

    /**
     * Creates a window that contains the given chart panel.
     */
    private JInternalFrame createChartWindow(GraphPanel panel) {
        JInternalFrame jif = new JInternalFrame( "", true, true, true, true );
        //jif.setLocation((2%4)*140,(2/4)*180);
        jif.getContentPane().add(panel);
        jif.setTitle(panel.getChartModel().getTitle());
        jif.setToolTipText(panel.getChartModel().getTitle());
        jif.setFrameIcon(logoIcon);
        jif.pack();
        jif.setVisible(true);
        return jif;
    }
    
    
    /**
     * Creates a window for showing debug messages.
     */
    private void createDebugWindow( int xLocation, int yLocation, int width,
            int height ) {
        debugTextArea = new JTextArea( 20, 40 );

        debugTextArea.setEditable( false );
        debugTextArea.setAutoscrolls( true );

        JScrollPane jsp = new JScrollPane( debugTextArea );
        jsp.setPreferredSize( new Dimension( 120, 140 ) );
        jsp.setAutoscrolls( true );

        debugWindow = new JInternalFrame( bundle
                .getString( "DEBUG_WINDOW_TITLE" ), true, true, true, true );

        debugWindow.getContentPane().add( jsp );
        debugWindow.pack();

        debugWindow.setSize( width, height );
        debugWindow.setLocation( xLocation, yLocation );

        chartPanel.add( debugWindow );
    }

    
    /**
     *  Shows the debug window in the chart panel
     */
    public void showDebugWindow(int xLocation, int yLocation, int width,
            int height) {
        if (debugWindow == null) {
            createDebugWindow(xLocation, yLocation, width, height);
        }
        
        // Check if the debug window exists in the chart panel
        Component[] compArrray = chartPanel.getComponents();

        boolean contain = false;
        for (int i = 0; i < compArrray.length; ++i) {
            if (compArrray[i].equals(debugWindow)) {
                contain = true;
            }
        }

        if (!contain) {
            chartPanel.add(debugWindow);
        }

        debugWindow.setVisible(true);
    }

    
    /**
     * Checks if current service model contains services
     * and sets service related tabs visible accordingly.
     */
    protected void checkServiceExistence() {
    	Model sModel = configuration.getServiceModel();
    	
    	if (sModel.getChildren(sModel.getModelRoot()).size() == 0) {
    		// No services: disable service tabs
    		tabbedPane.setEnabledAt(tabbedPane.indexOfComponent(servicePanel), false);
    		tabbedPane.setEnabledAt(tabbedPane.indexOfComponent(serviceReportPanel), false);
    		
    		// If service tab selected, change to control tab
    		Component comp = tabbedPane.getSelectedComponent();
    		if (comp == servicePanel) {
    			tabbedPane.setSelectedComponent(controlPanel);
    		}
    	} else {
    		tabbedPane.setEnabledAt(tabbedPane.indexOfComponent(servicePanel), true);
    		tabbedPane.setEnabledAt(tabbedPane.indexOfComponent(serviceReportPanel), true);
    	}
    	
    	validate();
    	repaint();
    }
    
    
    /**
     * @return The container panel for charts
     */
    public JDesktopPane getChartContainerPanel() {
    	return chartPanel;
    }
    
    
    /**
     * @return The container panel for charts
     */
    public List<GraphPanel> getChartPanels() {
    	return new ArrayList<GraphPanel>(chartPanels.values());
    }

    
    /**
     * @return The debug window
     */
    public JInternalFrame getDebugWindow() {
    	return debugWindow;
    }

    
    /**
     * @return The configuration that is in use.
     */
    public Configuration getConfiguration() {
    	return configuration;
    }
    
        
    /**
     * Sets the auto activation on/off.
     */
    public void setAutoActive(boolean autoActive) {
    	autoActiveCheckBox.setSelected(autoActive);
    }
    
    
    /**
     * @return True if auto activation is on.
     */
    public boolean isAutoActive() {
        return autoActiveCheckBox.isSelected();
    }
    
    
    /**
	 * @return Returns the signals panel.
	 */
	public SignalsPanel getSignalsPanel() {
		return signalsPanel;
	}


	/**
	 * @return Returns the pe report panel.
	 */
	public ReportPanel getPeReportPanel() {
		return peReportPanel;
	}
	
	/**
	 * @return Returns the processes panel.
	 */
	public ProcessesPanel getProcessesPanel() {
		return processesPanel;
	}

	/**
	 * @return Returns the service panel.
	 */
	public ServiceGraphPanel getServicePanel() {
		return servicePanel;
	}

	
	/**
     * @return The handler for the board connection.
     */
    public Handler getConnectionHandler() {
    	return connectionHandler;
    }

    
    /**
     * Sets process dragging on/off.
     */
    public void setDraggedProcess(ProcessIcon process) {
    	if (draggedProcess != null) {
    		draggedProcess.removeMouseListener(dragListener);
    		draggedProcess.removeMouseMotionListener(dragListener);
    	}
        draggedProcess = process;
    	
        if (draggedProcess == null) {
        	return;
        }
        
        Point pLoc = new Point(0,0);
        
        // Calculate the location of the process in main window's
        // coordinate space
        Point convertedLoc = SwingUtilities.convertPoint(draggedProcess, pLoc, this);
        setProcessDraggingPoint(new Point(convertedLoc.x, convertedLoc.y));

        draggedProcess.addMouseListener(dragListener);
        draggedProcess.addMouseMotionListener(dragListener);
    }

    
    /**
     * Sets the point where the dragged process is.
     */
    protected void setProcessDraggingPoint(Point p) {
        dragPoint = p;
    }

    
    /**
     * Sends the mapping.
     */
    public void sendMappingString() {
    	if (activatingProcesses) {
    		return;
    	}
        try {
            if ( connectionHandler != null ) {
                connectionHandler.writeToDestination(UIConfigIO.createMappingXMLString(configuration));
            }
        } catch (Exception e1) {
            e1.printStackTrace();
        }
    }
    
    
    /**
     * Returns a chart panel.
     * @param peId    Id of the PE, or null if a non-PE chart.
     * @param chartId Id of the chart.
     * @return        The chart, or null if not found.
     */
    public GraphPanel getChartPanel(String peId, String chartId) {
    	GraphPanel panel = null;
    	if (peId == null) {
    		panel = chartPanels.get(chartId);
    	} else {
    		// Find the pe chart panel
    		PePanel pePanel = controlPanel.getPePanel(peId);
    		if (pePanel != null) {
    			panel = pePanel.getChartPanel(chartId);
    		}
    	}
    	
    	return panel;
    }
    
    
    /**
     * Adds new value to chart.
     * @param peId    Id of the PE, or null if a non-PE chart.
     * @param chartId Id of the chart.
     * @param value The new value.
     */
    public void addValueToChart(String peId, String chartId, double value) {
    	GraphPanel panel = getChartPanel(peId, chartId);
        if (panel != null) {
        	panel.addData(value);
        }
    }
    
    
    /**
     * Adds a chart to chart container panel with given location
     * and size, and makes the chart visible.
     */
    public void addChart(GraphPanel panel,
    	int xLocation, int yLocation, int width, int height) {
    	addChart(panel, xLocation, yLocation, width, height, true);
    }
    
    
    /**
     * Adds a chart to chart container panel with given location
     * and size.
     */
    public void addChart(GraphPanel panel,
    	int xLocation, int yLocation, int width, int height, boolean showIt) {
    	String id = panel.getChartModel().getId();
    	GraphPanel existingPanel = chartPanels.get(id);
        JInternalFrame iFrame = null;
    	
    	if (existingPanel == null) {
    		chartPanels.put(id, panel);
            iFrame = createChartWindow(panel);
            iFrame.setLocation(xLocation, yLocation);
            iFrame.setSize(width, height);
            iFrame.addInternalFrameListener(new MyInternalFrameListener(id));
            chartPanel.add(iFrame);
            
    	} else {
    		JInternalFrame[] iFrames = chartPanel.getAllFrames();
    		boolean found = false;
    		for (int i=0; i<iFrames.length && !found; ++i) {
    			JInternalFrame f = iFrames[i];
    			
    			for (int j=0; j<f.getComponentCount(); ++j) {
    				if (SwingUtilities.isDescendingFrom(existingPanel, f.getComponent(j))) {
    					iFrame = f;
    					found = true;
    					break;
    				}
    			}
    		}
    		
    		if (!found) {
    			// shouldn't happen...anymore :)
    			return;
    		}
    	}

        iFrame.setVisible(true);

        try {
            iFrame.setSelected(true);    
        } catch (PropertyVetoException e) {
            // do nothing
        }

        chartPanel.moveToFront(iFrame);
        if (showIt) {
        	tabbedPane.setSelectedIndex(tabbedPane.indexOfComponent(chartPanel));
        }
        repaint();
    }

    
    /**
     * Removes all charts.
     */
    public void removeCharts() {
    	chartPanel.removeAll();
    	chartPanels.clear();
    	repaint();
    }
    
    public void setSimulationTime(long time) {
    	GraphPanel.setSimulationTime(time);
    }
    
    public void configurationChanged() {
    	Set<Map.Entry<String, GraphPanel>> chartPanelSet = chartPanels.entrySet();
        for (Map.Entry<String, GraphPanel> chartPanelEntry : chartPanelSet) {
        	chartPanelEntry.getValue().addMarker();
        }
    }
    
    /**
     * Adds a debug message.
     * @param message
     */
    public void addDebugMessage(String message) {
        if ( debugTextArea == null ) {
            createDebugWindow( 0, 0, 200, 300 );
        }

        debugTextArea.append( new java.text.SimpleDateFormat(
                "dd.MM.yyyy, HH:mm:ss" ).format( new java.util.Date( System
                .currentTimeMillis() ) )
                + " " + message + "\n" );

        debugTextArea.setCaretPosition(debugTextArea.getText().length());
    }

    
    /**
     * Stops chart data logging
     * @param errorMsg The message to be shown if logging ended due to an error.
     */
    public void stopLogging(String errorMsg) {
		if (currentLogDir == null) {
			// logging hasn't started
			return;
		}
		
		ChartDataLogger.getInstance().stopLogging();
		
		if (errorMsg == null) {
			JOptionPane.showMessageDialog(MainWindow.getInstance(),
					bundle.getString("LOGGING_ENDED_MSG")+" "+currentLogDir.getAbsolutePath(),
					bundle.getString("LOGGING_ENDED_TITLE"), JOptionPane.INFORMATION_MESSAGE);
		} else {
			JOptionPane.showMessageDialog(MainWindow.getInstance(),
					bundle.getString("LOGGING_ERROR_MSG")+" "+errorMsg,
					bundle.getString("LOGGING_ENDED_TITLE"), JOptionPane.ERROR_MESSAGE);
		}
		currentLogDir = null;
		doLoggingButton.setSelected(false);
    }
    
    
    /**
     * Starts execution trace data logging
     */
    public boolean startTracing() {
    	try {
			String traceParentDir = SettingsDialog.getInstance().getTraceDirectoryPath();
			TraceDataLogger.getInstance().setTraceDirectory(new File(traceParentDir));
			currentTraceDir = TraceDataLogger.getInstance().startTracing();
			return true;
		} catch (IOException ioe) {
			JOptionPane.showMessageDialog(MainWindow.getInstance(),
					bundle.getString("TRACING_START_ERROR_MSG")+" "+ioe.getMessage(),
					null, JOptionPane.ERROR_MESSAGE);
			return false;
		}
    }
    
    
    /**
     * Stops execution trace data logging
     * @param errorMsg The message to be shown if logging ended due to an error.
     */
    public void stopTracing(String errorMsg) {
		if (currentTraceDir == null) {
			// tracing hasn't started
			return;
		}
		
		TraceDataLogger.getInstance().stopTracing();
		
		if (errorMsg == null) {
			JOptionPane.showMessageDialog(MainWindow.getInstance(),
					bundle.getString("TRACING_ENDED_MSG")+" "+currentTraceDir.getAbsolutePath(),
					bundle.getString("TRACING_ENDED_TITLE"), JOptionPane.INFORMATION_MESSAGE);
		} else {
			JOptionPane.showMessageDialog(MainWindow.getInstance(),
					bundle.getString("TRACING_ERROR_MSG")+" "+errorMsg,
					bundle.getString("TRACING_ENDED_TITLE"), JOptionPane.ERROR_MESSAGE);
		}
		currentTraceDir = null;
		doTracingButton.setSelected(false);
    }

    
    public void paint(Graphics g) {
        super.paint(g);

        // If a process is being dragged, draw the shadow icon
        if (draggedProcess != null) {
            Graphics2D g2 = (Graphics2D)g.create();    
            g2.setColor(ProcessIcon.MOVE_COLOR);

            if (ProcessIcon.SQUARE_SHAPE) {
                g2.fillRect(dragPoint.x, dragPoint.y,
                		ProcessIcon.ICON_WIDTH, ProcessIcon.ICON_HEIGHT);

            } else {
                g2.fillOval(dragPoint.x, dragPoint.y,
                		ProcessIcon.ICON_WIDTH, ProcessIcon.ICON_HEIGHT);
            }

            g2.dispose(); 
        }
    }

    
    /**
     * Listener for changes in activated tab.
     */
    private class MyTabChangeListener implements ChangeListener {
    	public void stateChanged(ChangeEvent e) {
    		Component selected = tabbedPane.getSelectedComponent();
    		
    		if (controlPanel != null
    				&& activateButton != null
    				&& autoActiveCheckBox != null
    				&& serviceActiveTimePanel != null) {
    			
    			// set visibility of process activation selections
    			if (selected == controlPanel) {
    				activateButton.setVisible(true);
    				autoActiveCheckBox.setVisible(true);

    			} else {
    				activateButton.setVisible(false);
    				autoActiveCheckBox.setVisible(false);    			
    			}
    			
    			// set visibility of service activation time selection
    			if (selected == servicePanel) {
    				serviceActiveTimePanel.setVisible(true);

    			} else {
    				serviceActiveTimePanel.setVisible(false);
    			}
    		}
    	}
    }
    
    
    /**
     * Listener for process drag events
     */
    private class MyProcessDragListener extends MouseInputAdapter {
    	public void mouseReleased(MouseEvent e) {
    		setDraggedProcess(null);
    		self.repaint();
    	}
    	public void mouseDragged(MouseEvent e) {
    		Point p = e.getPoint();
    		Point offset = draggedProcess.getDragOffset();
    		Point shadowIconOrigin = new Point(p.x-offset.x, p.y-offset.y);
    		
    		// Convert the icon origin to main window coordinate space
    		Point convertedOrigin = SwingUtilities.convertPoint(draggedProcess, shadowIconOrigin, self);

    		self.setProcessDraggingPoint(convertedOrigin);
    		self.repaint();
    	}
    }
    
    
    /**
     * Listener for MouseEvents from chart panel
     */
    private class MyMouseListener extends MouseAdapter {
        public void mouseClicked(MouseEvent e) {
            if (SwingUtilities.isRightMouseButton(e)) {
                popupMenu.show( e.getComponent(), e.getX(), e.getY() );
            }
        }
    }

    
    /**
     * Listener for ActionEvents.
     */
    private class MyActionListener implements ActionListener {
    	public void actionPerformed(ActionEvent e) {
    		String cmd = e.getActionCommand();
    		
    		if (cmd.equals(RECONNECT_CMD)) {
				if (connectionHandler != null) {
					connectionHandler.reconnect();
					connectionMenuItem.setSelected(true);
				}
				
    		} else if (cmd.equals(RUN_USECASE_CMD)) {
    			// Ask device to run use case
    			try {
    				startTracing();
    				connectionHandler.writeToDestination("<run_usecase>");
    			} catch (IOException exception) {
    				// TODO Auto-generated catch block
    				exception.printStackTrace();
    			}
    		
			} else if (cmd.equals(ACTIVATE_PROCESSES_CMD)) {
				activatingProcesses = true;
				controlPanel.commitProcessMoves();
				activatingProcesses = false;
	            sendMappingString();
			}
    	}
    }
    
    
    /**
     * Listener for ItemEvents from doLoggingButton
     */
    private class MyLoggingButtonListener implements ItemListener {
    	private boolean ignoreChange = false;

    	public void itemStateChanged(ItemEvent e) {
    		if (!ignoreChange) {
    			if (e.getStateChange() == ItemEvent.SELECTED) {
    				try {
    					String logParentDir = SettingsDialog.getInstance().getLogDirectoryPath();
    					ChartDataLogger.getInstance().setLogDirectory(new File(logParentDir));
    					currentLogDir = ChartDataLogger.getInstance().startLogging();
    				} catch (IOException ioe) {
    					Object source = e.getSource();
    					if (source instanceof JToggleButton) {
    						JToggleButton tButton = (JToggleButton) source;
    						ignoreChange = true;
    						tButton.setSelected(false);
    						ignoreChange = false;
    					}
    					JOptionPane.showMessageDialog(MainWindow.getInstance(),
    							bundle.getString("LOGGING_START_ERROR_MSG")+" "+ioe.getMessage(),
    							null, JOptionPane.ERROR_MESSAGE);
    				}
    			} else {
    				stopLogging(null);
    			}
    		}
    	}
    }
    
    
    /**
     * Listener for ItemEvents from doTracingButton
     */
    private class MyTracingButtonListener implements ItemListener {
    	private boolean ignoreChange = false;

    	public void itemStateChanged(ItemEvent e) {
    		boolean ret_val;
    		
    		if (!ignoreChange) {
    			if (e.getStateChange() == ItemEvent.SELECTED) {
    				ret_val = startTracing();
    				if (ret_val == false) {
    					Object source = e.getSource();
    					if (source instanceof JToggleButton) {
    						JToggleButton tButton = (JToggleButton) source;
    						ignoreChange = true;
    						tButton.setSelected(false);
    						ignoreChange = false;
    					}
    				}
    			} else {
    				stopTracing(null);
    			}
    		}
    	}
    }
    
    
    /**
     * Listener for events from chart windows.
     */
    private class MyInternalFrameListener extends InternalFrameAdapter {
    	private String chartId;
    	public MyInternalFrameListener(String chartId) {
    		this.chartId = chartId;
    	}
    	public void internalFrameClosed(InternalFrameEvent e) {
    		chartPanels.remove(chartId);
    	}
    }
    
    
    /**
     * Listener for service model events.
     */
    private class MyServiceModelListener implements ModelListener {
    	public void nodeInserted(ModelEvent e) {
    		checkServiceExistence();
    	}
    	public void nodeRemoved(ModelEvent e) {
    		checkServiceExistence();
    	}
    	public void nodeMoved(ModelEvent e) {
    		checkServiceExistence();
    	}
    	public void structureChanged(ModelEvent e) {
    		checkServiceExistence();
    	}
    }
    
    /**
     * Listener for Window changes
     * 
     * @author perttul5
     *
     */
    public class MyWindowListener implements WindowListener {

    	@Override
    	public void windowActivated(WindowEvent e) {
    		// TODO Auto-generated method stub
    		
    	}

    	@Override
    	public void windowClosed(WindowEvent e) {
    		connectionHandler.disconnect();
    		System.exit(0);
    	}

    	@Override
    	public void windowClosing(WindowEvent e) {
    		connectionHandler.disconnect();
    		System.exit(0);
    	}

    	@Override
    	public void windowDeactivated(WindowEvent e) {
    		// TODO Auto-generated method stub
    		
    	}

    	@Override
    	public void windowDeiconified(WindowEvent e) {
    		// TODO Auto-generated method stub
    		
    	}

    	@Override
    	public void windowIconified(WindowEvent e) {
    		// TODO Auto-generated method stub
    		
    	}

    	@Override
    	public void windowOpened(WindowEvent e) {
    		// TODO Auto-generated method stub
    		
    	}
    	
    }
    
}
