/**
 * CVS: $Id: ServerApp.java 1399 2010-08-26 13:56:45Z lehton87 $
 * 
 * File:    ServerApp.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 9.11.2006
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.tcp.test;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.ServerSocket;
import java.net.Socket;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;


/**
 * Application for testing messaging through TCP socket.
 */
public class ServerApp extends JFrame {
	protected ServerApp self_;

	protected ServerSocket sSocket_;
	protected Socket socket_;
	protected BufferedReader bufferedReader_;
	protected int receivedMsgsCount = 0;
	
	private JTextArea textArea_;
	private JPanel connectionPanel_;
	private JPanel confPanel_;
	private JPanel measPanel_;
	private JPanel signalPanel_;
	private JPanel execPanel_;
	private JPanel tracePanel_;

	// graph measurement fields
	protected JTextField graphIdField_;
	protected JTextField cpuIdField_;
	protected JTextField measValueField_;

	// connection fields
	protected JTextField tcpPortField_;

	// configuration fields
	protected JTextField cpuNumField_;
	protected JTextField accNumField_;
	protected JTextField threadNumField_;
	protected JTextField procNumField_;
	protected JTextField serviceNumField_;
	protected JTextField serviceProcNumField;
	protected JCheckBox varNumOfThreads;

	// signal fields
	protected JTextField sigSourceField_;
	protected JTextField sigDestField_;
	protected JTextField sigValueField_;

	// execution fields
	protected JTextField execPidField_;
	protected JTextField execCountField_;
	protected JTextField execTimeField_;
	protected JTextField exec_lct_;
	protected JTextField exec_lcb_;
	protected JTextField exec_rct_;
	protected JTextField exec_rcb_;
	protected JTextField execSigQueue_;
	protected JTextField execLatency_;
	protected JTextField execRespTime_;
	
	// execution trace fields
	protected JTextField tracePartField_;
	protected JTextField traceStartStateField_;
	protected JTextField traceTriggerSignalField_;
	protected JTextField traceOutputSignalField_;
	protected JTextField traceEndStateField_;

	private static final int TEXT_ROWS = 15;
	private static final int TEXT_COLUMNS = 30;
	private static final int TEXTFIELD_COLUMNS = 15;
	
	private static final String SEND_CONF_CMD = "sendConfiguration";
	private static final String SEND_MEAS_CMD = "sendMeasurement";
	private static final String SEND_SIGNAL_CMD = "sendSignal";
	private static final String SEND_EXEC_VALS_CMD = "sendExecutionValues";
	private static final String SEND_EXEC_TRACE_CMD = "sendExecutionTrace";
	private static final String CONNECT_CMD = "connect";
	private static final String EXIT_CMD = "exit";

	private static final String DEFAULT_PORT = "9990";

	/**
	 * @param args
	 */
	public static void main(String[] args) {
        System.out.println("Starting server");

        String lafClass = UIManager.getSystemLookAndFeelClassName();
		try {
			UIManager.setLookAndFeel(lafClass);
		}
		catch(Exception e) {
			System.out.println("Couldn't set look and feel: "+e.getMessage());
		}
		
        showAppFrame(new ServerApp(), false);
	}
	
	
	/**
	 * Shows the application frame given as an argument.
	 * For thread-safety, the task is done by the 
	 * event-dispatching thread.
	 * 
	 * @param frame      Frame to be shown
	 * @param maximized  If true, frame is shown in maximized state
	 */
	private static void showAppFrame(final JFrame frame, final boolean maximized) {
		try {
			SwingUtilities.invokeAndWait(new Runnable() {
				public void run() {
					frame.pack();
					frame.setVisible(true);
					if (maximized) {
						frame.setExtendedState(frame.getExtendedState() | JFrame.MAXIMIZED_BOTH);
					}
				}
			});
		}
		catch (Exception e) {
			// Empty
		}
	}

	
	public ServerApp() {
		self_ = this;
		setDefaultCloseOperation(EXIT_ON_CLOSE);
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				try {
					closeSocket();
				} catch (Exception ex) {
					// don't mind
				}
			}
		});
		
		init();
	}

	private void init() {
		MyActionListener actionListener = new MyActionListener();
		
		// Create menu
		JMenuBar mainMenu = new JMenuBar();

		JMenuItem exitItem = new JMenuItem("Exit");
		exitItem.setActionCommand(EXIT_CMD);
		exitItem.addActionListener(actionListener);
		mainMenu.add(exitItem);
		setJMenuBar(mainMenu);
		
		setLayout(new BorderLayout());
		JTabbedPane tabbedPane = new JTabbedPane();
		textArea_ = new JTextArea(TEXT_ROWS, TEXT_COLUMNS);
		
		add(tabbedPane, BorderLayout.NORTH);
		add(new JScrollPane(textArea_, ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS,
				ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS), BorderLayout.CENTER);
		
		connectionPanel_ = new JPanel();
		confPanel_ = new JPanel();
		measPanel_ = new JPanel();
		signalPanel_ = new JPanel();
		execPanel_ = new JPanel();
		tracePanel_ = new JPanel();
		
		tabbedPane.add(connectionPanel_, "Connection");
		tabbedPane.add(confPanel_, "Configuration");
		tabbedPane.add(measPanel_, "Measurement");
		tabbedPane.add(signalPanel_, "Signals");
		tabbedPane.add(execPanel_, "Execution values");
		tabbedPane.add(tracePanel_, "Execution trace");
		
		
		// Create connection panel
		connectionPanel_.setLayout(new GridBagLayout());
		tcpPortField_ = new JTextField(30);
		tcpPortField_.setText(DEFAULT_PORT);
		
		JLabel[] connLabels = {new JLabel("TCP port:")};
		JTextField[] connFields = {tcpPortField_};
		
		for (int i=0; i<connLabels.length; ++i) {
			connectionPanel_.add(connLabels[i], new GridBagConstraints(
					0, i, 1, 1, 0.0, 0.0,
					GridBagConstraints.NORTHWEST,
					GridBagConstraints.NONE,
					new Insets(2, 2, 2, 2),
					0, 0));

			connectionPanel_.add(connFields[i], new GridBagConstraints(
					1, i, 1, 1, 0.0, 0.0,
					GridBagConstraints.NORTHWEST,
					GridBagConstraints.NONE,
					new Insets(2, 2, 2, 2),
					0, 0));			
		}
		
		JButton connectButton = new JButton("(Re)connect");
		connectButton.setActionCommand(CONNECT_CMD);
		connectButton.addActionListener(actionListener);
		connectionPanel_.add(connectButton, new GridBagConstraints(
				1, connLabels.length, 1, 1, 0.0, 0.0,
				GridBagConstraints.NORTHWEST,
				GridBagConstraints.NONE,
				new Insets(2, 2, 2, 2),
				0, 0));

		
		// Create configuration panel
		confPanel_.setLayout(new GridBagLayout());
		cpuNumField_ = new JTextField(TEXTFIELD_COLUMNS);
		accNumField_ = new JTextField(TEXTFIELD_COLUMNS);
		threadNumField_ = new JTextField(TEXTFIELD_COLUMNS);
		procNumField_ = new JTextField(TEXTFIELD_COLUMNS);
		serviceNumField_ = new JTextField(TEXTFIELD_COLUMNS);
		serviceProcNumField = new JTextField(TEXTFIELD_COLUMNS);
		varNumOfThreads = new JCheckBox("Variable num. of threads");
		
		JLabel[] confLabels = {
				new JLabel("CPUs:"),
				new JLabel("ACCs:"),
				new JLabel("Threads/PE:"),
				new JLabel("Processes/Thread:"),
				new JLabel("Services:"),
				new JLabel("Processes/Service:")};
		JTextField[] confFields = {
				cpuNumField_,
				accNumField_,
				threadNumField_,
				procNumField_,
				serviceNumField_,
				serviceProcNumField};
		
		for (int i=0; i<confLabels.length; ++i) {
			confPanel_.add(confLabels[i], new GridBagConstraints(
					0, i, 1, 1, 0.0, 0.0,
					GridBagConstraints.NORTHWEST,
					GridBagConstraints.NONE,
					new Insets(2, 2, 2, 2),
					0, 0));

			confPanel_.add(confFields[i], new GridBagConstraints(
					1, i, 1, 1, 0.0, 0.0,
					GridBagConstraints.NORTHWEST,
					GridBagConstraints.NONE,
					new Insets(2, 2, 2, 2),
					0, 0));			
		}

		JButton sendConfButton = new JButton("Send");
		sendConfButton.setActionCommand(SEND_CONF_CMD);
		sendConfButton.addActionListener(actionListener);
		confPanel_.add(sendConfButton, new GridBagConstraints(
				0, confLabels.length, 1, 1, 0.0, 0.0,
				GridBagConstraints.NORTHWEST,
				GridBagConstraints.NONE,
				new Insets(2, 2, 2, 2),
				0, 0));
		confPanel_.add(varNumOfThreads, new GridBagConstraints(
				1, confLabels.length, 1, 1, 0.0, 0.0,
				GridBagConstraints.NORTHWEST,
				GridBagConstraints.NONE,
				new Insets(2, 2, 2, 2),
				0, 0));


		// Create measurement panel
		measPanel_.setLayout(new GridBagLayout());
		graphIdField_ = new JTextField(TEXTFIELD_COLUMNS);
		cpuIdField_ = new JTextField(TEXTFIELD_COLUMNS);
		measValueField_ = new JTextField(TEXTFIELD_COLUMNS);
		
		JLabel[] measLabels = {new JLabel("Graph ID:"), new JLabel("CPU ID:"), new JLabel("Measurement value:")};
		JTextField[] measFields = {graphIdField_, cpuIdField_, measValueField_};

		for (int i=0; i<measLabels.length; ++i) {
			measPanel_.add(measLabels[i], new GridBagConstraints(
					0, i, 1, 1, 0.0, 0.0,
					GridBagConstraints.NORTHWEST,
					GridBagConstraints.NONE,
					new Insets(2, 2, 2, 2),
					0, 0));

			measPanel_.add(measFields[i], new GridBagConstraints(
					1, i, 1, 1, 0.0, 0.0,
					GridBagConstraints.NORTHWEST,
					GridBagConstraints.NONE,
					new Insets(2, 2, 2, 2),
					0, 0));			
		}

		JButton sendMeasButton = new JButton("Send");
		sendMeasButton.setActionCommand(SEND_MEAS_CMD);
		sendMeasButton.addActionListener(actionListener);
		measPanel_.add(sendMeasButton, new GridBagConstraints(
				1, measLabels.length, 1, 1, 0.0, 0.0,
				GridBagConstraints.NORTHWEST,
				GridBagConstraints.NONE,
				new Insets(2, 2, 2, 2),
				0, 0));

		
		// Create signals panel
		signalPanel_.setLayout(new GridBagLayout());
		sigDestField_ = new JTextField(TEXTFIELD_COLUMNS);
		sigSourceField_ = new JTextField(TEXTFIELD_COLUMNS);
		sigValueField_ = new JTextField(TEXTFIELD_COLUMNS);

		JLabel[] sigLabels = {new JLabel("Source pid:"), new JLabel("Destination pid:"), new JLabel("Signal value:")};
		JTextField[] sigFields = {sigSourceField_, sigDestField_, sigValueField_};

		for (int i=0; i<sigLabels.length; ++i) {
			signalPanel_.add(sigLabels[i], new GridBagConstraints(
					0, i, 1, 1, 0.0, 0.0,
					GridBagConstraints.NORTHWEST,
					GridBagConstraints.NONE,
					new Insets(2, 2, 2, 2),
					0, 0));

			signalPanel_.add(sigFields[i], new GridBagConstraints(
					1, i, 1, 1, 0.0, 0.0,
					GridBagConstraints.NORTHWEST,
					GridBagConstraints.NONE,
					new Insets(2, 2, 2, 2),
					0, 0));			
		}


		JButton sendSigButton = new JButton("Send");
		sendSigButton.setActionCommand(SEND_SIGNAL_CMD);
		sendSigButton.addActionListener(actionListener);
		signalPanel_.add(sendSigButton, new GridBagConstraints(
				1, sigLabels.length, 1, 1, 0.0, 0.0,
				GridBagConstraints.NORTHWEST,
				GridBagConstraints.NONE,
				new Insets(2, 2, 2, 2),
				0, 0));
		
		
		// Create execution values panel
		execPanel_.setLayout(new GridBagLayout());
		execPidField_ = new JTextField(TEXTFIELD_COLUMNS);
		execCountField_ = new JTextField(TEXTFIELD_COLUMNS);
		execTimeField_ = new JTextField(TEXTFIELD_COLUMNS);
		exec_lct_ = new JTextField(TEXTFIELD_COLUMNS);
		exec_lcb_ = new JTextField(TEXTFIELD_COLUMNS);
		exec_rct_ = new JTextField(TEXTFIELD_COLUMNS);
		exec_rcb_ = new JTextField(TEXTFIELD_COLUMNS);
		execSigQueue_ = new JTextField(TEXTFIELD_COLUMNS);
		execLatency_ = new JTextField(TEXTFIELD_COLUMNS);
		execRespTime_ = new JTextField(TEXTFIELD_COLUMNS);
		
		
/*		
		protected JTextField execPidField_;
		protected JTextField execCountField_;
		protected JTextField execTimeField_;
		protected JTextField exec_lct_;
		protected JTextField exec_lcb_;
		protected JTextField exec_rct_;
		protected JTextField exec_rcb_;
		protected JTextField execSigQueue_;
		protected JTextField execLatency_;
		protected JTextField execRespTime_;
*/
		
		JLabel[] execLabels = {
				new JLabel("Pid:"),
				new JLabel("Exec count:"),
				new JLabel("Avg Exec time:"),
				new JLabel("Local comm. time:"),
				new JLabel("Local comm. bytes:"),
				new JLabel("Remote comm. time:"),
				new JLabel("Remote comm. bytes:"),
				new JLabel("Signal queue:"),
				new JLabel("Latency:"),
				new JLabel("Response time:")
				};

		JTextField[] execFields = {
				execPidField_,
				execCountField_,
				execTimeField_,
				exec_lct_,
				exec_lcb_,
				exec_rct_,
				exec_rcb_,
				execSigQueue_,
				execLatency_,
				execRespTime_
				};

		for (int i=0; i<execLabels.length; ++i) {
			execPanel_.add(execLabels[i], new GridBagConstraints(
					0, i, 1, 1, 0.0, 0.0,
					GridBagConstraints.NORTHWEST,
					GridBagConstraints.NONE,
					new Insets(2, 2, 2, 2),
					0, 0));

			execPanel_.add(execFields[i], new GridBagConstraints(
					1, i, 1, 1, 0.0, 0.0,
					GridBagConstraints.NORTHWEST,
					GridBagConstraints.NONE,
					new Insets(2, 2, 2, 2),
					0, 0));			
		}

		JButton sendExecButton = new JButton("Send");
		sendExecButton.setActionCommand(SEND_EXEC_VALS_CMD);
		sendExecButton.addActionListener(actionListener);
		execPanel_.add(sendExecButton, new GridBagConstraints(
				1, execLabels.length, 1, 1, 0.0, 0.0,
				GridBagConstraints.NORTHWEST,
				GridBagConstraints.NONE,
				new Insets(2, 2, 2, 2),
				0, 0));
		
		
		// Create execution trace panel
		tracePanel_.setLayout(new GridBagLayout());
		tracePartField_ = new JTextField(TEXTFIELD_COLUMNS);
		traceStartStateField_ = new JTextField(TEXTFIELD_COLUMNS);
		traceTriggerSignalField_ = new JTextField(TEXTFIELD_COLUMNS);
		traceOutputSignalField_ = new JTextField(TEXTFIELD_COLUMNS);
		traceEndStateField_ = new JTextField(TEXTFIELD_COLUMNS);
		
		JLabel[] traceLabels = {new JLabel("Part:"), new JLabel("Start state:"), new JLabel("Trigger signal:"), new JLabel("Output signal:"), new JLabel("End state:")};
		JTextField[] traceFields = {tracePartField_, traceStartStateField_, traceTriggerSignalField_, traceOutputSignalField_, traceEndStateField_};

		for (int i=0; i<traceLabels.length; ++i) {
			tracePanel_.add(traceLabels[i], new GridBagConstraints(
					0, i, 1, 1, 0.0, 0.0,
					GridBagConstraints.NORTHWEST,
					GridBagConstraints.NONE,
					new Insets(2, 2, 2, 2),
					0, 0));

			tracePanel_.add(traceFields[i], new GridBagConstraints(
					1, i, 1, 1, 0.0, 0.0,
					GridBagConstraints.NORTHWEST,
					GridBagConstraints.NONE,
					new Insets(2, 2, 2, 2),
					0, 0));			
		}

		JButton sendTraceButton = new JButton("Send");
		sendTraceButton.setActionCommand(SEND_EXEC_TRACE_CMD);
		sendTraceButton.addActionListener(actionListener);
		tracePanel_.add(sendTraceButton, new GridBagConstraints(
				1, traceLabels.length, 1, 1, 0.0, 0.0,
				GridBagConstraints.NORTHWEST,
				GridBagConstraints.NONE,
				new Insets(2, 2, 2, 2),
				0, 0));
				
		setPreferredSize(new Dimension(600, 400));
	}
	
	
	protected void initSocket(int port) throws IOException {
		closeSocket();

		sSocket_ = new ServerSocket( port );
		printMsg( "- Waiting client...\n" );
		socket_ = sSocket_.accept();
        bufferedReader_ = new BufferedReader( new InputStreamReader( socket_.getInputStream() ) );
		printMsg( "- Client has connected!\n" );
		
		new TCPReader().start();
	}
	
	
	protected void closeSocket() throws IOException {
		if (socket_ != null) {
			socket_.close();
			socket_ = null;
		}
		if (sSocket_ != null) {
			sSocket_.close();
			sSocket_ = null;
		}
		bufferedReader_ = null;
	}
	
	
	protected void sendData(String data) throws IOException {
        PrintWriter out = new PrintWriter( socket_.getOutputStream(), true );
//        printMsg( "Sending: '" + data + "'\n" );
        out.print( data );
        out.println();		
	}

	protected String readData() throws IOException {
		String tmp = null;
		tmp = bufferedReader_.readLine();
		return tmp;
	}
	
	
	protected void printMsg(String msg) {
		if (textArea_ != null) {
			textArea_.append(msg);
			textArea_.scrollRectToVisible(new Rectangle(0, textArea_.getHeight(), 0, 0));
//			textArea_.repaint();
		} else {
			System.out.print(msg);
		}
	}
	
	protected String createMeasurementMsg() {
		StringBuilder sb = new StringBuilder();
	    sb.append("<measurement><graph id=\"");
	    sb.append(graphIdField_.getText());
	    sb.append("\" cpu_id=\"");
	    sb.append(cpuIdField_.getText());
	    sb.append("\" value=\"");
	    sb.append(measValueField_.getText());
	    sb.append("\"/></measurement>");
	    return sb.toString();
	}

	protected String createSignalMsg() {
		StringBuilder sb = new StringBuilder();
	    sb.append("<measurement><signal source_pid=\"");
	    sb.append(sigSourceField_.getText());
	    sb.append("\" dest_pid=\"");
	    sb.append(sigDestField_.getText());
	    sb.append("\" value=\"");
	    sb.append(sigValueField_.getText());
	    sb.append("\"/></measurement>");
	    return sb.toString();
	}

	protected String createExecMsg() {
		StringBuilder sb = new StringBuilder();
	    sb.append("<measurement><process_exec pid=\"");
	    sb.append(execPidField_.getText());
	    sb.append("\" count=\"");
	    sb.append(execCountField_.getText());
	    sb.append("\" exec_time=\"");
	    sb.append(execTimeField_.getText());

	    sb.append("\" local_comm_time=\"");
	    sb.append(exec_lct_.getText());
	    sb.append("\" local_comm_bytes=\"");
	    sb.append(exec_lcb_.getText());
	    sb.append("\" remote_comm_time=\"");
	    sb.append(exec_rct_.getText());
	    sb.append("\" remote_comm_bytes=\"");
	    sb.append(exec_rcb_.getText());

	    sb.append("\" signal_queue=\"");
	    sb.append(execSigQueue_.getText());
	    sb.append("\" latency=\"");
	    sb.append(execLatency_.getText());
	    sb.append("\" response_time=\"");
	    sb.append(execRespTime_.getText());
	    
	    sb.append("\"/></measurement>");
	    return sb.toString();
	}
	
	protected String createTraceMsg() {
		StringBuilder sb = new StringBuilder();
	    sb.append("<log><transition part=\"");
	    sb.append(tracePartField_.getText());
	    sb.append("\"><start state=\"");
	    sb.append(traceStartStateField_.getText());
	    sb.append("\" time=\"0\"/><trigger signal=\"");
	    sb.append(traceTriggerSignalField_.getText());
	    sb.append("\" sender=\"x\"/><output signal=\"");
	    sb.append(traceOutputSignalField_.getText());
	    sb.append("\" receiver=\"y\"/><end state=\"");
	    sb.append(traceEndStateField_.getText());
	    sb.append("\" time=\"0\"/></transition></log>");
	    return sb.toString();
	}

	protected String createConfMsg() {
		StringBuilder data = new StringBuilder();
		data.append("<configuration>");
		int peID = 0;
		int threadID = 0;
		int pID = 0;
		
		int cpuCount = Integer.parseInt(cpuNumField_.getText());
		int accCount = Integer.parseInt(accNumField_.getText());
		int threadCount = Integer.parseInt(threadNumField_.getText());
		int processCount = Integer.parseInt(procNumField_.getText());
		int serviceCount = Integer.parseInt(serviceNumField_.getText());
		int serviceProcCount = Integer.parseInt(serviceProcNumField.getText());

		int numOfThreads = threadCount;

		// for all cpus
		for (int i=0; i<cpuCount; ++i) {
			data.append("<cpu ");
			
			data.append("id=\"");
			data.append(peID);
			data.append("\" name=\"");
			data.append("CPU"+peID);
			data.append("\">");
			
			++peID;
			
			if (varNumOfThreads.isSelected()) {
				++numOfThreads;
				numOfThreads = (numOfThreads%threadCount)+1;
			}
						
			// for all threads
			for (int j=0; j<numOfThreads; ++j) {
				data.append("<thread id=\"");
				data.append(threadID);
				data.append("\" name=\"T");
				data.append(threadID);
				data.append("\" priority=\"");
				data.append(threadID);
				data.append("\">");
				
				++threadID;
				
				// for all processes
				for (int k=0; k<processCount; ++k) {
					data.append("<process id=\"");
					data.append(pID);
					data.append("\" name=\"");
					data.append("Process"+pID);
					data.append("\"/>");
					
					++pID;
				}
				
				data.append("</thread>");				
			}
			data.append("</cpu>");
		}

		// for all accs
		for (int i=0; i<accCount; ++i) {

			data.append("<accelerator ");
			
			data.append("id=\"");
			data.append(peID);
			data.append("\" name=\"");
			data.append("ACC"+peID);
			data.append("\">");
			
			++peID;

			if (varNumOfThreads.isSelected()) {
				++numOfThreads;
				numOfThreads = (numOfThreads%threadCount)+1;
			}

			// for all threads
			for (int j=0; j<numOfThreads; ++j) {
				data.append("<thread id=\"");
				data.append(threadID);
				data.append("\" name=\"T");
				data.append(threadID);
				data.append("\" priority=\"");
				data.append(threadID);
				data.append("\">");
				
				++threadID;
				
				// for all processes
				for (int k=0; k<processCount; ++k) {
					data.append("<process id=\"");
					data.append(pID);
					data.append("\" name=\"");
					data.append("Process"+pID);
					data.append("\"/>");
					
					++pID;
				}
				
				data.append("</thread>");				
			}
			
			data.append("</accelerator>");
		}
		
		if (serviceCount > 0) {
			pID = 0;
			
			data.append("<services>");
			//  for all services
			for (int i=0; i<serviceCount; ++i) {
				data.append("<service ");
				
				data.append("id=\"");
				data.append(i);
				data.append("\" name=\"");
				data.append("Service"+i);
				data.append("\">");
				
				// for all processes
				for (int j=0; j<serviceProcCount; ++j) {
					data.append("<process id=\"");
					data.append(pID);
					data.append("\" name=\"");
					data.append("Process"+pID);
					data.append("\"/>");
					
					++pID;
				}
				
				data.append("</service>");
			}				

			//  connect services in chain
			for (int i=0; i<serviceCount-1; ++i) {
				data.append("<connection ");
				data.append("id=\"");
				
				data.append(i);					
				data.append("\" src=\"");
				data.append(i);
				data.append("\" dest=\"");
				data.append(i+1);
				data.append("\"/>");
			}
			
			// connect first service to other services
			for (int i=2; i<serviceCount; ++i) {
				data.append("<connection ");
				data.append("id=\"");
				data.append(serviceCount - 2 + i);					
				data.append("\" src=\"");
				data.append("0");
				data.append("\" dest=\"");
				data.append(i);
				data.append("\"/>");
			}	

			data.append("</services>");
		}

	    data.append("</configuration>");
	    return data.toString();
	}
	
	
	protected String intendData(String data) {
		String multiline = data.replace(">", ">\n");
		return multiline;
	}
	
	
	
	
	private class MyActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			String cmd = e.getActionCommand();
			
			if (cmd.equals(SEND_MEAS_CMD)) {
				try {
					sendData(createMeasurementMsg());
					printMsg("- Measurement was sent!\n");
				} catch (IOException ioe) {
					printMsg("- Error when sending message: "+ioe.getMessage()+"\n");
				}

			} else if (cmd.equals(SEND_CONF_CMD)) {
				try {
					sendData(createConfMsg());
					printMsg("- Configuration was sent!\n");
				} catch (IOException ioe) {
					printMsg("- Error when sending message: "+ioe.getMessage()+"\n");
				} catch (NumberFormatException nfe) {
					printMsg("- Error when converting string to integer: "+nfe.getMessage()+"\n");					
				}
				
			} else if (cmd.equals(SEND_SIGNAL_CMD)) {
				try {
					sendData(createSignalMsg());
					printMsg("- Signal value was sent!\n");
				} catch (IOException ioe) {
					printMsg("- Error when sending message: "+ioe.getMessage()+"\n");
				}

			} else if (cmd.equals(SEND_EXEC_VALS_CMD)) {
				try {
					sendData(createExecMsg());
					printMsg("- Process exec. values were sent!\n");
				} catch (IOException ioe) {
					printMsg("- Error when sending message: "+ioe.getMessage()+"\n");
				}

			} else if (cmd.equals(SEND_EXEC_TRACE_CMD)) {
				try {
					sendData(createTraceMsg());
					printMsg("- Execution trace were sent!\n");
				} catch (IOException ioe) {
					printMsg("- Error when sending message: "+ioe.getMessage()+"\n");
				}
				
			} else if (cmd.equals(CONNECT_CMD)) {
				try {
					initSocket(Integer.parseInt(tcpPortField_.getText()));
				} catch (Exception ex) {
					printMsg("- Error when connecting: "+ex.getMessage()+"\n");
				}
				
			} else if (cmd.equals(EXIT_CMD)) {
				try {
					closeSocket();
				} catch (IOException ioe) {
					//
				}
				self_.dispose();
			}
		}
	}
	
	private class TCPReader extends Thread {
		public TCPReader() {
			super("TCP_reader_thread");
		}
		
		public void run() {
			while (true) {
				while (bufferedReader_ == null) {
            		try {
						sleep(500);
					} catch (InterruptedException e) {
						//
					}
				}
				
				try {
					String data = readData();
					if (data != null && !data.equals("")) {
//						print("- Received: "+intendData(data)+"\n");
						print("- Received msg #"+receivedMsgsCount+"\n");
						++receivedMsgsCount;
					}
				} catch (Exception ex) {
					print("- Error when reading data: "+ex.getMessage()+"\n");
					print("- TCPReader exits. Reconnect.\n");
					break;
				}
			}
		}
		
		private void print(final String msg) {
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					printMsg(msg);
				}
			});
		}
	}
}
