/**
 * File:    Service.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 26.2.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.data;

import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.tree.TreeNode;

import fi.cpu.event.ModelNodeEvent;
import fi.cpu.event.ModelNodeListener;


/**
 * Class for information about one service.
 */
public class Service extends ModelNode implements Comparable<Service> {
	public static final int DATA_CHANGED_EVENT = 1;

	private int id;
	private String name;
	private Point2D location;
	private int execCount;
	private double avgExecTime;
	private double totExecTime;
	private boolean definesLocation;
	private MyProcessNodeListener processListener = new MyProcessNodeListener();

	
	/**
	 * Createas a new Service.
	 * 
	 * @param id   The id of the service.
	 * @param name The name of the service.
	 */
	public Service(int id, String name) {
		super(name);
		this.id = id;
		this.name = name;
		this.location = new Point2D.Double(0, 0);
		
		definesLocation = false;
	}

	
	/**
	 * @return Returns the location.
	 */
	public Point2D getLocation() {
		return location;
	}

	
	/**
	 * Sets the location.
	 */
	public void setLocation(Point2D location) {
		this.location = location;
		definesLocation = true;
	}
	
	
	/**
	 * @return Returns the definesLocation.
	 */
	public boolean definesLocation() {
		return definesLocation;
	}


	/**
	 * @param definesLocation The definesLocation to set.
	 */
	public void setDefinesLocation(boolean definesLocation) {
		this.definesLocation = definesLocation;
	}


	/**
	 * @return Returns the id.
	 */
	public int getId() {
		return id;
	}

	
	/**
	 * @return Returns the name.
	 */
	public String getName() {
		return name;
	}
	
	
	/**
	 * Returns the average execution time.
	 */
	public double getAvgExecTime() {
		return avgExecTime;
	}


	/**
	 * Returns the execution count.
	 */
	public int getExecCount() {
		return execCount;
	}


	/**
	 * Returns the total execution time.
	 */
	public double getTotExecTime() {
		return totExecTime;
	}


	public String toString() {
		return getName();
	}
	
	
	/**
	 * Calculates execution values.
	 */
	protected void calcExecValues() {
		int eCount = 0;
		double avgETime = 0;
		double totETime = 0;

		List<Process> pList = new ArrayList<Process>();
		for (int i=0; i<getChildCount(); ++i) {
			TreeNode child = getChildAt(i);
			if (child instanceof Process) {
				Process p = (Process) child;
				pList.add(p);
			}
		}
		
		Iterator<Process> pIter = pList.iterator();
		
		// calculate sums from all processes
		while (pIter.hasNext()) {
			Process p = pIter.next();
			eCount += p.getExecCount();
			totETime += p.getTotExecTime();
		}

		// calculate the thread average exec time
		if (eCount > 0) {
			avgETime = totETime / eCount;
		}
		
		execCount = eCount;
		avgExecTime = avgETime;
		totExecTime = totETime;
		
		fireModelNodeChanged(new ModelNodeEvent(this, DATA_CHANGED_EVENT));
	}
	
	
	/**
	 * Implements the Compararable-interface
	 */
	public int compareTo(Service o) {
		return name.compareTo(o.name);
	}
	
	
	// Methods for tracking changes in this nodes children.

	/* (non-Javadoc)
	 * @see fi.cpu.data.ModelNode#childAdded(fi.cpu.data.ModelNode)
	 */
	public void childAdded(ModelNode child) {
		if (child instanceof Process) {
			Process p = (Process) child;
			p.addModelNodeListener(processListener);
		}
		calcExecValues();
	}

	/* (non-Javadoc)
	 * @see fi.cpu.data.ModelNode#childRemoved(fi.cpu.data.ModelNode)
	 */
	public void childRemoved(ModelNode child) {
		if (child instanceof Process) {
			Process p = (Process) child;
			p.removeModelNodeListener(processListener);
			calcExecValues();
		}
	}

	
	/**
	 * Listener for events from contained processes.
	 */
	private class MyProcessNodeListener implements ModelNodeListener {
		public void modelNodeChanged(ModelNodeEvent e) {
			int eventId = e.getEventId();
			if  (eventId == Process.DATA_CHANGED_EVENT) {
				calcExecValues();
			}
		}
	}
}
