/**
 * File:    Connection.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 26.2.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.data;

import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Class for information about one connection between services.
 */

public class Connection extends ModelNode {
    private static final char PID_DELIMETER = '#';
    
	private int id;
	private int srcService;
	private int destService;
	private boolean definesPoints;
	private List<Point2D> points;
    private Map<String, Integer> signalValues;
    private int signalValueSum;
	
	/**
	 * Createas a new Connection.
	 * 
	 * @param id          The id of the connection.
	 * @param srcService  The id of the source service.
	 * @param destService The id of the destination service
	 */
	public Connection(int id, int srcService, int destService) {
		super(""+id);
		this.id = id;
		this.srcService = srcService;
		this.destService = destService;
		
        points = new ArrayList<Point2D>();
		definesPoints = false;
        signalValues = new HashMap<String, Integer>();
        signalValueSum = 0;
	}

		
	/**
	 * @return Returns the id.
	 */
	public int getId() {
		return id;
	}


	/**
	 * @return Returns the source service id.
	 */
	public int getSrcService() {
		return srcService;
	}


	/**
	 * @return Returns the destination service id.
	 */
	public int getDestService() {
		return destService;
	}


	/**
	 * @return Returns the connection points.
	 */
	public List<Point2D> getPoints() {
		return points;
	}


	/**
	 * Sets the connection points.
	 */
	public void setPoints(List<Point2D> pointsList) {
		if (pointsList == null) {
			points.clear();
		} else {
			points = new ArrayList<Point2D>(pointsList);
		}
		definesPoints = true;
	}

	
	/**
	 * Adds a point to the connection.
	 */
	public void addPoint(Point2D p) {
		points.add(p);
		definesPoints = true;
	}
	
	
	/**
	 * @return Returns the definesPoints.
	 */
	public boolean definesPoints() {
		return definesPoints;
	}


	/**
	 * @param definesPoints The definesPoints to set.
	 */
	public void setDefinesPoints(boolean definesPoints) {
		this.definesPoints = definesPoints;
	}
    
    
    /**
     * Sets the value of a signal belonging to this connection.
     * 
     * @param srcPid  source process id
     * @param destPid destination process id
     * @param value   value of the signal
     */
    public void setSignalValue(int srcPid, int destPid, int value) {
        String key = ""+srcPid+PID_DELIMETER+destPid;
        if (value <= 0) {
            // no need to remember zero values
            signalValues.remove(key);
        } else {
            signalValues.put(key, new Integer(value));
        }
        calcSignalValueSum();
    }
    
    
    /**
     * Calculates the sum of all signal values.
     */
    private void calcSignalValueSum() {
        int sum = 0;
        Iterator<Integer> valueIter = signalValues.values().iterator();
        while (valueIter.hasNext()) {
            Integer val = valueIter.next();
            sum += val.intValue();
        }
        signalValueSum = sum;
    }
    
    
    /**
     * Returns the sum of all signals of this connection.
     * @return the sum
     */
    public int getSignalValuesSum() {
        return signalValueSum;
    }


	public String toString() {
		return ""+getId();
	}
}
