/**
 * File:    Configuration.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 5.3.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.data;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
 

/**
 * Class for configuration information
 */
public class Configuration {
	private Model peModel;
	private Model serviceModel;
	
	
	/**
	 * Creates a new Configuration. 
	 */
	public Configuration() {
		peModel = new Model();
		serviceModel = new Model();
	}
	
	
	/**
	 * @return Returns the peModel.
	 */
	public Model getPeModel() {
		return peModel;
	}

	
	/**
	 * @return Returns the serviceModel.
	 */
	public Model getServiceModel() {
		return serviceModel;
	}


	/**
	 * Initializes the peModel.
	 * @param newModel The model used in the initialization.
	 */
	public void initializePeModel(Model newModel) {
		peModel.enableEvents(false);
		
		List<ModelNode> oldPeNodes = peModel.getChildren(peModel.getModelRoot());
		Iterator<ModelNode> newPeIter = newModel.getChildren(newModel.getModelRoot()).iterator();

		// Clear pes from the old model
		ModelNode peModelRoot = peModel.getModelRoot();
		peModel.removeAllChildren(peModelRoot);
		
		// Add pes from the new model
		while (newPeIter.hasNext()) {
			ModelNode node = newPeIter.next();
			if (!(node instanceof ProcessingElement)) {
				continue;
			}
			ProcessingElement newPe = (ProcessingElement) node;

			// If the new model doesn't define pe charts, copy
			// old charts to new model.
			if (!newPe.definesCharts()) {
				// find corresponding old node
				ProcessingElement oldPe = null;
				for (int i=0; i<oldPeNodes.size(); ++i) {
					ProcessingElement tmpPe = (ProcessingElement)oldPeNodes.get(i);
					if (tmpPe.getName().equals(newPe.getName())
							&& tmpPe.getId().equals(newPe.getId())) {
						oldPe = tmpPe;
						break;
					}
				}
				
				if (oldPe != null) {
					// Copy charts
					Iterator<Chart> chartIter = oldPe.getCharts().iterator();
					while (chartIter.hasNext()) {
						Chart c = chartIter.next();
						newPe.setChart(c.getId(), c);
					}
					newPe.setVisibleChart(oldPe.getVisibleChart());
				}
				newPe.setDefinesCharts(true);
			}
			
			peModel.insertNode(newPe, peModelRoot);
		}
		
		peModel.enableEvents(true);
		peModel.modelStructureChanged(peModelRoot);
	}

	
	/**
	 * Initializes the serviceModel.
	 * @param newModel The model used in the initialization.
	 */
	public void initializeServiceModel(Model newModel) {
		serviceModel.enableEvents(false);
		
		List<ModelNode> oldNodes = serviceModel.getChildren(serviceModel.getModelRoot());
		Iterator<ModelNode> newServiceIter = newModel.getChildren(newModel.getModelRoot()).iterator();

		// Clear services and connections from the old model
		ModelNode serviceModelRoot = serviceModel.getModelRoot();
		serviceModel.removeAllChildren(serviceModelRoot);
		
		// Add services and connections from the new model
		while (newServiceIter.hasNext()) {
			ModelNode node = newServiceIter.next();
			if (node instanceof Service) {
				Service newService = (Service) node;
				// If the new model doesn't define location, copy
				// old location to new model.
				if (!newService.definesLocation()) {
					// find corresponding old node
					Service oldService = null;
					for (int i=0; i<oldNodes.size(); ++i) {
						ModelNode tmpNode = oldNodes.get(i);
						if (tmpNode instanceof Service) {
							Service tmpService = (Service) tmpNode;
							
							if (tmpService.getName().equals(newService.getName())
									&& tmpService.getId() == newService.getId()) {
								oldService = tmpService;
								break;
							}
						}
					}
					
					if (oldService != null) {
						// Copy location
						newService.setLocation(oldService.getLocation());
					}
					newService.setDefinesLocation(true);
				}

				serviceModel.insertNode(node, serviceModelRoot);
				
			} else if (node instanceof Connection) {
				Connection newConn = (Connection) node;
				// If the new model doesn't define points, copy
				// old points to new model.
				if (!newConn.definesPoints()) {
					// find corresponding old node
					Connection oldConn = null;
					for (int i=0; i<oldNodes.size(); ++i) {
						ModelNode tmpNode = oldNodes.get(i);
						if (tmpNode instanceof Connection) {
							Connection tmpConn = (Connection) tmpNode;
							
							if (tmpConn.getId() == newConn.getId()) {
								oldConn = tmpConn;
								break;
							}
						}
					}
					
					if (oldConn != null) {
						// Copy points
						newConn.setPoints(oldConn.getPoints());
					}
				}
				serviceModel.insertNode(node, serviceModelRoot);
			}
		}
		
		serviceModel.enableEvents(true);
		serviceModel.modelStructureChanged(serviceModelRoot);
	}

	
	/**
	 * Clears processes from given processing element to others.
	 * @param pe The pe to be cleared.
	 */
	public void clearProsessingElement(ProcessingElement pe) {
		ArrayList<Process> movedProcesses = new ArrayList<Process>();
		
		// If the pe to be cleared is the only cpu, clearing can't
		// be done
		boolean onlyOne = true;
		List<ModelNode> peNodeList = peModel.getChildren(peModel.getModelRoot());
		for (int i=0; i<peNodeList.size(); ++i) {
			ProcessingElement otherPe = (ProcessingElement)peNodeList.get(i);
			if (otherPe.getId() != pe.getId()
					&& otherPe.getType() == ProcessingElementType.CPU) {
				onlyOne = false;
				break;
			}
		}
		if (onlyOne) {
			return;
		}
		
		// Find processes that are to be moved
		Iterator<ModelNode> threadIter = peModel.getChildren(pe).iterator();
		while (threadIter.hasNext()) {
			ModelNode tNode = threadIter.next();
			
			if (tNode instanceof Thread) {
				Thread thread = (Thread) tNode;
				
				Iterator<ModelNode> processIter = peModel.getChildren(thread).iterator();
				while (processIter.hasNext()) {
					ModelNode pNode = processIter.next();
					
					if (pNode instanceof Process) {
						Process p = (Process) pNode;
						movedProcesses.add(p);
					}
				}
			}
		}
		
		peModel.enableEvents(false);
		
		// Redistribute processes to other cpus
		int processIndex = 0;
		// as long as processes left
		while (processIndex < movedProcesses.size()) {
			
			// go through all processing elements
			for (int i=0; i<peNodeList.size(); ++i) {
				ProcessingElement otherPe = (ProcessingElement)peNodeList.get(i);

				// handle only cpus other than the cleared one
				if (otherPe.getId() != pe.getId()
						&& otherPe.getType() == ProcessingElementType.CPU) {

					// go through all threads adding one removed
					// process to each thread
					threadIter = peModel.getChildren(otherPe).iterator();
					while (threadIter.hasNext()) {
						ModelNode tNode = threadIter.next();
						if (tNode instanceof Thread) {
							Thread thread = (Thread) tNode;
							Process p = movedProcesses.get(processIndex);
							
							peModel.moveNode(p, thread, peModel.getChildCount(thread));
							++processIndex;
						}
						// stop if all processes handled
						if (processIndex >= movedProcesses.size()) {
							break;
						}
					}
				}
				// stop if all processes handled
				if (processIndex >= movedProcesses.size()) {
					break;
				}
			}
		}
		
		peModel.enableEvents(true);
		peModel.modelStructureChanged(peModel.getModelRoot());
	}
}
