/**
 * File:    ExecutionMonitor.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 14.3.2007
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu;

import java.io.File;

import javax.swing.JFrame;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

import fi.cpu.ui.MainWindow;
import fi.cpu.ui.xml.UIConfigIO;


/**
 * Main class for Execution monitor
 */
public class ExecutionMonitor {
	
    public static void main(String[] args) {
    	// check arguments
    	for (int i=0; i<args.length; ++i) {
    		String arg = args[i];
    		
    		int eq = arg.indexOf("=");
    		String option = null;
    		String value = null;
    		
    		if (eq != -1) {
    			option = arg.substring(1, eq);
    			value = arg.substring(eq+1);
    		} else {
    			option = arg.substring(1);
    		}
    		
    		if (option.equals("help")) {
     			printHelp();
     			System.exit(0);

    		} else if (option.equals("history")) {
    			if (value == null || value.equals("")) {
        			System.err.println("Invalid option (empty value): "+arg);
        			System.err.flush();
        			printHelp();
        			System.exit(-1);    				
    			}
    			
    			try {
    				Integer.parseInt(value);
    			} catch (NumberFormatException e) {
        			System.err.println("Invalid option (expected integer value): "+arg);
        			System.err.flush();
        			printHelp();
        			System.exit(-1);    				    				
    			}
    			System.setProperty("history", value);

    		} else if (option.equals("ip")) {
    			if (value == null || value.equals("")) {
        			System.err.println("Invalid option (empty value): "+arg);
        			System.err.flush();
        			printHelp();
        			System.exit(-1);    				
    			}
    			System.setProperty("ip", value);
    			
    		} else if (option.equals("port")) {
    			if (value == null || value.equals("")) {
        			System.err.println("Invalid option (empty value): "+arg);
        			System.err.flush();
        			printHelp();
        			System.exit(-1);    				
    			}
    			
    			try {
    				Integer.parseInt(value);
    			} catch (NumberFormatException e) {
        			System.err.println("Invalid option (expected integer value): "+arg);
        			System.err.flush();
        			printHelp();
        			System.exit(-1);    				    				
    			}
    			System.setProperty("port", value);
    			
    		} else if (option.equals("serial")) {
    			System.setProperty("serial", "true");

    		} else if (option.equals("systemlaf")) {
    			System.setProperty("systemlaf", "true");

    		} else {
    			System.err.println("Invalid option: "+arg);
    			System.err.flush();
    			printHelp();
    			System.exit(-1);
    		}
    	}
    	
		// Set the system look and feel
    	if (System.getProperty("systemlaf") != null) {
    		String lafClass = UIManager.getSystemLookAndFeelClassName();
    		try {
    			UIManager.setLookAndFeel(lafClass);
    		}
    		catch(Exception e) {
    			System.out.println("Couldn't set look and feel: "+e.getMessage());
    		}
    	}
    	
        MainWindow main = MainWindow.getInstance();
        String filename = System.getProperty("configFile");
        
        if (filename != null) {
            if (new File(filename).exists()) {
            	UIConfigIO.readConfiguration(main.getConfiguration(), main, filename);
            }
        }
        
        showAppFrame(main, false);
    }

    
    /**
     * Prints help message
     */
    private static void printHelp() {
    	StringBuilder sb = new StringBuilder();
    	sb.append("\n");
    	sb.append("Possible options:\n");
    	sb.append("    -help             Print this message.\n");
    	sb.append("    -history=INT      Set the amount of chart points remembered.\n");
    	sb.append("    -ip=IP_NUM        Set the target board IP address.\n");
    	sb.append("    -port=PORT_NUM    Set the TCP port that is used.\n");
    	sb.append("    -serial           Use serial port instead of TCP/IP.\n");
    	sb.append("    -systemlaf        Use system look-and-feel.\n");
    	sb.append("\n");
    	System.out.println(sb.toString());
    }
    
    
	/**
	 * Shows the application frame given as an argument.
	 * For thread-safety, the task is done by the 
	 * event-dispatching thread.
	 * 
	 * @param frame      Frame to be shown
	 * @param maximized  If true, frame is shown in maximized state
	 */
	private static void showAppFrame(final JFrame frame, final boolean maximized) {
		try {
			SwingUtilities.invokeAndWait(new Runnable() {
				public void run() {
					frame.pack();
					frame.setVisible(true);
					if (maximized) {
						frame.setExtendedState(frame.getExtendedState() | JFrame.MAXIMIZED_BOTH);
					}
				}
			});
		}
		catch (Exception e) {
			// Empty
		}
	}
}
