/**
 *
 * @file measure.cc
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: measure.cc 1923 2011-08-08 09:41:57Z lehton87 $
 *
 */


#include "measure.hh"

#ifdef SCTG_USE_EXECMON
#ifndef MTI_SYSTEMC
#include "tcp_server_if.hh"
#endif
#endif

#include <vector>
#include <iomanip>
#include <boost/date_time.hpp>
#include <sstream>

namespace sctg
{

   Measure::Measure(sc_core::sc_module_name name, sctg::Configuration& config,
		    std::vector<sctg::ProcessingElement*> pes,
		    std::vector<sctg::Task*> tasks,
		    std::vector<sctg::MemoryModel*> memories,
		    std::vector<sctg::MemArea*> memAreas)
      : sc_core::sc_module(name),
	_config(config),
	_pes(pes),
	_tasks(tasks),
	_memories(memories),
	_memAreas(memAreas)
   {
      SC_THREAD(measureThread);
#ifdef SCTG_USE_EXECMON      
      if(_config.useExecMon() || _config.getExecMonStream())
      {
	 SC_THREAD(execMonThread);      
      }
#endif      

      std::ostringstream oss;
      oss << "Log created ; "
	  << boost::posix_time::second_clock::local_time() 
	  << std::endl << std::endl;
      
      _measure_start_time = oss.str();

   }
  
  
   Measure::~Measure()
   {
    
   }
  
  
   void Measure::measureThread()
   { 
      sc_core::sc_time interval = _config.getMeasurementInterval();
      if(interval == sc_core::SC_ZERO_TIME)
      { return; }

      std::cout << "Measuring statistics with " << interval
		<< " interval" << std::endl;

      std::vector<PeMeasurements> oldPeData;
      std::vector<PeMeasurements> newPeData;
      std::vector<BufferMeasurements> oldBufferData;
      std::vector<BufferMeasurements> newBufferData;
      oldPeData.resize(_pes.size());
      newPeData.resize(_pes.size());
      oldBufferData.resize(_pes.size());
      newBufferData.resize(_pes.size());

      if(_config.getSummaryStream())
      {
	 **_config.getSummaryStream() 
	    << Measure::_measure_start_time;
      }

      if(_config.getPacketStream())
      {
	 **_config.getPacketStream() 
	    << Measure::_measure_start_time
	    << "Time;ID;Src port; Dst port;Size" 
	    << std::endl;
      }

      if(_config.getTokenStream())
      {
	 **_config.getTokenStream() 
	    << Measure::_measure_start_time
	    << "Time sent;Time received;ID;Src port; Dst port;Size"
	    << std::endl;
      }
    
      if(_config.getPeStream())
	 {
	    **_config.getPeStream()
	       << Measure::_measure_start_time
	       << "Time;Name;Id;Utilization;Received bytes;Sent bytes;"
	       << "RX buffer usage;TX buffer usage;Idle cycles;"
	       << "Busy cycles;"
	       << "Execution cycles;Reading cycles;Sending cycles;"
	       << "RX wait cycles;TX wait cycles;Intra PE comm cycles;"
	       << "I-cache cycles;I-cache misses;"
	       << "D-cache cycles;D-cache misses"
	       << std::endl;
	 }

      if(_config.getMemStream())
      {
	 **_config.getMemStream()
	    << Measure::_measure_start_time
	    << "Start Time;End Time;CMD;Name;Area ID;Size;SRC port;"
	    << "DST port;"
	    << std::endl;
      }
		  

      if(_config.getAppStream())
	 {
	    **_config.getAppStream()
	       << Measure::_measure_start_time
	       << "Time;Name;Buffer usage;State"
	       << std::endl;
	 }

      while(true)
      {
	 wait(interval);

	 std::cout << "Measuring. Current simulation time is "
		   << sc_core::sc_time_stamp() << std::endl; 
	 
	
	 for(unsigned int i = 0; i < _pes.size(); ++i)
	 {
	    // Calculate tx/rx bytes
	    oldBufferData.at(i) = newBufferData.at(i);
	    newBufferData.at(i) = 
	       _config.getBufferByResource
	       (_pes.at(i)->getId())->getMeasurements();

	    unsigned long int rxBytes = 
	       newBufferData.at(i).rxBytes - oldBufferData.at(i).rxBytes;
	    unsigned long int txBytes = 
	       newBufferData.at(i).txBytes - oldBufferData.at(i).txBytes;
	    unsigned long int intBytes = 
	       newBufferData.at(i).internalBytes - 
	       oldBufferData.at(i).internalBytes;

	    // Calculate utilization and other PE related stuff
	    oldPeData.at(i) = newPeData.at(i);
	    newPeData.at(i) = _pes.at(i)->getMeasurements();
	    sc_core::sc_time idle = 
	       newPeData.at(i).idleTime - oldPeData.at(i).idleTime;
	    sc_core::sc_time exec = 
	       newPeData.at(i).execTime - oldPeData.at(i).execTime;
	    
	    unsigned long int iCacheMisses = 
	       newPeData.at(i).iCacheMisses -
	       oldPeData.at(i).iCacheMisses;
	    unsigned long int dCacheMisses = 
	       newPeData.at(i).dCacheMisses -
	       oldPeData.at(i).dCacheMisses;

	    
	    double util = (exec / (exec + idle));	    

	    unsigned long int idleCycles = 0;
	    unsigned long int busyCycles = 0;
	    unsigned long int execCycles = 0;
	    unsigned long int readCycles = 0;
	    unsigned long int sendCycles = 0;
	    unsigned long int rxCycles = 0;
	    unsigned long int txCycles = 0;
	    unsigned long int intraCycles = 0;
	    unsigned long int iCacheCycles = 0;
	    unsigned long int dCacheCycles = 0;

	    idleCycles = newPeData.at(i).idleCycles - 
	       oldPeData.at(i).idleCycles;
	    busyCycles = newPeData.at(i).busyCycles - 
	       oldPeData.at(i).busyCycles;
	    
	    execCycles = (newPeData.at(i).execTime - 
			  oldPeData.at(i).execTime) /
	       newPeData.at(i).cycleLength;
	    readCycles = (newPeData.at(i).readingTime - 
			  oldPeData.at(i).readingTime) /
	       newPeData.at(i).cycleLength;
	    sendCycles = (newPeData.at(i).sendingTime - 
			  oldPeData.at(i).sendingTime) / 
	       newPeData.at(i).cycleLength;
	    rxCycles = (newPeData.at(i).rxWaitTime - 
			oldPeData.at(i).rxWaitTime) / 
	       newPeData.at(i).cycleLength;
	    txCycles = (newPeData.at(i).txWaitTime - 
			oldPeData.at(i).txWaitTime) / 
	       newPeData.at(i).cycleLength;
	    intraCycles = (newPeData.at(i).intraTxWait - 
			    oldPeData.at(i).intraTxWait) / 
	       newPeData.at(i).cycleLength;
	    iCacheCycles = (newPeData.at(i).iCacheTime - 
			   oldPeData.at(i).iCacheTime) / 
	       newPeData.at(i).cycleLength;
	    dCacheCycles = (newPeData.at(i).dCacheTime - 
			   oldPeData.at(i).dCacheTime) / 
	       newPeData.at(i).cycleLength;



	    if(_config.getPeStream())
	    {
	       **_config.getPeStream()
		  << sc_core::sc_time_stamp().value()
		  << ";" << _pes.at(i)->getName()
		  << ";" << _pes.at(i)->getId()
		  << ";" << util 
		  << ";" << rxBytes - intBytes
		  << ";" << txBytes
		  << ";" << newBufferData.at(i).rxUsed 
		  << ";" << newBufferData.at(i).txUsed
		  << ";" << idleCycles
		  << ";" << busyCycles
		  << ";" << execCycles
		  << ";" << readCycles
		  << ";" << sendCycles
		  << ";" << rxCycles
		  << ";" << txCycles
		  << ";" << intraCycles
		  << ";" << iCacheCycles
		  << ";" << iCacheMisses
		  << ";" << dCacheCycles
		  << ";" << dCacheMisses
		  << std::endl;
	    }
	 }

	 if(_config.getAppStream())
	 {
	    for(unsigned int i = 0; i < _tasks.size(); ++i)
	    {
	       **_config.getAppStream()
		  << sc_core::sc_time_stamp().value()
		  << ";" << _tasks.at(i)->getName() 
		  << ";" << _tasks.at(i)->getBufferUsage()
		  << ";" << stateToString(_tasks.at(i)->getState())
		  << std::endl;
	       (**_config.getAppStream()).flush();
	    }
	 }	 
      }
      
   }

#ifdef SCTG_USE_EXECMON
   void Measure::execMonThread()
   {
      sc_core::sc_time interval = sc_core::sc_time(1, sc_core::SC_MS);
      unsigned long int currentTime = 0;

      std::ostringstream oss;

      std::vector<PeMeasurements> oldPeData;
      std::vector<PeMeasurements> newPeData;
      std::vector<BufferMeasurements> oldBufferData;
      std::vector<BufferMeasurements> newBufferData;
      oldPeData.resize(_pes.size());
      newPeData.resize(_pes.size());
      oldBufferData.resize(_pes.size()+_memories.size());
      newBufferData.resize(_pes.size()+_memories.size());

      std::vector<PeMeasurements> oldMemData;
      std::vector<PeMeasurements> newMemData;
      oldMemData.resize(_memories.size());
      newMemData.resize(_memories.size());
    
      while(true)
      {	 
	 oss.str("");
	 oss << "<measurement time=\"" << currentTime << "\">";
	 	
	 for(unsigned int i = 0; i < _pes.size(); ++i)
	 {
	    // Calculate tx/rx bytes
	    oldBufferData.at(i) = newBufferData.at(i);
	    newBufferData.at(i) = 
	       _config.getBufferByResource
	       (_pes.at(i)->getId())->getMeasurements();
	       

	    unsigned long int rxBytes = 
	       newBufferData.at(i).rxBytes - oldBufferData.at(i).rxBytes;
	    unsigned long int txBytes = 
	       newBufferData.at(i).txBytes - oldBufferData.at(i).txBytes;
	    unsigned long int intBytes = 
	       newBufferData.at(i).internalBytes - 
	       oldBufferData.at(i).internalBytes;

	    // Calculate utilization
	    oldPeData.at(i) = newPeData.at(i);
	    newPeData.at(i) = _pes.at(i)->getMeasurements();
	    sc_core::sc_time idle = 
	       newPeData.at(i).idleTime - oldPeData.at(i).idleTime;
	    sc_core::sc_time exec = 
	       newPeData.at(i).execTime - oldPeData.at(i).execTime;

	    double util = (exec / (exec + idle));

	    if((exec + idle) == sc_core::SC_ZERO_TIME)
	    {
	       util = 0.0;
	    }

	    oss << "<graph cpu_id=\"" << _pes.at(i)->getId() << "\" value=\""
		<< (100.0 * util) << "\" id=\"0\"/>";

	 }

	 for(unsigned int i = 0; i < _memories.size(); ++i)
	 {
	    // Calculate tx/rx bytes
	    oldBufferData.at(i+_pes.size()) = newBufferData.at(i+_pes.size());
	    newBufferData.at(i+_pes.size()) = 
	       _config.getBufferByResource
	       (_memories.at(i)->getId())->getMeasurements();
	       

	    unsigned long int rxBytes = 
	       newBufferData.at(i+_pes.size()).rxBytes 
	       - oldBufferData.at(i+_pes.size()).rxBytes;
	    unsigned long int txBytes = 
	       newBufferData.at(i+_pes.size()).txBytes 
	       - oldBufferData.at(i+_pes.size()).txBytes;
	    unsigned long int intBytes = 
	       newBufferData.at(i+_pes.size()).internalBytes - 
	       oldBufferData.at(i+_pes.size()).internalBytes;

	    // Calculate utilization
	    oldMemData.at(i) = newMemData.at(i);
	    newMemData.at(i) = _memories.at(i)->getMeasurements();
	    sc_core::sc_time idle = 
	       newMemData.at(i).idleTime - oldMemData.at(i).idleTime;
	    sc_core::sc_time exec = 
	       newMemData.at(i).busyTime - oldMemData.at(i).busyTime;

	    double util = (exec / (exec + idle));

	    if((exec + idle) == sc_core::SC_ZERO_TIME)
	    {
	       util = 0.0;
	    }

	    oss << "<graph cpu_id=\"" << _memories.at(i)->getId() 
		<< "\" value=\""
		<< (100.0 * util) << "\" id=\"0\"/>";
	    
//  	    std::cout << "MEM " << _memories.at(i)->getName() 
//  		      << " " << util << std::endl;
	 }

	 for(unsigned int i = 0; i < _pes.size(); ++i)
	 {
	    while(!(_pes.at(i)->getTokenQueue().empty()))
	    {
	       
	       oss << "<signal " 
		   << "dest_pid=\"" << 
		  _config.getResourceUserByInPort
		  (_pes.at(i)->getTokenQueue().front().destination)->getId() 
		   << "\" "
		   << "value=\"" << _pes.at(i)->getTokenQueue().front().size 
		   << "\" " << "source_pid=\"";
	       bool f1 = false;
	       unsigned long int src = _pes.at(i)->getTokenQueue().front().source;
	       for(unsigned int j = 0; j < _tasks.size(); ++j)
	       {
		  if(_tasks.at(j)->hasOutPort(src))
		  {
		     oss << _tasks.at(j)->getId() << "\"/>";
		     f1 = true;
		     break;
		  }
	       }	       
	       if(!f1) oss << "0\"/>";
	       _pes.at(i)->getTokenQueue().pop();
	    }
	 }

	 for(unsigned int i = 0; i < _tasks.size(); ++i)
	 {
	    double latency = _tasks.at(i)->getLastLatency() /
	       sc_core::sc_time(1.0, sc_core::SC_US);
	    double response = _tasks.at(i)->getLastResponse() /
	       sc_core::sc_time(1.0, sc_core::SC_US);
	    oss << "<process_exec " 
		<< "latency=\"" << latency << "\" "
		<< "local_comm_bytes=\"" << _tasks.at(i)->getLocalBytesSent() 
		<< "\" local_comm_time=\"" << 0 << "\" "
		<< "response_time=\"" << response << "\" "
		<< "remote_comm_bytes=\"" << _tasks.at(i)->getRemoteBytesSent()
		<< "\" remote_comm_time=\"" << 0 << "\" "
		<< "count=\"" << _tasks.at(i)->getTimesTriggered() << "\" "
		<< "pid=\"" << _tasks.at(i)->getId() << "\" "
		<< "signal_queue=\"" << _tasks.at(i)->getBufferUsage() << "\" "
		<< "exec_time=\"" << _tasks.at(i)->getCyclesExecuted() 
		<< "\"/>";
	    	       
	 }

	 oss << "</measurement>" << std::endl;

	 std::string str = oss.str();
#ifndef MTI_SYSTEMC
	 if(_config.useExecMon())
	 {
	    _config.getTcpServer()->send(str);
	 }
#endif
	 if(_config.getExecMonStream())
	 {
	    **_config.getExecMonStream()
	       << oss.str();
	    (**_config.getExecMonStream()).flush();
	 }

	 wait(interval);
	 currentTime++;

      }
   }
#endif
   

}


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
