/**
 *
 * @file resource.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: resource.hh 1916 2011-07-06 12:44:26Z lehton87 $
 *
 */


#ifndef SCTG_RESOURCE_HH
#define SCTG_RESOURCE_HH

#include "buffer.hh"
#include <systemc>
#include <boost/property_tree/ptree.hpp>
#include <string>
#include <vector>
#include <map>

namespace sctg
{
   class Task;

   /** Base class for ProcessingElement and MemoryModel classes
    *
    *
    */
   class Resource : public sc_core::sc_module
   {
   public:
      
      /** Constructor
       */
      Resource(sc_core::sc_module_name name, 
	       const boost::property_tree::ptree& pt,
	       sctg::Configuration& config);
      
      /** Destructor
       */
      virtual ~Resource();
      
      /** Returns name
       */
      const std::string& getName() const;
      
      /** Returns ID
       */
      unsigned long int getId() const;
      
      /** Returns operating frequency
       */
      unsigned long int getFrequency() const;
      
      /** Returns type
       */
      const std::string& getType() const;

      /** Returns terminal id
       */
      unsigned long int getTerminal() const;
      
      /** Set terminal's address
       */
      void setTerminal(unsigned long int terminal,
		       unsigned long int address,
		       unsigned long int width);

      /** Get terminal's address
       */
      unsigned long int getTerminalAddress(unsigned long int terminal) const;

      /** Get terminal's width
       */
      unsigned long int getTerminalWidth(unsigned long int terminal) const;
      
      /** Maps task or mem_area to this Resource
       */
      virtual void mapResourceUser(ResourceUser* task) = 0;

   private:
      
      /** Name
       */
      std::string       _name;
      /** Unique ID number
       */
      unsigned long int _id;
      /** Operating frequency
       */
      unsigned long int _freq;
      /** Resources type
       */
      std::string       _type;

      /** Terminal connected to this resource
       */
      std::vector<unsigned long int> _terminals;
      /** Terminal's network address
       */
      std::map<unsigned long int, unsigned long int> _terminalAddress;
      /** Terminal's width in bits
       */
      std::map<unsigned long int, unsigned long int> _terminalWidth;

   protected:      

      /** Pointer to resource's buffer
       */
      std::auto_ptr<Buffer> _buffer;
      /** Maximum size of the packets this resource sends
       */
      unsigned long int _packetSize;
   };
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
