/**
 *
 * @file ring_with_pkt_codec_vhd_bfm.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: ring_with_pkt_codec_vhd_bfm.hh 1957 2011-08-23 09:32:15Z lehton87 $
 *
 */

#ifndef RING_WITH_PKT_CODEC_VHD_BFM_HH
#define RING_WITH_PKT_CODEC_VHD_BFM_HH

#include "vhd/ring_with_pkt_codec_top.hh"
#include "buffer_if.hh"
#include "tg_packet.hh"
#include "noc_conf_if.hh"

#include <systemc>
#include <vector>
#include <queue>
#include <iostream>
#include <stdexcept>


namespace sctg
{
namespace ring_vhd
{
   
   template < int n_ag_g,
	      int data_width_g,  
	      int tx_len_width_g>
   class ring_with_pkt_codec_vhd_bfm : public sc_core::sc_module
   {
   public:
  
      SC_HAS_PROCESS(ring_with_pkt_codec_vhd_bfm);

      //* Constructor
      ring_with_pkt_codec_vhd_bfm(sc_core::sc_module_name name,
				      sctg::NocConfIf* confIf,
				      int diag_en_g,
				      int stfwd_en_g,
				      int addr_width_g,
				      int packet_length_g,
				      int timeout_g,
				      int fill_packet_g, 
				      int len_flit_en_g, 
				      int oaddr_flit_en_g, 
				      int fifo_depth_g, 
				      int noc_freq_g, 
				      int ip_freq_g)
      : sc_module(name),
	_confIf(confIf)
   {

      clk_ip = new sc_core::sc_clock("clk_ip", 
				     sc_core::sc_time(1000000000.0/ip_freq_g,
						      sc_core::SC_NS));
      clk_noc = new sc_core::sc_clock("clk_noc", 
				      sc_core::sc_time(1000000000.0/noc_freq_g, 
						       sc_core::SC_NS));

      ring = new ring_with_pkt_codec_top< 
      n_ag_g, // Number of agents
	 data_width_g,    // data width in bits
	 tx_len_width_g  // width of tx_len line
	 >("Ring", "ring_with_pkt_codec_top",
	   diag_en_g,
	   stfwd_en_g,
	   addr_width_g,
	   packet_length_g,
	   timeout_g,
	   fill_packet_g,
	   len_flit_en_g,
	   oaddr_flit_en_g,
	   fifo_depth_g,
	   noc_freq_g,
	   ip_freq_g);
      
      // Bind ports
      ring->clk_net(*clk_noc);
      ring->clk_ip(*clk_ip);
      ring->rst_n(rst_n);
      ring->tx_av_in(rx_av);
      ring->tx_data_in(rx_data);
      ring->tx_we_in(rx_we);
      ring->tx_txlen_in(rx_txlen);
      ring->tx_full_out(rx_full);
      ring->tx_empty_out(rx_empty);
      ring->rx_av_out(tx_av);
      ring->rx_data_out(tx_data);
      ring->rx_re_in(tx_re);
      ring->rx_empty_out(tx_empty);   

      // Initial values
      rst_n.write(sc_dt::SC_LOGIC_0);
      rx_av = "0";
      rx_data = "0";
      rx_we = "0";
      rx_txlen = "0";
      tx_re = "0";

      SC_THREAD(thread);
      
      for(int i = 0; i < n_ag_g; ++i)
      {
	 sc_spawn(sc_bind(&ring_with_pkt_codec_vhd_bfm::sender, this, i));
	 sc_spawn(sc_bind(&ring_with_pkt_codec_vhd_bfm::receiver, this, i));
      }
   
   }

      //* Destructor
      ~ring_with_pkt_codec_vhd_bfm()
      {
	 delete clk_ip;
	 delete clk_noc;
	 delete ring;
      }

      void sender(unsigned int agent)
      {   
	 wait(reset);

	 sctg::BufferInterface* buffer = _confIf->getBufferIf(agent);

	 if(!buffer) 
	 {return;}

	 tgPacket* packet = 0;

	 sc_core::sc_lv<n_ag_g>  rx_av_s;
	 sc_core::sc_lv<n_ag_g*data_width_g> rx_data_s;
	 sc_core::sc_lv<n_ag_g> rx_we_s;
	 sc_core::sc_lv<n_ag_g*tx_len_width_g> rx_txlen_s;   

	 while(true)
	 {
	    // Wait for packets to send
	    if(!buffer->txPacketAvailable())
	    {
	       sc_core::wait(*(buffer->txGetPacketAvailableEvent()));
	    }
	    // Read packet from buffer
	    packet = buffer->txGetPacket();
	    sc_core::wait(clk_ip->posedge_event());
	    // Send address - set address valid
	    rx_av_s = rx_av.get_new_value();
	    rx_av_s[agent] = sc_dt::SC_LOGIC_1;
	    rx_av = rx_av_s;
	    // Put first data (is address) to bus
	    rx_data_s = rx_data.get_new_value();
	    rx_data_s.range((agent+1)*data_width_g-1, agent*data_width_g)
	       = sc_dt::sc_uint<data_width_g>(packet->address);
	    rx_data = rx_data_s;

// 	    std::cout << "Agent " << agent << " setting addr "
// 		      << rx_data_s.range((agent+1)*data_width_g-1, 
// 					 agent*data_width_g).to_string() 
// 		      << " at " << sc_core::sc_time_stamp() << std::endl;

	    // Set transfer length
	    rx_txlen_s = rx_txlen.get_new_value();
	    rx_txlen_s.range((agent+1)*tx_len_width_g-1, agent*tx_len_width_g) 
	       = sc_dt::sc_uint<tx_len_width_g>(packet->size/4);
	    rx_txlen = rx_txlen_s;
	    // Set write enable
	    rx_we_s = rx_we.get_new_value();
	    rx_we_s[agent] = sc_dt::SC_LOGIC_1;
	    rx_we = rx_we_s;
	    // Wait until NoC is ready to read
	    do { sc_core::wait(clk_ip->posedge_event()); }
	    while(rx_full.read()[agent] == sc_dt::SC_LOGIC_1);
	    // Negate address valid
	    rx_av_s = rx_av.get_new_value();
	    rx_av_s[agent] = sc_dt::SC_LOGIC_0;
	    rx_av = rx_av_s;
	    // Send data
	    for(unsigned int i = 0; i < packet->size/4; ++i)
	    {
	       // Set data to line
	       rx_data_s = rx_data.get_new_value();
	       if(i == 0)
	       {
		  // Send first byte actually, fill rest with dummy data
		  rx_data_s.range((agent+1)*data_width_g-1, agent*data_width_g)
		     = sc_dt::sc_uint<data_width_g>
		     (*reinterpret_cast<int*>(packet->data));
	       }
	       else
	       {
		  rx_data_s.range((agent+1)*data_width_g-1, agent*data_width_g)
		     = sc_dt::sc_uint<data_width_g>(agent);
	       }
	       rx_data = rx_data_s;
	       
// 	       std::cout << "Agent " << agent << " setting data "
// 			 << rx_data_s.range((agent+1)*data_width_g-1, 
// 					    agent*data_width_g).to_string()
// 			 << " at " << sc_core::sc_time_stamp() << std::endl;

	       // Wait clock cycle or more if NoC can't read it
	       do { sc_core::wait(clk_ip->posedge_event()); }
	       while(rx_full.read()[agent] == sc_dt::SC_LOGIC_1);
	    }
	    // Negate write enable
	    rx_we_s = rx_we.get_new_value();
	    rx_we_s[agent] = sc_dt::SC_LOGIC_0;
	    rx_we = rx_we_s;
	    // Set data bus to zero (not necessary, for easier debugging in sim)
	    rx_data_s = rx_data.get_new_value();
	    rx_data_s.range((agent+1)*data_width_g-1, agent*data_width_g)
	       = sc_dt::sc_uint<data_width_g>(0);
	    rx_data = rx_data_s;
	    // Set tx len lines to zeros (not necessary)
	    rx_txlen_s = rx_txlen.get_new_value();
	    rx_txlen_s.range((agent+1)*tx_len_width_g-1, agent*tx_len_width_g) 
	       = sc_dt::sc_uint<tx_len_width_g>(0);
	    rx_txlen = rx_txlen_s;
	    // Delete packet
	    delete [] packet->data;
	    delete packet; packet = 0;      
	 }
      }


      void receiver(unsigned int agent)
      {
	 wait(reset);

	 sc_core::sc_lv<n_ag_g> tx_re_s;

	 sctg::BufferInterface* buffer = _confIf->getBufferIf(agent);
	 tgPacket* packet = 0;

	 if(!buffer) 
	 {return;}

	 while(true)
	 {
	    sc_core::wait(clk_ip->posedge_event());

	    if(tx_empty.read()[agent] == sc_dt::SC_LOGIC_0)
	    {
	       // Set read enable
	       tx_re_s = tx_re.get_new_value();
	       tx_re_s[agent] = sc_dt::SC_LOGIC_1;
	       tx_re = tx_re_s;
	       // Check whether it's address or data
	       if(tx_av.read()[agent] == sc_dt::SC_LOGIC_1)
	       {
// 		  std::cout << "Agent " << agent << " getting addr "
// 			    << tx_data.read().range((agent+1)*data_width_g-1, 
// 				 agent*data_width_g).to_string() 
// 			    << " at " << sc_core::sc_time_stamp() << std::endl;
		  
		  if(packet == 0)
		  {
		     packet = new tgPacket;
		     packet->address =
			tx_data.read().range((agent+1)*data_width_g-1, 
					     agent*data_width_g).to_ulong();
		     packet->size = 0;
		     packet->data = 0;
		  }
		  else if(packet->size != 0)
		  {
// 		     std::cout << "Agent " << agent 
// 			       << " received packet with size "
// 			       << packet->size
// 			       << " at " << sc_core::sc_time_stamp() 
// 			       << std::endl;

		     // Clear read enable
		     tx_re_s = tx_re.get_new_value();
		     tx_re_s[agent] = sc_dt::SC_LOGIC_0;
		     tx_re = tx_re_s;

		     // Wait until packet fits to agent's buffer
		     while(buffer->rxSpaceLeft() < packet->size)
		     {	       
			wait(*(buffer->rxGetReadEvent()));
		     }
		     buffer->rxPutPacket(packet);
		     packet = 0;
		     sc_core::wait(clk_ip->posedge_event());

		     packet = new tgPacket;
		     packet->address =
			tx_data.read().range((agent+1)*data_width_g-1, 
					     agent*data_width_g).to_ulong();
		     packet->size = 0;
		     packet->data = 0;

		     // Set read enable
		     tx_re_s = tx_re.get_new_value();
		     tx_re_s[agent] = sc_dt::SC_LOGIC_1;
		     tx_re = tx_re_s;
		  }

	       }
	       else
	       {
// 		  std::cout << "Agent " << agent << " getting data "
// 			    << tx_data.read().range((agent+1)*data_width_g-1,
// 				 agent*data_width_g).to_string() 
// 			    << " at " << sc_core::sc_time_stamp() << std::endl;
		  if(packet == 0)
		  {
		     std::cout << "MESH DBG2" << std::endl;
		     throw std::runtime_error
			("mesh_receiver: got data without address");
		  }
		  if(packet->size == 0)
		  {
		     packet->data = new unsigned char[sizeof(unsigned int)];
		     unsigned int data = tx_data.read().
			range((agent+1)*data_width_g-1, agent*data_width_g).to_uint();
// 		     std::cout << "Agent " << agent << " getting packet id "
// 			       << data 
// 			       << " at " << sc_core::sc_time_stamp() 
// 			       << std::endl;

		     *reinterpret_cast<unsigned int*>(packet->data) = data;
	       
		  }
		  packet->size += data_width_g/8;
	       }	 
	    }
	    else
	    {
	 
	       // Set read enable low
	       tx_re_s = tx_re.get_new_value();
	       tx_re_s[agent] = sc_dt::SC_LOGIC_0;
	       tx_re = tx_re_s;
	 
	       // If we have packet put it to buffer
	       if(packet != 0)
	       {
// 		  std::cout << "Agent " << agent 
// 			    << " received packet with size "
// 			    << packet->size
// 			    << " at " << sc_core::sc_time_stamp() << std::endl;
		  // Wait until packet fits to agent's buffer
		  while(buffer->rxSpaceLeft() < packet->size)
		  {	       
		     wait(*(buffer->rxGetReadEvent()));
		  }
		  buffer->rxPutPacket(packet);
		  packet = 0;
		  sc_core::wait(clk_ip->posedge_event());
	       }
	       else
	       {
// 		  std::cout << "Agent " << agent << " idling"
// 			    << " at " << sc_core::sc_time_stamp() << std::endl;
	       }
	    }
	 }
      }

      void thread()
      {      
	 // Handles reset only
	 sc_core::wait(1, sc_core::SC_NS);
	 rst_n.write(sc_dt::SC_LOGIC_1);
	 reset.notify();
      }

   private:

      sc_core::sc_event reset;

      sctg::NocConfIf* _confIf;
 
      sc_core::sc_clock* clk_ip;
      sc_core::sc_clock* clk_noc;
   
      sc_core::sc_signal<sc_dt::sc_logic> rst_n;
      sc_core::sc_signal<sc_dt::sc_lv<n_ag_g> >  rx_av; 
      sc_core::sc_signal<sc_dt::sc_lv<n_ag_g*data_width_g> > rx_data;
      sc_core::sc_signal<sc_dt::sc_lv<n_ag_g> > rx_we;
      sc_core::sc_signal<sc_dt::sc_lv<n_ag_g*tx_len_width_g> > rx_txlen;
      sc_core::sc_signal<sc_dt::sc_lv<n_ag_g> > rx_full;
      sc_core::sc_signal<sc_dt::sc_lv<n_ag_g> > rx_empty;
      sc_core::sc_signal<sc_dt::sc_lv<n_ag_g> > tx_av;
      sc_core::sc_signal<sc_dt::sc_lv<n_ag_g*data_width_g> > tx_data;
      sc_core::sc_signal<sc_dt::sc_lv<n_ag_g> > tx_re;
      sc_core::sc_signal<sc_dt::sc_lv<n_ag_g> > tx_empty;

      sc_dt::sc_logic              rx_av_v[n_ag_g];    // Out
      sc_dt::sc_lv<data_width_g>   rx_data_v[n_ag_g];  // Out
      sc_dt::sc_logic              rx_we_v[n_ag_g];    // Out
      sc_dt::sc_lv<tx_len_width_g> rx_txlen_v[n_ag_g]; // Out
      sc_dt::sc_logic              rx_full_v[n_ag_g];  // In
      sc_dt::sc_logic              rx_empty_v[n_ag_g]; // In
      sc_dt::sc_logic              tx_av_v[n_ag_g];    // In
      sc_dt::sc_lv<data_width_g>   tx_data_v[n_ag_g];  // In
      sc_dt::sc_logic              tx_re_v[n_ag_g];    // Out
      sc_dt::sc_logic              tx_empty_v[n_ag_g]; // in
 
      ring_with_pkt_codec_top< 
	 n_ag_g,
	 data_width_g,    // data width in bits
	 tx_len_width_g  // width of tx_len line
	 >* ring;

   };

}
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
