/**
 *
 * @file pkt_counter.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: pkt_counter.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#ifndef ASEBT_SC_RTL_1_PKT_COUNTER_HH
#define ASEBT_SC_RTL_1_PKT_COUNTER_HH

#include <systemc>

namespace asebt
{
namespace sc_rtl_1
{

   template <int tx_len_width_g = 8>
   class pkt_counter : public sc_core::sc_module
   {
   public:
      sc_core::sc_in_clk                              clk;
      sc_core::sc_in<sc_dt::sc_logic>               rst_n;
      sc_core::sc_in<sc_dt::sc_lv<tx_len_width_g> > len_in;
      sc_core::sc_in<sc_dt::sc_logic>               new_tx_in;
      sc_core::sc_in<sc_dt::sc_logic>               new_pkt_in;
      sc_core::sc_in<sc_dt::sc_logic>               idle_in;

      SC_HAS_PROCESS(pkt_counter);
  
      pkt_counter(sc_core::sc_module_name name)
	 : sc_core::sc_module(name),
	   clk("clk"),
	   rst_n("rst_n"),
	   len_in("len_in"),
	   new_tx_in("new_tx_in"),
	   new_pkt_in("net_pkt_in"),
	   idle_in("idle_in"),
	   idle_counter_(0),
	   active_counter_(0),
	   min_tx_size_(0),
	   max_tx_size_(0),
	   sum_of_len_(0),
	   pkt_count_(0),
	   tx_count_(0),
	   current_tx_size_(0),
	   old_new_tx_("old_new_tx"),
	   old_new_pkt_("old_new_pkt")
      {
   
	 old_new_tx_  = sc_dt::SC_LOGIC_0;
	 old_new_pkt_ = sc_dt::SC_LOGIC_0;

	 SC_METHOD(method);
	 sensitive << clk.pos() << rst_n;
      }

      virtual ~pkt_counter()
      {

      }

      void method()
      {
	 if(rst_n.read() == sc_dt::SC_LOGIC_0)
	 {
	    idle_counter_    = 0;
	    active_counter_  = 0;
	    min_tx_size_     = 0;
	    max_tx_size_     = 0;
	    sum_of_len_      = 0;
	    pkt_count_       = 0;
	    tx_count_        = 0;
	    current_tx_size_ = 0;
	    old_new_tx_      = sc_dt::SC_LOGIC_0;
	    old_new_pkt_     = sc_dt::SC_LOGIC_0;
	 }
	 else
	 {
	    old_new_pkt_.write(new_pkt_in.read());
	    old_new_tx_.write(new_tx_in.read());

	    if(new_tx_in.read() == sc_dt::SC_LOGIC_1 &&
	       old_new_tx_.read() == sc_dt::SC_LOGIC_0)
	    {
	       ++tx_count_;
	       if(min_tx_size_ == 0 || current_tx_size_ < min_tx_size_)
	       {
		  min_tx_size_ = current_tx_size_;
	       }
	       current_tx_size_ = 0;
	    }

	    if(new_pkt_in.read() == sc_dt::SC_LOGIC_1 &&
	       old_new_pkt_.read() == sc_dt::SC_LOGIC_0)
	    {
	       ++pkt_count_;
	       current_tx_size_ += len_in.read().to_uint();

	       if(current_tx_size_ > max_tx_size_)
	       { max_tx_size_ = current_tx_size_; }

	       sum_of_len_ += len_in.read().to_uint();
	    }

	    if(idle_in.read() == sc_dt::SC_LOGIC_1)
	    { ++idle_counter_; }
	    else
	    { ++active_counter_; }
	 }
      }

   private:
  
      unsigned long idle_counter_;
      unsigned long active_counter_;
      unsigned long min_tx_size_;
      unsigned long max_tx_size_;
      unsigned long sum_of_len_;
      unsigned long pkt_count_;
      unsigned long tx_count_;
      unsigned long current_tx_size_;
      sc_core::sc_signal<sc_dt::sc_logic> old_new_tx_;
      sc_core::sc_signal<sc_dt::sc_logic> old_new_pkt_;

   };

}
}


#endif



// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
