/**
 *
 * @file mesh_router.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: mesh_router.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#ifndef ASEBT_MESH_2D_SC_2_MESH_ROUTER_HH
#define ASEBT_MESH_2D_SC_2_MESH_ROUTER_HH

#include "sc_rtl_2/fifo.hh"
#include "sc_rtl_2/multiclk_fifo.hh"

#include <systemc>

namespace asebt
{
namespace mesh_2d_sc_2
{
   namespace {
      const unsigned int N = 0;
      const unsigned int W = 1;
      const unsigned int S = 2;
      const unsigned int E = 3;
      const unsigned int IP = 4;
      const unsigned int NO_DIR_C = 5;
      
      const unsigned int len_width_g = 8;
   }

   template<int stfwd_en_g, 
	    int data_width_g,
	    int addr_width_g,
	    int fifo_depth_g,
	    int pkt_len_g,
	    int len_flit_en_g,
	    int oaddr_flit_en_g,
	    int ip_freq_g,
	    int mesh_freq_g,	 
	    int num_cols_g,
	    int num_rows_g>
   class mesh_router : public sc_core::sc_module
   {
   public:
      //* ports
      sc_core::sc_in_clk                clk_ip;
      sc_core::sc_in_clk                clk_mesh;
      sc_core::sc_in<bool> rst_n;

      sc_core::sc_in<sc_dt::sc_bv<data_width_g> > data_in[4];
      sc_core::sc_in<bool>             empty_in[4];
      sc_core::sc_in<bool>             full_in[4];
      sc_core::sc_in<bool>             re_in[4];

      sc_core::sc_in<sc_dt::sc_bv<data_width_g> > data_ip_tx_in;
      sc_core::sc_in<bool>             we_ip_tx_in;
      sc_core::sc_out<bool>            empty_ip_tx_out;
      sc_core::sc_out<bool>            full_ip_tx_out;

      sc_core::sc_out<sc_dt::sc_bv<data_width_g> > data_out[4];
      sc_core::sc_out<bool>             empty_out[4];
      sc_core::sc_out<bool>             full_out[4];
      sc_core::sc_out<bool>             re_out[4];

      sc_core::sc_out<sc_dt::sc_bv<data_width_g> > data_ip_rx_out;
      sc_core::sc_in<bool>              re_ip_rx_in;
      sc_core::sc_out<bool>             full_ip_rx_out;
      sc_core::sc_out<bool>             empty_ip_rx_out;

      SC_HAS_PROCESS(mesh_router);

      //* Constructor
      mesh_router(sc_core::sc_module_name name,
		  int col_addr,
		  int row_addr)
	 : sc_core::sc_module(name),
	   clk_ip("clk_ip"),
	   clk_mesh("clk_mesh"),
	   rst_n("rst_n"),
	   data_ip_tx_in("data_ip_tx_in"),
	   we_ip_tx_in("we_ip_tx_in"),
	   empty_ip_tx_out("empty_ip_tx_out"),
	   full_ip_tx_out("full_ip_tx_out"),
	   data_ip_rx_out("data_ip_rx_out"),
	   re_ip_rx_in("re_ip_rx_in"),
	   full_ip_rx_out("full_ip_rx_out"),
	   empty_ip_rx_out("empty_ip_rx_out"),

	   fifo_n(0),
	   fifo_s(0),
	   fifo_w(0),
	   fifo_e(0),
	   multiclk_fifo_rx("multiclk_fifo_ip_rx"),
	   multiclk_fifo_tx("multiclk_fifo_ip_tx"),
	   col_addr_g(col_addr),
	   row_addr_g(row_addr)
      {

	 if(row_addr_g == 0)
	 {
	    // First row, don't connect north bridge
	    full_from_fifo[N].write(false);
	    data_from_fifo[N].write("0");
	    empty_from_fifo[N].write(true);
	 }
	 else
	 {
	    fifo_n = 
	       new asebt::sc_rtl_2::fifo<data_width_g, fifo_depth_g>("fifo_n");
	    fifo_n->clk(clk_mesh);
	    fifo_n->rst_n(rst_n);
	    fifo_n->data_in(data_ctrl_fifo_r[N]);
	    fifo_n->we_in(we_tmp[N]);
	    fifo_n->full_out(full_from_fifo[N]);
	    fifo_n->data_out(data_from_fifo[N]);
	    fifo_n->re_in(re_in[N]);
	    fifo_n->empty_out(empty_from_fifo[N]);

	    fifo_n->one_d_out(dummy_one_d[N]);
	    fifo_n->one_p_out(dummy_one_p[N]);
	 }

	 if(row_addr_g == num_rows_g-1)
	 {
	    // Last row, don't connect south bridge
	    full_from_fifo[S].write(false);
	    data_from_fifo[S].write("0");
	    empty_from_fifo[S].write(true);
	 }
	 else
	 {
	    fifo_s = 
	       new asebt::sc_rtl_2::fifo<data_width_g, fifo_depth_g>("fifo_s");
	    fifo_s->clk(clk_mesh);
	    fifo_s->rst_n(rst_n);
	    fifo_s->data_in(data_ctrl_fifo_r[S]);
	    fifo_s->we_in(we_tmp[S]);
	    fifo_s->full_out(full_from_fifo[S]);
	    fifo_s->data_out(data_from_fifo[S]);
	    fifo_s->re_in(re_in[S]);
	    fifo_s->empty_out(empty_from_fifo[S]);

	    fifo_s->one_d_out(dummy_one_d[S]);
	    fifo_s->one_p_out(dummy_one_p[S]);
	 }

	 if(col_addr_g == 0)
	 {
	    // First column, don't connect west bridge
	    full_from_fifo[W].write(false);
	    data_from_fifo[W].write("0");
	    empty_from_fifo[W].write(true);
	 }
	 else
	 {
	    fifo_w = 
	       new asebt::sc_rtl_2::fifo<data_width_g, fifo_depth_g>("fifo_w");
	    fifo_w->clk(clk_mesh);
	    fifo_w->rst_n(rst_n);
	    fifo_w->data_in(data_ctrl_fifo_r[W]);
	    fifo_w->we_in(we_tmp[W]);
	    fifo_w->full_out(full_from_fifo[W]);
	    fifo_w->data_out(data_from_fifo[W]);
	    fifo_w->re_in(re_in[W]);
	    fifo_w->empty_out(empty_from_fifo[W]);

	    fifo_w->one_d_out(dummy_one_d[W]);
	    fifo_w->one_p_out(dummy_one_p[W]);
	 }

	 if(col_addr_g == num_cols_g-1)
	 {
	    // Last column, don't connect east bridge
	    full_from_fifo[E].write(false);
	    data_from_fifo[E].write("0");
	    empty_from_fifo[E].write(true);
	 }
	 else
	 {
	    fifo_e = 
	       new asebt::sc_rtl_2::fifo<data_width_g, fifo_depth_g>("fifo_e");
	    fifo_e->clk(clk_mesh);
	    fifo_e->rst_n(rst_n);
	    fifo_e->data_in(data_ctrl_fifo_r[E]);
	    fifo_e->we_in(we_tmp[E]);
	    fifo_e->full_out(full_from_fifo[E]);
	    fifo_e->data_out(data_from_fifo[E]);
	    fifo_e->re_in(re_in[E]);
	    fifo_e->empty_out(empty_from_fifo[E]);

	    fifo_e->one_d_out(dummy_one_d[E]);
	    fifo_e->one_p_out(dummy_one_p[E]);
	 }

	 // Data to IP
	 multiclk_fifo_rx.clk_re(clk_ip);
	 multiclk_fifo_rx.clk_we(clk_mesh);
	 multiclk_fifo_rx.rst_n(rst_n);
	 multiclk_fifo_rx.data_in(data_ctrl_fifo_r[IP]);
	 multiclk_fifo_rx.we_in(we_tmp[IP]);
	 multiclk_fifo_rx.full_out(full_from_fifo[IP]);
	 multiclk_fifo_rx.data_out(data_from_fifo[IP]);
	 multiclk_fifo_rx.re_in(re_ip_rx_in);
	 multiclk_fifo_rx.empty_out(empty_from_fifo[IP]);

	 multiclk_fifo_rx.one_d_out(dummy_one_d[IP]);
	 multiclk_fifo_rx.one_p_out(dummy_one_p[IP]);
    
	 // Data from IP
	 multiclk_fifo_tx.clk_re(clk_mesh);
	 multiclk_fifo_tx.clk_we(clk_ip);
	 multiclk_fifo_tx.rst_n(rst_n);
	 multiclk_fifo_tx.data_in(data_ip_tx_in);
	 multiclk_fifo_tx.we_in(we_ip_tx_in);
	 multiclk_fifo_tx.full_out(incoming_full[IP]);
	 multiclk_fifo_tx.data_out(incoming_data[IP]);
	 multiclk_fifo_tx.re_in(re_tmp[IP]);
	 multiclk_fifo_tx.empty_out(incoming_empty[IP]);

	 multiclk_fifo_tx.one_d_out(dummy_one_d[IP+1]);
	 multiclk_fifo_tx.one_p_out(dummy_one_p[IP+1]);

	 SC_METHOD(read_en_tmp);
	 for(unsigned int i = 0; i < 5; ++i)
	 {
	    sensitive << we_ctrl_fifo_r[i]
		      << re_r[i]
		      << data_reg_valid_r[i]
		      << incoming_empty[i]
		      << state_dst_r[i]
		      << state_src_r[i]
		      << full_from_fifo[i];
	 }
	
    
	 SC_METHOD(check_dst);
	 for(unsigned int i = 0; i < 5; ++i)
	 {
	    sensitive << state_reading_r[i]
		      << incoming_data[i]
		      << incoming_empty[i]
		      << incoming_full[i];
	 }

	 SC_METHOD(resolve_conflicts);
	 for(unsigned int i = 0; i < 5; ++i)
	 {
	    sensitive << curr_dst[i];
	 }

	 SC_METHOD(main_control);
	 sensitive << clk_mesh.pos()
		   << rst_n;

	 SC_METHOD(inputs);
	 for(unsigned int i = 0; i < 4; ++i)
	 {
	    sensitive << data_in[i]
		      << empty_in[i]
		      << full_in[i];
	 }

	 SC_METHOD(outputs);
	 for(unsigned int i = 0; i < 4; ++i)
	 {
	    sensitive << empty_from_fifo[i]
		      << full_from_fifo[i]
		      << re_tmp[i]
		      << data_from_fifo[i];
	 }
	 sensitive << empty_from_fifo[IP]
		   << full_from_fifo[IP]
		   << incoming_empty[IP]
		   << incoming_full[IP]
		   << data_from_fifo[IP];
    
      }


      //* Destructor
      virtual ~mesh_router()
      {
	 if(fifo_n) { delete fifo_n; }
	 if(fifo_w) { delete fifo_w; }
	 if(fifo_s) { delete fifo_s; }
	 if(fifo_e) { delete fifo_e; }
      }

      void outputs()
      {
	 empty_out[N].write(empty_from_fifo[N]);
	 empty_out[W].write(empty_from_fifo[W]);
	 empty_out[S].write(empty_from_fifo[S]);
	 empty_out[E].write(empty_from_fifo[E]);
	 empty_ip_rx_out.write(empty_from_fifo[IP]);

	 full_out[N].write(full_from_fifo[N]);
	 full_out[W].write(full_from_fifo[W]);
	 full_out[S].write(full_from_fifo[S]);
	 full_out[E].write(full_from_fifo[E]);
	 full_ip_rx_out.write(full_from_fifo[IP]);
    
	 empty_ip_tx_out.write(incoming_empty[IP]);
    
	 full_ip_tx_out.write(incoming_full[IP]);

	 re_out[N].write(re_tmp[N]);
	 re_out[W].write(re_tmp[W]);
	 re_out[S].write(re_tmp[S]);
	 re_out[E].write(re_tmp[E]);

	 data_out[N].write(data_from_fifo[N]);
	 data_out[W].write(data_from_fifo[W]);
	 data_out[S].write(data_from_fifo[S]);
	 data_out[E].write(data_from_fifo[E]);
	 data_ip_rx_out.write(data_from_fifo[IP]);

      }

      void inputs()
      {
	 (incoming_data[N]).write(data_in[N].read());
	 (incoming_data[W]).write(data_in[W].read());
	 (incoming_data[S]).write(data_in[S].read());
	 (incoming_data[E]).write(data_in[E].read());

	 (incoming_empty[N]).write(empty_in[N].read());
	 (incoming_empty[W]).write(empty_in[W].read());
	 (incoming_empty[S]).write(empty_in[S].read());
	 (incoming_empty[E]).write(empty_in[E].read());

	 (incoming_full[N]).write(full_in[N].read());
	 (incoming_full[W]).write(full_in[W].read());
	 (incoming_full[S]).write(full_in[S].read());
	 (incoming_full[E]).write(full_in[E].read());
      }


      void read_en_tmp()
      {
	 for(unsigned int i = 0; i < 5; ++i)
	 {
	    if(state_src_r[i].get_new_value() == NO_DIR_C)
	    {
	       we_tmp[i].write(false);
	    }
	    else
	    {
	       we_tmp[i].write(we_ctrl_fifo_r[i].get_new_value() &
			       data_reg_valid_r[i].get_new_value() &
			       (~full_from_fifo[i].get_new_value()));
	    }
      
	    if(state_dst_r[i].get_new_value() == NO_DIR_C)
	    {
	       re_tmp[i].write(false);
	    }
	    else
	    {
	       re_tmp[i].write(re_r[i].get_new_value() &
			       (~incoming_empty[i].get_new_value()) &
			       (~full_from_fifo[state_dst_r[i].get_new_value()].
				get_new_value()));
	    }
	 }
      }
    
      void check_dst()
      {
	 for(unsigned int i = 0; i < 5; ++i)
	 {
	    if(state_reading_r[i].read() == false &&
	       ((incoming_full[i].read() == true && stfwd_en_g == 1) ||
		(incoming_empty[i].read() == false && stfwd_en_g == 0)))
	    {
	       /*std::cout << "router_" << row_addr_g << "_" << col_addr_g << " :: "
			 << incoming_data[i].read().
		  range(addr_width_g-len_width_g-1, addr_width_g/2) << "+"
			 << incoming_data[i].read().
			 range(addr_width_g/2 - 1, 0) << std::endl;*/

	       if(incoming_data[i].read().
		  range(addr_width_g-len_width_g-1, addr_width_g/2).
		  to_uint() == row_addr_g)
	       {
		  if(incoming_data[i].read().
		     range(addr_width_g/2 - 1, 0).to_uint() == col_addr_g)
		  {
		     // Data heading here
		     curr_dst[i].write(IP);
		  }
		  else if(incoming_data[i].read().
			  range(addr_width_g/2 - 1, 0).to_uint() < col_addr_g)
		  {
		     curr_dst[i].write(W);
		  }
		  else
		  {
		     curr_dst[i].write(E);
		  }
	       }
	       else if(incoming_data[i].read().
		       range(addr_width_g-len_width_g-1, addr_width_g/2).to_uint()
		       < row_addr_g)
	       {
		  curr_dst[i].write(N);
	       }
	       else
	       {
		  curr_dst[i].write(S);
	       }
	    }
	    else
	    {
	       curr_dst[i].write(NO_DIR_C);
	    }
	 }
      }


      void resolve_conflicts()
      {
	 unsigned int n_req_v;

	 for(unsigned int src = 0; src < 5; ++src)
	 {
	    curr_dst_resolved[src].write(curr_dst[src].read());
	 }

	 for(unsigned int dst = 0; dst < 5; ++dst)
	 {
	    n_req_v = 0;
	    for(unsigned int src = 0; src < 5; ++src)
	    {
	       if(curr_dst[src].read() == dst)
	       {
		  if(n_req_v > 0)
		  {
		     curr_dst_resolved[src].write(NO_DIR_C);
		  }
		  n_req_v++;
	       }
	    }
	    n_req_dbgr[dst].write(n_req_v);
	 }
      }


      void main_control()
      {    

	 if(rst_n.read() == false)
	 {
	    // Active low reset
	    for(unsigned int i = 0; i < 5; ++i)
	    {
	       data_ctrl_fifo_r[i].write("0");
	       send_counter_r[i].write(0);
	       pkt_len_arr_r[i].write(pkt_len_g);

	       we_ctrl_fifo_r[i].write(false);
	       re_r[i].write(false);
	       state_writing_r[i].write(false);
	       state_reading_r[i].write(false);
	       state_src_r[i].write(NO_DIR_C);
	       state_dst_r[i].write(NO_DIR_C);

	       data_reg_valid_r[i].write(false);
	    }	
	 }
	 else // rising clock edge
	 {
	    for(unsigned int i = 0; i < 5; ++i)
	    {
	       if(state_reading_r[i].read() == true)
	       {
		  if(send_counter_r[i].read() == pkt_len_arr_r[i].read())
		  {
		     if(full_from_fifo[state_dst_r[i].read()].read() == false)
		     {
			we_ctrl_fifo_r[state_dst_r[i].read()].write(false);
			re_r[i].write(false);
			state_writing_r[state_dst_r[i].read()]
			   .write(false);
			state_reading_r[i].write(false);
			state_src_r[state_dst_r[i].read()].write(NO_DIR_C);
			state_dst_r[i].write(NO_DIR_C);
			send_counter_r[i].write(0);
			data_ctrl_fifo_r[state_dst_r[i].read()].write('X');
			data_reg_valid_r[state_dst_r[i].read()]
			   .write(false);
			pkt_len_arr_r[i].write(pkt_len_g);
		     }
		  }
		  else
		  {
		     we_ctrl_fifo_r[state_dst_r[i].read()]
			.write(true);
		    

		     if(data_reg_valid_r[state_dst_r[i].read()].read() == false)
		     {
			if(re_tmp[i].get_new_value() == true)
			{
			   data_reg_valid_r[state_dst_r[i].read()]
			      .write(true);
			   send_counter_r[i].write(send_counter_r[i].read()+1);
			   data_ctrl_fifo_r[state_dst_r[i].read()].
			      write(incoming_data[i].read());
			}
			else
			{
			   data_reg_valid_r[state_dst_r[i].read()]
			      .write(false);
			}
		     }
		     else
		     {
			if(full_from_fifo[state_dst_r[i].read()].read() == false)
			{
			   if(re_tmp[i].get_new_value() == true)
			   {
			      data_reg_valid_r[state_dst_r[i].read()].
				 write(true);
			      send_counter_r[i].
				 write(send_counter_r[i].read()+1);
			      data_ctrl_fifo_r[state_dst_r[i].read()].
				 write(incoming_data[i].read());
			   }
			   else
			   {
			      data_reg_valid_r[state_dst_r[i].read()].
				 write(false);
			   }
			}
			else
			{
			   data_reg_valid_r[state_dst_r[i].read()].
			      write(true);
			}
		     }

		     if(stfwd_en_g == 1)
		     {
			pkt_len_arr_r[i].write(pkt_len_g);
		     }
		     else
		     {
			if(len_flit_en_g == 0 && 
			   re_tmp[i].get_new_value() == true &&
			   send_counter_r[i].read() == 0)
			{
			   pkt_len_arr_r[i].
			      write(incoming_data[i].read().
				    range(data_width_g-1, 
					  data_width_g-len_width_g).to_uint() +
				    len_flit_en_g + 1);
			}
			else if(len_flit_en_g == 1 &&
				re_tmp[i].get_new_value() == true &&
				send_counter_r[i].read() == 1)
			{
			   pkt_len_arr_r[i].
			      write(incoming_data[i].read().
				    range(len_width_g-1, 
					  0).to_uint() +
				    len_flit_en_g + 1);
			}  
		     }

		     if(send_counter_r[i].read() == pkt_len_arr_r[i].read()-1 &&
			incoming_empty[i].read() == false &&
			full_from_fifo[state_dst_r[i].read()].read() == false)
		     {
			re_r[i].write(false);
		     }
		     else
		     {
			re_r[i].write(true);
		     }		    		    
		  }
	       }
	       else
	       {
		  if(curr_dst_resolved[i].get_new_value() != NO_DIR_C &&
		     empty_from_fifo[curr_dst_resolved[i].get_new_value()].
		     read() == true &&
		     state_writing_r[curr_dst_resolved[i].get_new_value()].
		     read() == 0)
		  {
		     we_ctrl_fifo_r[curr_dst_resolved[i].get_new_value()].
			write(false);
		     state_writing_r[curr_dst_resolved[i].get_new_value()].
			write(true);
		     re_r[i].write(true);
		     state_reading_r[i].write(true);
		     state_src_r[curr_dst_resolved[i].get_new_value()].
			write(i);
		     state_dst_r[i].write(curr_dst_resolved[i].get_new_value());
		     data_ctrl_fifo_r[curr_dst_resolved[i].get_new_value()].
			write(incoming_data[i].read());
		     data_reg_valid_r[curr_dst_resolved[i].get_new_value()].
			write(true);
		  }


	       }
	    }
	 }        
      }

   private:

      asebt::sc_rtl_2::fifo<data_width_g, fifo_depth_g>* fifo_n;
      asebt::sc_rtl_2::fifo<data_width_g, fifo_depth_g>* fifo_s;
      asebt::sc_rtl_2::fifo<data_width_g, fifo_depth_g>* fifo_w;
      asebt::sc_rtl_2::fifo<data_width_g, fifo_depth_g>* fifo_e;

      asebt::sc_rtl_2::multiclk_fifo
      <ip_freq_g, mesh_freq_g, fifo_depth_g, data_width_g> multiclk_fifo_rx;

      asebt::sc_rtl_2::multiclk_fifo
      <mesh_freq_g, ip_freq_g, fifo_depth_g, data_width_g> multiclk_fifo_tx;

      sc_core::sc_signal<sc_dt::sc_uint<16> > pkt_len_arr_r[5];
      sc_core::sc_signal<sc_dt::sc_bv<data_width_g> > incoming_data[5];
      sc_core::sc_signal<bool> incoming_empty[5];
      sc_core::sc_signal<bool> incoming_full[5];
  
      sc_core::sc_signal<sc_dt::sc_bv<data_width_g> > data_from_fifo[5];
      sc_core::sc_signal<bool> full_from_fifo[5];
      sc_core::sc_signal<bool> empty_from_fifo[5];

      sc_core::sc_signal<sc_dt::sc_bv<data_width_g> > data_ctrl_fifo_r[5];
      sc_core::sc_signal<bool> we_ctrl_fifo_r[5];
      sc_core::sc_signal<bool> re_r[5];
      sc_core::sc_signal<sc_dt::sc_uint<16> > send_counter_r[5];

      sc_core::sc_signal<bool> re_tmp[5];
      sc_core::sc_signal<bool> we_tmp[5];

      sc_core::sc_signal<bool> data_reg_valid_r[5];
  
      sc_core::sc_signal<bool> state_writing_r[5];
      sc_core::sc_signal<bool> state_reading_r[5];

      sc_core::sc_signal<sc_dt::sc_uint<3> > state_src_r[5];
      sc_core::sc_signal<sc_dt::sc_uint<3> > state_dst_r[5];

      sc_core::sc_signal<sc_dt::sc_uint<3> > curr_src_r;

      sc_core::sc_signal<sc_dt::sc_uint<3> > curr_dst[5];
      sc_core::sc_signal<sc_dt::sc_uint<3> > curr_dst_resolved[5];
      sc_core::sc_signal<sc_dt::sc_uint<3> > n_req_dbgr[5];

      sc_core::sc_signal<bool> dummy_one_d[6];
      sc_core::sc_signal<bool> dummy_one_p[6];

      int col_addr_g;
      int row_addr_g;

   };

}
}

#endif



// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
