/**
 *
 * @file fifo.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: fifo.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#ifndef ASEBT_SC_RTL_1_FIFO_HH
#define ASEBT_SC_RTL_1_FIFO_HH

#include <systemc>
#include <cmath>

namespace asebt
{
namespace sc_rtl_1
{

   template< int data_width_g, int depth_g>
   class fifo : public sc_core::sc_module
   {
   public:
      sc_core::sc_in_clk clk;
      sc_core::sc_in<sc_dt::sc_logic> rst_n;

      sc_core::sc_in<sc_dt::sc_lv<data_width_g> > data_in;
      sc_core::sc_in<sc_dt::sc_logic> we_in;
      sc_core::sc_out<sc_dt::sc_logic> full_out;
      sc_core::sc_out<sc_dt::sc_logic> one_p_out;
  
      sc_core::sc_in<sc_dt::sc_logic> re_in;
      sc_core::sc_out<sc_dt::sc_lv<data_width_g> > data_out;
      sc_core::sc_out<sc_dt::sc_logic> empty_out;
      sc_core::sc_out<sc_dt::sc_logic> one_d_out;

      SC_HAS_PROCESS(fifo);

      fifo(sc_core::sc_module_name name)
	 : sc_core::sc_module(name),
	   clk("clk"),
	   rst_n("rst_n"),
	   data_in("data_in"),
	   we_in("we_in"),
	   full_out("full_out"),
	   one_p_out("one_p_out"),
	   re_in("re_in"),
	   data_out("data_out"),
	   empty_out("empty_out"),
	   one_d_out("one_d_out"),
	   full_r("full_r"),
	   empty_r("empty_r"),
	   one_d_r("one_d_r"),
	   one_p_r("one_p_r"),
	   data_amount_r("data_amount_r"),
	   in_ptr_r("in_ptr_r"),
	   out_ptr_r("out_ptr_r")
      {
	 for(unsigned int i = 0; i < depth_g; ++i)
	 {
	    fifo_buffer_r[i].write("0");
	 }

	 SC_METHOD(method);
	 sensitive << clk.pos() << rst_n;

	 SC_METHOD(output);
	 sensitive << out_ptr_r << full_r
		   << empty_r << one_d_r
		   << one_p_r;
	 for(unsigned int i = 0; i < depth_g; ++i)
	 {
	    sensitive << fifo_buffer_r[i];
	 }
      }
  
      virtual ~fifo()
      {
      }

      void output()
      {
	 data_out.write(fifo_buffer_r[out_ptr_r.read()].read());
	 full_out.write(full_r);
	 empty_out.write(empty_r);
	 one_d_out.write(one_d_r);
	 one_p_out.write(one_p_r);
      }

      void method()
      {
	 if(rst_n.read() == 0)
	 {
	    full_r.write(sc_dt::SC_LOGIC_0);
	    empty_r = sc_dt::SC_LOGIC_1;
	    one_d_r = sc_dt::SC_LOGIC_0;
	    in_ptr_r = 0;
	    out_ptr_r = 0;
	    data_amount_r = 0;
	    one_p_r = depth_g == 1 ? sc_dt::SC_LOGIC_1 : sc_dt::SC_LOGIC_0;
	 }
	 else
	 {	
	    if(we_in.read() == sc_dt::SC_LOGIC_1 && 
	       re_in.read() == sc_dt::SC_LOGIC_0)
	    {
	       if(full_r.read() == sc_dt::SC_LOGIC_0)
	       {
		  empty_r = sc_dt::SC_LOGIC_0;
		  if(in_ptr_r.read() == depth_g - 1) 
		  { in_ptr_r = 0; }
		  else 
		  { in_ptr_r = in_ptr_r.read() + 1; }
		  data_amount_r = data_amount_r.read() + 1;
		  fifo_buffer_r[in_ptr_r.read()] = data_in;
	       }

	       if(data_amount_r.read() + 2 == depth_g)
	       {
		  full_r = sc_dt::SC_LOGIC_0;
		  one_p_r = sc_dt::SC_LOGIC_1;
	       }
	       else if(data_amount_r.read() + 1 == depth_g)
	       {
		  full_r = sc_dt::SC_LOGIC_1;
		  one_p_r = sc_dt::SC_LOGIC_0;
	       }
	       else
	       {
		  full_r = sc_dt::SC_LOGIC_0;
		  one_p_r = sc_dt::SC_LOGIC_0;
	       }

	       if(empty_r.read() == 1)
	       { one_d_r = sc_dt::SC_LOGIC_1; }
	       else
	       { one_d_r = sc_dt::SC_LOGIC_0; }	  

	    }
      
	    else if(we_in.read() == sc_dt::SC_LOGIC_0 && re_in.read() == 1)
	    {
	       if(empty_r.read() == sc_dt::SC_LOGIC_0)
	       {
		  if(out_ptr_r.read() == depth_g - 1)
		  {
		     out_ptr_r = 0;
		  }
		  else
		  {
		     out_ptr_r = out_ptr_r.read() + 1;
		  }
	      
		  full_r = sc_dt::SC_LOGIC_0;
		  data_amount_r = data_amount_r.read() - 1;
	      
		  if(data_amount_r.read() == 2)
		  {
		     empty_r = sc_dt::SC_LOGIC_0;
		     one_d_r = sc_dt::SC_LOGIC_1;
		  }
		  else if(data_amount_r.read() == 1)
		  {
		     empty_r = sc_dt::SC_LOGIC_1;
		     one_d_r = sc_dt::SC_LOGIC_0;
		  }
		  else
		  {
		     empty_r = sc_dt::SC_LOGIC_0;
		     one_d_r = sc_dt::SC_LOGIC_0;
		  }
	      
		  if(full_r.read() == sc_dt::SC_LOGIC_1)
		  { one_p_r = sc_dt::SC_LOGIC_1; }
		  else
		  { one_p_r = sc_dt::SC_LOGIC_0; }	    
	      
	       }
	    }
	    else if(we_in.read() == sc_dt::SC_LOGIC_1 && 
		    re_in.read() == sc_dt::SC_LOGIC_1)
	    {
	       if(full_r.read() == sc_dt::SC_LOGIC_0 && 
		  empty_r == sc_dt::SC_LOGIC_0)
	       {
		  if(in_ptr_r.read() == depth_g - 1)
		  { in_ptr_r = 0; }
		  else 
		  { in_ptr_r = in_ptr_r.read() + 1;}
	      
		  if(out_ptr_r.read() == depth_g - 1)
		  { out_ptr_r = 0; }
		  else 
		  { out_ptr_r = out_ptr_r.read() + 1;}
	      
		  full_r = sc_dt::SC_LOGIC_0;
		  empty_r = sc_dt::SC_LOGIC_0;
	      
		  fifo_buffer_r[in_ptr_r.read()] = data_in;
	       }
	       else if(full_r.read() == sc_dt::SC_LOGIC_1 && 
		       empty_r.read() == sc_dt::SC_LOGIC_0)
	       {
		  if(out_ptr_r.read() == depth_g - 1)
		  { out_ptr_r = 0; }
		  else 
		  { out_ptr_r = out_ptr_r.read() + 1;}
	      
		  full_r = sc_dt::SC_LOGIC_0;
		  one_p_r = sc_dt::SC_LOGIC_1;
		  data_amount_r = data_amount_r.read() - 1;
	      
		  if(data_amount_r.read() == 2)
		  {
		     empty_r = sc_dt::SC_LOGIC_0;
		     one_d_r = sc_dt::SC_LOGIC_1;
		  }
		  else if(data_amount_r.read() == 1)
		  {
		     empty_r = sc_dt::SC_LOGIC_1;
		     one_d_r = sc_dt::SC_LOGIC_0;
		  }
		  else
		  {
		     empty_r = sc_dt::SC_LOGIC_0;
		     one_d_r = sc_dt::SC_LOGIC_0;
		  }
	       }
	       else if(full_r.read() == sc_dt::SC_LOGIC_0 && 
		       empty_r.read() == sc_dt::SC_LOGIC_1)
	       {
		  if(in_ptr_r.read() == depth_g - 1)
		  { in_ptr_r = 0; }
		  else 
		  { in_ptr_r = in_ptr_r.read() + 1;}
	      
		  empty_r = sc_dt::SC_LOGIC_0;
		  one_d_r = sc_dt::SC_LOGIC_1;
		  fifo_buffer_r[in_ptr_r.read()] = data_in;
		  data_amount_r = data_amount_r.read() + 1;
	      
		  if(data_amount_r.read() == 2)
		  {
		     empty_r = sc_dt::SC_LOGIC_0;
		     one_d_r = sc_dt::SC_LOGIC_1;
		  }
		  else if(data_amount_r.read() == 1)
		  {
		     empty_r = sc_dt::SC_LOGIC_1;
		     one_d_r = sc_dt::SC_LOGIC_0;
		  }
		  else
		  {
		     empty_r = sc_dt::SC_LOGIC_0;
		     one_d_r = sc_dt::SC_LOGIC_0;
		  }
	       }
	    }
	 }    
      }


   private:

      sc_core::sc_signal<sc_dt::sc_logic> full_r;
      sc_core::sc_signal<sc_dt::sc_logic> empty_r;
      sc_core::sc_signal<sc_dt::sc_logic> one_d_r;
      sc_core::sc_signal<sc_dt::sc_logic> one_p_r;
      sc_core::sc_signal<sc_dt::sc_uint<16> > data_amount_r;
      sc_core::sc_signal<sc_dt::sc_uint<16> > in_ptr_r;
      sc_core::sc_signal<sc_dt::sc_uint<16> > out_ptr_r;

      sc_core::sc_signal<sc_dt::sc_lv<data_width_g> > fifo_buffer_r[depth_g];

   };

}
}


#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
