/**
 * CVS:
 * $Id: SettingsDialog.java 1399 2010-08-26 13:56:45Z lehton87 $
 * 
 * File:    SettingsDialog.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 18.12.2006
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui;

import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;
import java.util.ResourceBundle;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

/**
 * A dialog for viewing and editing different settings.
 */
public class SettingsDialog extends JDialog {
	private static final int TEXT_FIELD_WIDTH = 22;
	private static final int PANEL_VGAP = 5;
	private static final int PANEL_HGAP = 5;
	private static final int CELL_VGAP = 4;
	private static final int CELL_HGAP = 2;
	private static final String SELECT_DOTDIR_CMD = "SelectDotDir";
	private static final String SELECT_LOGDIR_CMD = "SelectLogDir";
	private static final String SELECT_TRACEDIR_CMD = "SelectTraceDir";
	private static final String OK_CMD = "Ok";
	private static final String CANCEL_CMD = "Cancel";
	
	protected static SettingsDialog self;

	protected String dotDirectoryPath;
	protected JTextField dotDirField;
	private JButton selectDotDirButton;
	
	protected String logDirectoryPath;
	protected JTextField logDirField;
	private JButton selectLogDirButton;
	
	protected String traceDirectoryPath;	
	protected JTextField traceDirField;
	private JButton selectTraceDirButton;
	
	private Properties userSettings;
	
	
    public static SettingsDialog getInstance() {
        if (self == null) {
            self = new SettingsDialog();
        }
        return self;
    }
    
    
    /**
     * Creates a new dialog. 
     */
    private SettingsDialog() {
        super(MainWindow.getInstance(), true);
        init();
    }
    
    
    /**
     * Builds the dialog.
     */
    private void init() {
    	ResourceBundle bundle = MainWindow.bundle;
    	setTitle(bundle.getString("SETTINGS_TITLE"));
    	setLayout(new GridBagLayout());
    	setResizable(false);
    	
    	MyActionListener actionListener = new MyActionListener();
    	
    	loadSettings();
    	
    	// Create directory selection panel
    	JPanel dirPanel = new JPanel();
    	dirPanel.setLayout(new GridBagLayout());

		dotDirField = new JTextField(dotDirectoryPath, TEXT_FIELD_WIDTH);
		selectDotDirButton = new JButton(bundle.getString("BROWSE"));
		selectDotDirButton.setActionCommand(SELECT_DOTDIR_CMD);
		selectDotDirButton.addActionListener(actionListener);

		dirPanel.add(new JLabel(bundle.getString("SETTINGS_DOTDIR_LABEL")),
				new GridBagConstraints(0, 0, 1, 1, 0.0, 0.0,
				GridBagConstraints.WEST, GridBagConstraints.NONE,
				new Insets(CELL_VGAP, CELL_HGAP, CELL_VGAP, CELL_HGAP), 0, 0));
		dirPanel.add(dotDirField,
				new GridBagConstraints(1, 0, 1, 1, 0.0, 0.0,
				GridBagConstraints.WEST, GridBagConstraints.NONE,
				new Insets(CELL_VGAP, CELL_HGAP, CELL_VGAP, CELL_HGAP), 0, 0));
		dirPanel.add(selectDotDirButton,
				new GridBagConstraints(2, 0, GridBagConstraints.REMAINDER, 1, 0.0, 0.0,
				GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL,
				new Insets(CELL_VGAP, CELL_HGAP, CELL_VGAP, CELL_HGAP), 0, 0));
    	
		logDirField = new JTextField(logDirectoryPath, TEXT_FIELD_WIDTH);
		selectLogDirButton = new JButton(bundle.getString("BROWSE"));
		selectLogDirButton.setActionCommand(SELECT_LOGDIR_CMD);
		selectLogDirButton.addActionListener(actionListener);

		dirPanel.add(new JLabel(bundle.getString("SETTINGS_LOGDIR_LABEL")),
				new GridBagConstraints(0, 1, 1, 1, 0.0, 0.0,
				GridBagConstraints.WEST, GridBagConstraints.NONE,
				new Insets(CELL_VGAP, CELL_HGAP, CELL_VGAP, CELL_HGAP), 0, 0));
		dirPanel.add(logDirField,
				new GridBagConstraints(1, 1, 1, 1, 0.0, 0.0,
				GridBagConstraints.WEST, GridBagConstraints.NONE,
				new Insets(CELL_VGAP, CELL_HGAP, CELL_VGAP, CELL_HGAP), 0, 0));
		dirPanel.add(selectLogDirButton,
				new GridBagConstraints(2, 1, GridBagConstraints.REMAINDER, 1, 0.0, 0.0,
				GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL,
				new Insets(CELL_VGAP, CELL_HGAP, CELL_VGAP, CELL_HGAP), 0, 0));
		
    	traceDirField = new JTextField(traceDirectoryPath, TEXT_FIELD_WIDTH);
		selectTraceDirButton = new JButton(bundle.getString("BROWSE"));
		selectTraceDirButton.setActionCommand(SELECT_TRACEDIR_CMD);
		selectTraceDirButton.addActionListener(actionListener);
		
		dirPanel.add(new JLabel(bundle.getString("SETTINGS_TRACEDIR_LABEL")),
				new GridBagConstraints(0, 2, 1, 1, 0.0, 0.0,
				GridBagConstraints.WEST, GridBagConstraints.NONE,
				new Insets(CELL_VGAP, CELL_HGAP, CELL_VGAP, CELL_HGAP), 0, 0));			
		dirPanel.add(traceDirField,
				new GridBagConstraints(1, 2, 1, 1, 0.0, 0.0,
				GridBagConstraints.WEST, GridBagConstraints.NONE,
				new Insets(CELL_VGAP, CELL_HGAP, CELL_VGAP, CELL_HGAP), 0, 0));			
		dirPanel.add(selectTraceDirButton,
				new GridBagConstraints(2, 2, GridBagConstraints.REMAINDER, 1, 0.0, 0.0,
				GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL,
				new Insets(CELL_VGAP, CELL_HGAP, CELL_VGAP, CELL_HGAP), 0, 0));	
		
		
		// Create buttons panel
		JPanel buttonsPanel = new JPanel();
		buttonsPanel.setLayout(new FlowLayout(FlowLayout.RIGHT));
		
		JButton okButton = new JButton(bundle.getString("OK"));
		okButton.setActionCommand(OK_CMD);
		okButton.addActionListener(actionListener);		

		JButton cancelButton = new JButton(bundle.getString("CANCEL"));
		cancelButton.setActionCommand(CANCEL_CMD);
		cancelButton.addActionListener(actionListener);		

		buttonsPanel.add(okButton);
		buttonsPanel.add(cancelButton);
		
		add(dirPanel,
				new GridBagConstraints(0, 0, GridBagConstraints.REMAINDER, 1, 1.0, 0.0,
				GridBagConstraints.NORTHWEST, GridBagConstraints.NONE,
				new Insets(PANEL_VGAP, PANEL_HGAP, PANEL_VGAP, PANEL_HGAP), 0, 0));

		add(buttonsPanel,
				new GridBagConstraints(0, 4, GridBagConstraints.REMAINDER, GridBagConstraints.REMAINDER, 1.0, 1.0,
				GridBagConstraints.NORTHEAST, GridBagConstraints.NONE,
				new Insets(PANEL_VGAP, PANEL_HGAP, PANEL_VGAP, PANEL_HGAP), 0, 0));
		
		pack();
    }
    
    
    /**
     * Load user settings from file
     */
    private void loadSettings() {
    	userSettings = new Properties();
    	File file = new File("user.settings");
    	
    	try {
    		FileInputStream in = new FileInputStream(file);
			userSettings.load(in);
			in.close();
			
    	} catch (FileNotFoundException e) {
			try {
				file.createNewFile();
			} catch (IOException e1) {
				e1.printStackTrace();
			}
			
		} catch (Exception e) {
			e.printStackTrace();
		}
		
		dotDirectoryPath = userSettings.getProperty("dotDirectory", "");
		logDirectoryPath = userSettings.getProperty("logDirectory", "");
		traceDirectoryPath = userSettings.getProperty("traceDirectory", "");
    }
    
    
    /**
     * Save user settings to file
     */
    private void saveSettings() {
		userSettings.setProperty("dotDirectory", dotDirectoryPath);
		userSettings.setProperty("logDirectory", logDirectoryPath);
		userSettings.setProperty("traceDirectory", traceDirectoryPath);
		
    	try {
    		FileOutputStream out = new FileOutputStream("user.settings");
    		userSettings.store(out, "Execution Monitor user settings");
    		out.close();
    	} catch (Exception e) {
			e.printStackTrace();
		}
    }
 
    
    /**
     * Returns the directory path of the external dot graph program.
     */
    public String getDotDirectoryPath() {
    	return dotDirectoryPath;
    }
    
    
    /**
     * Returns the log directory path.
     */
    public String getLogDirectoryPath() {
    	return logDirectoryPath;
    }
    
    
    /**
     * Returns the trace directory path.
     */
    public String getTraceDirectoryPath() {
    	return traceDirectoryPath;
    }
    
    
    private class MyActionListener implements ActionListener {
    	public void actionPerformed(ActionEvent e) {
    		String cmd = e.getActionCommand();
    		
    		if (cmd.equals(OK_CMD)) {
    			dotDirectoryPath = dotDirField.getText();
    			logDirectoryPath = logDirField.getText();
    			traceDirectoryPath = traceDirField.getText();
    			saveSettings();
    			setVisible(false);
    				
    		} else if (cmd.equals(CANCEL_CMD)) {
    			loadSettings();
    			dotDirField.setText(dotDirectoryPath);
    			logDirField.setText(logDirectoryPath);
    			traceDirField.setText(traceDirectoryPath);
    			setVisible(false);
    		
    		} else if (cmd.equals(SELECT_DOTDIR_CMD)) {
    			// Ask user to select dot directory 
    			JFileChooser fileChooser = new JFileChooser(dotDirField.getText());	
    			fileChooser.setDialogTitle(MainWindow.bundle.getString("SELECT_DIR_DIALOG_TITLE"));
    			fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
    			int returnVal = fileChooser.showSaveDialog(self);
    		
    			if (returnVal == JFileChooser.APPROVE_OPTION) {
    				File selected = fileChooser.getSelectedFile();
    				dotDirField.setText(selected.getAbsolutePath());
    			}
    		
    		} else if (cmd.equals(SELECT_LOGDIR_CMD)) {
    			// Ask user to select a log directory
    			JFileChooser fileChooser = new JFileChooser(logDirField.getText());	
    			fileChooser.setDialogTitle(MainWindow.bundle.getString("SELECT_DIR_DIALOG_TITLE"));
    			fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
    			int returnVal = fileChooser.showSaveDialog(self);
    		
    			if (returnVal == JFileChooser.APPROVE_OPTION) {
    				File selected = fileChooser.getSelectedFile();
    				logDirField.setText(selected.getAbsolutePath());
    			}
    		
			} else if (cmd.equals(SELECT_TRACEDIR_CMD)) {
				// Ask user to select a trace directory
				JFileChooser fileChooser = new JFileChooser(traceDirField.getText());	
				fileChooser.setDialogTitle(MainWindow.bundle.getString("SELECT_DIR_DIALOG_TITLE"));
				fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
				int returnVal = fileChooser.showSaveDialog(self);
			
				if (returnVal == JFileChooser.APPROVE_OPTION) {
					File selected = fileChooser.getSelectedFile();
					traceDirField.setText(selected.getAbsolutePath());
				}
			}
    	}
    }
}
