/**
 * CVS:
 * $Id: ChartDataLogger.java 1399 2010-08-26 13:56:45Z lehton87 $
 * 
 * File:    ChartDataLogger.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 14.12.2006
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.data;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.text.DecimalFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import fi.cpu.ui.graph.GraphPanel;

/**
 * ChartDataLogger writes the data in charts to files.
 */
public class ChartDataLogger {
	private static ChartDataLogger self;
	private File parentDirectory;
	private File logDirectory;
	private boolean loggingEnabled;
	private HashMap<GraphPanel, PrintStream> fileStreams;
	private String errorMessage;

	
	public static ChartDataLogger getInstance() {
		if (self == null) {
			self = new ChartDataLogger();
		}
		return self;
	}
	
	
	/**
	 * Creates a new ChartDataLogger.
	 */
	public ChartDataLogger() {
    	fileStreams = new HashMap<GraphPanel, PrintStream>();
		loggingEnabled = false;
	}
	
	
	/**
	 * Sets the directory where log files are created.
	 */
	public void setLogDirectory(File directory) throws IOException {
		directory.mkdir();
		parentDirectory = directory;
	}
	
	
	/**
	 * Returns the directory where log files are created.
	 */
	public File getLogDirectory() {
		return parentDirectory;
	}
	
	
	/**
	 * Returns the message of an error during logging. If no error happened,
	 * null returned.
	 */
	public String getErrorMessage() {
		return errorMessage;
	}
	
	
	/**
	 * Begins logging.
	 */
	public File startLogging() throws IOException {
		// Get current date and time
    	long time = System.currentTimeMillis();
    	Calendar calendar = Calendar.getInstance();
    	calendar.setTimeInMillis(time);

    	int year = calendar.get(Calendar.YEAR);
    	int month = calendar.get(Calendar.MONTH);
    	int day = calendar.get(Calendar.DATE);
    	int hour = calendar.get(Calendar.HOUR_OF_DAY);
    	int min = calendar.get(Calendar.MINUTE);
    	int sec = calendar.get(Calendar.SECOND);

    	DecimalFormat format = new DecimalFormat("00"); // two digits
    	
    	StringBuilder name = new StringBuilder();
    	name.append(year);
    	name.append(format.format(month));
    	name.append(format.format(day));
    	name.append("_");
    	name.append(format.format(hour));
    	name.append(format.format(min));
    	name.append(format.format(sec));

    	logDirectory = new File(parentDirectory, name.toString());
    	logDirectory.mkdir();
		loggingEnabled = true;
		errorMessage = null;
		
		return logDirectory;
	}
	
	
	/**
	 * Stops logging.
	 */
	public void stopLogging() {
		loggingEnabled = false;
		Collection<PrintStream> streams = fileStreams.values();
		Iterator<PrintStream> iterator = streams.iterator();
		
		// Close all opened streams
		while (iterator.hasNext()) {
			PrintStream stream = iterator.next();
			stream.close();
		}

		fileStreams.clear();
		logDirectory = null;
	}
	
	
	/**
	 * Adds data to log.
	 */
	public void logData(GraphPanel graph, String xvalue, String yvalue) throws IOException {
		if (!loggingEnabled || logDirectory == null || graph == null) {
			return;
		}
		
		try {
			PrintStream stream = fileStreams.get(graph);
			
			if (stream == null) {
				// create new stream
				File logFile = new File(logDirectory, graph.getLogName());
				if (logFile.exists()) {
					logFile.delete();
				}
				
				logFile.createNewFile();
				stream = new PrintStream(logFile);
				fileStreams.put(graph, stream);
			}
			stream.println(xvalue + "\t" + yvalue);

		} catch (IOException e) {
			errorMessage = "Couldn't log data to: "+logDirectory.getAbsolutePath()+": "+e.getMessage();
			throw e;
		}
	}
}
