// 
//  Copyright 2003 OCP-IP
//  OCP-IP Confidential & Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level
//       Author : Norman Weyrich, Synopsys Inc.
//                (modified by Joe Chou, Sonics Inc.)
//         Date : 03/25/2003
//
//  Description : Transaction Level - Layer-2 Bus
//                Header file for the example implementation of the 
//                Layer-2 Bus. See the ocp_tl2_bus.cpp file for more
//                information.
//
// ============================================================================

#ifndef _OCP_TL2_BUS_H
#define _OCP_TL2_BUS_H

#include "ocp_tl2_globals.h"
#include "tl_master_if.h"
#include "tl_slave_if.h"

// Some constants.

// Large number, used in minimum search
#define TL2_MAX_DOUBLE 1.0e+37  

// Maximum number of Masters which can beattached to the Bus
#define TL2_MAX_MASTERS     32

// Maximum number of Slaves
#define TL2_MAX_SLAVES      32 

// Length of the arbitration register
#define TL2_ARB_REG_LENGTH  TL2_MAX_MASTERS + 1  

// Index in the arbitration register for the Master ID who has the
// default grant
#define TL2_DF_GRANT_ID     TL2_MAX_MASTERS 

// Don't change them, multiple memory regions per Slave are currently
// not supported
#define TL2_MAX_REGIONS     1
#define TL2_BOOT_MODE       0
#define TL2_NORMAL_MODE     1
#define TL2_MAX_MODES       1

// Address allignment of the Slave start address
// A value of 10 means 1K (2**10) allignment is assumed
#define TL2_ADDR_ALLIGNMENT 10
#define TL2_SADDR_MASK      (1 << TL2_ADDR_ALLIGNMENT) - 1

// Index for the dummy Master (don't change)
#define TL2_DUMMY_MASTER    TL2_MAX_MASTERS

// Switch to use a cache table for finding out the addressed Slave.
// If the address range in the system is not exorbitant large, it
// is recommended to use a cache table. It's a memory consumptions
// versus simulation speed trade off.
#define TL2_USE_MEM_CACHE

template<class TdataClSport, class TdataClMport> class OCP_TL2_Bus
  : public sc_module,
    public MdirectIF<TdataClMport>,
    public SdirectIF<TdataClSport>
{
 public:

  // We assume the same templatized data and address type for both
  // template classes TdataClSport and TdataClMport.
  typedef typename TdataClSport::DataType Td;
  typedef typename TdataClSport::AddrType Ta;

  // We use here multi ports.
  // Note that the Slave port is connected to master modules, while
  // the Master port is connected to Slaves.
  sc_port< TLslaveIF<TdataClSport>, TL2_MAX_MASTERS > SlaveP;
  sc_port< TLmasterIF<TdataClMport>, TL2_MAX_SLAVES > MasterP;

  //
  SC_HAS_PROCESS(OCP_TL2_Bus);

  // constructor
  OCP_TL2_Bus(sc_module_name name_, int BusID, bool Pipelined = false,
    bool WriteResponse = true, int ReadWaitCycles = 0, int WriteWaitCycles = 0);

  // destructor
  ~OCP_TL2_Bus();

  // methods
  void Bus();
  void BusRequest();
  void BusResponse();
  void BusError();
  void Initialize();
  int  AddressDecoding(Ta);
  int  Arbitration();
  int  SetConnID(int GrantedMaster, int AddressedSlave);
  int  GetMasterID(int SConnID, int AddressedSlave);
  void ForwardRequest(int, int);
  void ForwardResponse(int, int);
  void PrintTime(const char* Pstring, int ID, Ta Addr);

  //
  virtual bool MputDirect(int, bool, Td*, Ta, int);
  virtual bool SputDirect(int, bool, Td*, Ta, int);

private:

  //
  void end_of_elaboration();

  // Pointer to the user defined data structures of the Channel
  TdataClSport* m_SdataCl[TL2_MAX_MASTERS];
  TdataClMport* m_MdataCl[TL2_MAX_SLAVES];

  ParamCl<TdataClSport>* m_SparamCl[TL2_MAX_MASTERS];
  ParamCl<TdataClMport>* m_MparamCl[TL2_MAX_SLAVES];

  CommCl* m_ScommCl[TL2_MAX_MASTERS];
  CommCl* m_McommCl[TL2_MAX_SLAVES];

  // Mapping from Slave port number to the ID of the Master
  // that is connected to the Bus via the Slave port.
  int m_MasterID[TL2_MAX_MASTERS];

  // Mapping from Master port number to the ID of the Slave
  // that is connected to the Bus via the Master port.
  int m_SlaveID[TL2_MAX_SLAVES];

  // Mapping from the ID of the Master to the number of the
  // Slave port that is connected to the Master.
  int m_Sport[TL2_MAX_MASTERS];

  // Mapping from the ID of the Slave to the number of the
  // Master port that is connected to the Slave.
  int m_Mport[TL2_MAX_SLAVES];

  // Arbiter register:
  // Index 0 to TL2_ARB_REG_LENGTH - 2 =  Master priorities.
  // Index TL2_ARB_REG_LENGTH - 1 = Slave port number of Master
  //                                with default grant.
  int m_ArbiterReg[TL2_ARB_REG_LENGTH];

  // Structure conating the memory map of the Slaves.
  // m_MemoryMap[i] = Start and end adress of the Slave connected with the
  //                  Bus via the Master port of index i.
  template <class ta>
  struct m_MemoryStruct {
    ta StartAddr;
    ta EndAddr;
  };
  m_MemoryStruct<Ta> m_MemoryMap[TL2_MAX_SLAVES];

  // Pending requests vector.
  int m_Requests[TL2_MAX_MASTERS];

  // Vector counting the number of grants for each master,
  // including the dummy master. This is used in the "fair among equals"
  // arbitartion algorithm.
  double m_NumGrants[TL2_MAX_MASTERS + 1];

  Ta* m_MemoryMapCache; // pointer to cache table
  Ta  m_FirstStartAddr;  // Lowest Slave start address
  Ta  m_LastEndAddr;     // Largest Slave end address

  // Bus parameters
  int  m_BusID;
  bool m_Pipelined;
  bool m_WriteResponse;
  int  m_ReadWaitCycles;
  int  m_WriteWaitCycles;

  bool m_IsInitialized; // Flag specifying if the Initialize() method
                        // has already been called.
  int m_NumMasters; // Number of masters attached to the Bus
  int m_NumSlaves;  // Number of Slaves attached to the Bus

  int m_GrantedMaster;  // Port number of the granted Master
  int m_AddressedSlave; // Port number of the addressed Slave

  // MasterID, Slave ID, and Connection ID
  // used in the error response method.
  int m_GrantedMasterE;
  int m_AddressedSlaveE;
  Ta  m_SaddrE;

  int m_TaNum; // Counter for the total number of transactions

  int m_NumBytesPerWord;

  // Event triggering the error response method.
  sc_event m_BusErrorEvent;
}; // end of class

#endif // _TL2_BUS_H
