// 
//  (c) Copyright OCP-IP 2003, 2004, 2005
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Author  : Herve Alexanian, Sonics Inc.
//
//  Description : proxy classes for user configured TL0 OCP ports
//                For use with tl1/tl0 layer adapters
//  $Id: ocp_tl0_port.h,v 1.3 2007/02/02 01:08:26 jaldis Exp $ 
//
// ============================================================================

#ifndef _OCP_TL0_PORT_H_
#define _OCP_TL0_PORT_H_

#include "systemc.h"
#include "ocp_globals.h"
#include <stdexcept>

// Input port proxy
class sc_proxy_in_signal_port_base {
  public:
    virtual void getLongValue( sc_unsigned& ) const = 0;
    virtual void getLongValue( uint64& ) const = 0;
    virtual void getLongValue( unsigned int& ) const = 0;
    // quick and simple version for signals <= 32 bits
    virtual unsigned int getValue() const = 0;
    virtual sc_event_finder& defaultEvent() const = 0;
};

template <typename T>
class sc_proxy_in_signal_port: public sc_proxy_in_signal_port_base,
                                public sc_in<T>
{
  public:
    virtual void getLongValue( sc_unsigned& ) const;
    virtual void getLongValue( uint64& ) const;
    virtual void getLongValue( unsigned int& ) const;
    virtual unsigned int getValue() const;
    virtual sc_event_finder& defaultEvent() const;
    sc_proxy_in_signal_port( const char* szName ) :
        sc_in<T>( szName )
        {}
};

// Specialization for sc_bv. Because of the N argument, this is a partial
// specialization thus requires the whole class to be re-defined.
template <int N >
class sc_proxy_in_signal_port<sc_bv<N> >: public sc_proxy_in_signal_port_base,
                                          public sc_in< sc_bv<N> >
{
  public:
    typedef sc_bv<N> T;
    virtual void getLongValue( sc_unsigned& ) const;
    virtual void getLongValue( uint64& ) const;
    virtual void getLongValue( unsigned int& ) const;
    virtual unsigned int getValue() const;
    virtual sc_event_finder& defaultEvent() const;
    sc_proxy_in_signal_port( const char* szName ) :
        sc_in<T>( szName )
        {}
};


// Output port proxy
class sc_proxy_out_signal_port_base {
  public:
    virtual void initValue( unsigned int ) = 0;
    //virtual void initValue( const sc_unsigned& ) = 0;
    virtual void putValue ( int ) = 0;
    virtual void putValue ( unsigned int ) = 0;
    virtual void putValue ( uint64 ) = 0;
    virtual void putValue ( const sc_unsigned& ) = 0;
};

template <typename T>
class sc_proxy_out_signal_port: public sc_proxy_out_signal_port_base,
                                public sc_out<T>
{
  public:
    virtual void initValue( unsigned int );
    //virtual void initValue( const sc_unsigned& );
    virtual void putValue ( int );
    virtual void putValue ( unsigned int );
    virtual void putValue ( uint64 );
    virtual void putValue ( const sc_unsigned& );
    sc_proxy_out_signal_port( const char* szName ) :
        sc_out<T>( szName )
        {}
    T m_shadow;
};

// The OCP (1.0) TL0 port base class for a slave adapter
class OCP_TL0_AdapterPorts {
  protected:
    void    mapPort( sc_proxy_out_signal_port_base&  actual,
                     sc_proxy_out_signal_port_base*& formal ) {
        formal = &actual;
    }
    void    mapPort( sc_proxy_in_signal_port_base&  actual,
                     sc_proxy_in_signal_port_base*& formal ) {
        formal = &actual;
    }
};

template<typename T>
struct OCP_TL0_PortFactory {
    virtual T* operator()() const = 0;
};

class OCP_TL0_MasterPorts : public OCP_TL0_AdapterPorts {
    template <typename TdataCl> friend class OCP_TL0_TL1_Slave_Adapter;
  public:
    template <typename DerivedT>
    struct Factory: public OCP_TL0_PortFactory<OCP_TL0_MasterPorts> {
        virtual OCP_TL0_MasterPorts* operator()() const {
            OCP_TL0_MasterPorts* pTmp = new DerivedT();
            pTmp->bind();
            return pTmp;
        }
    };

    // supposed to be protected, but don't know how to let factory be a friend
    virtual void bind() = 0;

  protected:
    // ins
    sc_proxy_in_signal_port_base* SReset_n;
    sc_proxy_in_signal_port_base* SCmdAccept;
    sc_proxy_in_signal_port_base* SThreadBusy;
    sc_proxy_in_signal_port_base* SDataAccept;
    sc_proxy_in_signal_port_base* SResp;
    sc_proxy_in_signal_port_base* SThreadID;
    sc_proxy_in_signal_port_base* SData;
    sc_proxy_in_signal_port_base* SFlag;
    sc_proxy_in_signal_port_base* SInterrupt;
    sc_proxy_in_signal_port_base* SError;
    
    // outs
    sc_proxy_out_signal_port_base* MReset_n;
    sc_proxy_out_signal_port_base* MCmd;
    sc_proxy_out_signal_port_base* MConnID;
    sc_proxy_out_signal_port_base* MReqInfo;
    sc_proxy_out_signal_port_base* MThreadID;
    sc_proxy_out_signal_port_base* MAddr;
    sc_proxy_out_signal_port_base* MAddrSpace;
    sc_proxy_out_signal_port_base* MBurst;
    sc_proxy_out_signal_port_base* MByteEn;
    sc_proxy_out_signal_port_base* MDataValid;
    sc_proxy_out_signal_port_base* MDataThreadID;
    sc_proxy_out_signal_port_base* MData;
    sc_proxy_out_signal_port_base* MRespAccept;
    sc_proxy_out_signal_port_base* MThreadBusy;
    sc_proxy_out_signal_port_base* MFlag;
    sc_proxy_out_signal_port_base* MError;
    
    OCP_TL0_MasterPorts() :
        SReset_n( NULL ),
        SCmdAccept( NULL ),
        SThreadBusy( NULL ),
        SDataAccept( NULL ),
        SResp( NULL ),
        SThreadID( NULL ),
        SData( NULL ),
        SFlag( NULL ),
        SInterrupt( NULL ),
        SError( NULL ),

        MReset_n( NULL ),
        MCmd( NULL ),
        MConnID( NULL ),
        MReqInfo( NULL ),
        MThreadID( NULL ),
        MAddr( NULL ),
        MAddrSpace( NULL ),
        MBurst( NULL ),
        MByteEn( NULL ),
        MDataValid( NULL ),
        MDataThreadID( NULL ),
        MData( NULL ),
        MRespAccept( NULL ),
        MThreadBusy( NULL ),
        MFlag( NULL ),
        MError( NULL )
        {}
};

class OCP_TL0_SlavePorts : public OCP_TL0_AdapterPorts {
    template <typename TdataCl> friend class OCP_TL0_TL1_Master_Adapter;
  public:
    template <typename DerivedT>
    struct Factory: public OCP_TL0_PortFactory<OCP_TL0_SlavePorts> {
        virtual OCP_TL0_SlavePorts* operator()() const {
            OCP_TL0_SlavePorts* pTmp = new DerivedT();
            pTmp->bind();
            return pTmp;
        }
    };

    // supposed to be protected, but don't know how to let factory be a friend
    virtual void bind() = 0;

  protected:
    // ins
    sc_proxy_in_signal_port_base* MReset_n;
    sc_proxy_in_signal_port_base* MCmd;
    sc_proxy_in_signal_port_base* MConnID;
    sc_proxy_in_signal_port_base* MReqInfo;
    sc_proxy_in_signal_port_base* MThreadID;
    sc_proxy_in_signal_port_base* MAddr;
    sc_proxy_in_signal_port_base* MAddrSpace;
    sc_proxy_in_signal_port_base* MBurst;
    sc_proxy_in_signal_port_base* MByteEn;
    sc_proxy_in_signal_port_base* MDataValid;
    sc_proxy_in_signal_port_base* MDataThreadID;
    sc_proxy_in_signal_port_base* MData;
    sc_proxy_in_signal_port_base* MRespAccept;
    sc_proxy_in_signal_port_base* MThreadBusy;
    sc_proxy_in_signal_port_base* MFlag;
    sc_proxy_in_signal_port_base* MError;
    
    // outs
    sc_proxy_out_signal_port_base* SReset_n;
    sc_proxy_out_signal_port_base* SCmdAccept;
    sc_proxy_out_signal_port_base* SThreadBusy;
    sc_proxy_out_signal_port_base* SDataAccept;
    sc_proxy_out_signal_port_base* SResp;
    sc_proxy_out_signal_port_base* SThreadID;
    sc_proxy_out_signal_port_base* SData;
    sc_proxy_out_signal_port_base* SFlag;
    sc_proxy_out_signal_port_base* SInterrupt;
    sc_proxy_out_signal_port_base* SError;
    
    OCP_TL0_SlavePorts() :
        MReset_n( NULL ),
        MCmd( NULL ),
        MConnID( NULL ),
        MReqInfo( NULL ),
        MThreadID( NULL ),
        MAddr( NULL ),
        MAddrSpace( NULL ),
        MBurst( NULL ),
        MByteEn( NULL ),
        MDataValid( NULL ),
        MDataThreadID( NULL ),
        MData( NULL ),
        MRespAccept( NULL ),
        MThreadBusy( NULL ),
        MFlag( NULL ),
        MError( NULL ),

        SReset_n( NULL ),
        SCmdAccept( NULL ),
        SThreadBusy( NULL ),
        SDataAccept( NULL ),
        SResp( NULL ),
        SThreadID( NULL ),
        SData( NULL ),
        SFlag( NULL ),
        SInterrupt( NULL ),
        SError( NULL )
        {}
};

class OCP2_TL0_MasterPorts : public OCP_TL0_AdapterPorts {
    template <typename TdataCl> friend class OCP2_TL0_TL1_Slave_Adapter;
  public:
    template <typename DerivedT>
    struct Factory: public OCP_TL0_PortFactory<OCP2_TL0_MasterPorts> {
        virtual OCP2_TL0_MasterPorts* operator()() const {
            OCP2_TL0_MasterPorts* pTmp = new DerivedT();
            pTmp->bind();
            return pTmp;
        }
    };

    // supposed to be protected, but don't know how to let factory be a friend
    virtual void bind() = 0;

  protected:
    // ins
    sc_proxy_in_signal_port_base* SReset_n;
    sc_proxy_in_signal_port_base* SCmdAccept;
    sc_proxy_in_signal_port_base* SThreadBusy;
    sc_proxy_in_signal_port_base* SDataAccept;
    sc_proxy_in_signal_port_base* SDataThreadBusy;
    sc_proxy_in_signal_port_base* SResp;
    sc_proxy_in_signal_port_base* SRespInfo;
    sc_proxy_in_signal_port_base* SRespLast;
    sc_proxy_in_signal_port_base* SThreadID;
    sc_proxy_in_signal_port_base* SData;
    sc_proxy_in_signal_port_base* SDataInfo;
    sc_proxy_in_signal_port_base* SFlag;
    sc_proxy_in_signal_port_base* SInterrupt;
    sc_proxy_in_signal_port_base* SError;
    
    // outs
    sc_proxy_out_signal_port_base* MReset_n;
    sc_proxy_out_signal_port_base* MCmd;
    sc_proxy_out_signal_port_base* MConnID;
    sc_proxy_out_signal_port_base* MThreadID;
    sc_proxy_out_signal_port_base* MByteEn;
    sc_proxy_out_signal_port_base* MReqInfo;
    sc_proxy_out_signal_port_base* MReqLast;
    sc_proxy_out_signal_port_base* MAddr;
    sc_proxy_out_signal_port_base* MAddrSpace;
    sc_proxy_out_signal_port_base* MAtomicLength;
    sc_proxy_out_signal_port_base* MBurstLength;
    sc_proxy_out_signal_port_base* MBurstPrecise;
    sc_proxy_out_signal_port_base* MBurstSeq;
    sc_proxy_out_signal_port_base* MBurstSingleReq;
    sc_proxy_out_signal_port_base* MDataValid;
    sc_proxy_out_signal_port_base* MDataByteEn;
    sc_proxy_out_signal_port_base* MDataThreadID;
    sc_proxy_out_signal_port_base* MDataLast;
    sc_proxy_out_signal_port_base* MData;
    sc_proxy_out_signal_port_base* MDataInfo;
    sc_proxy_out_signal_port_base* MRespAccept;
    sc_proxy_out_signal_port_base* MThreadBusy;
    sc_proxy_out_signal_port_base* MFlag;
    sc_proxy_out_signal_port_base* MError;
    
    OCP2_TL0_MasterPorts() :
        SReset_n( NULL ),
        SCmdAccept( NULL ),
        SThreadBusy( NULL ),
        SDataAccept( NULL ),
        SDataThreadBusy( NULL ),
        SResp( NULL ),
        SRespInfo( NULL ),
        SRespLast( NULL ),
        SThreadID( NULL ),
        SData( NULL ),
        SDataInfo( NULL ),
        SFlag( NULL ),
        SInterrupt( NULL ),
        SError( NULL ),
        
        MReset_n( NULL ),
        MCmd( NULL ),
        MConnID( NULL ),
        MThreadID( NULL ),
        MByteEn( NULL ),
        MReqInfo( NULL ),
        MReqLast( NULL ),
        MAddr( NULL ),
        MAddrSpace( NULL ),
        MAtomicLength( NULL ),
        MBurstLength( NULL ),
        MBurstPrecise( NULL ),
        MBurstSeq( NULL ),
        MBurstSingleReq( NULL ),
        MDataValid( NULL ),
        MDataByteEn( NULL ),
        MDataThreadID( NULL ),
        MDataLast( NULL ),
        MData( NULL ),
        MDataInfo( NULL ),
        MRespAccept( NULL ),
        MThreadBusy( NULL ),
        MFlag( NULL ),
        MError( NULL )
        {}
};

class OCP2_TL0_SlavePorts : public OCP_TL0_AdapterPorts {
    template <typename TdataCl> friend class OCP2_TL0_TL1_Master_Adapter;
  public:
    template <typename DerivedT>
    struct Factory: public OCP_TL0_PortFactory<OCP2_TL0_SlavePorts> {
        virtual OCP2_TL0_SlavePorts* operator()() const {
            OCP2_TL0_SlavePorts* pTmp = new DerivedT();
            pTmp->bind();
            return pTmp;
        }
    };

    // supposed to be protected, but don't know how to let factory be a friend
    virtual void bind() = 0;

  protected:
    // ins
    sc_proxy_in_signal_port_base* MReset_n;
    sc_proxy_in_signal_port_base* MCmd;
    sc_proxy_in_signal_port_base* MConnID;
    sc_proxy_in_signal_port_base* MThreadID;
    sc_proxy_in_signal_port_base* MByteEn;
    sc_proxy_in_signal_port_base* MReqInfo;
    sc_proxy_in_signal_port_base* MReqLast;
    sc_proxy_in_signal_port_base* MAddr;
    sc_proxy_in_signal_port_base* MAddrSpace;
    sc_proxy_in_signal_port_base* MAtomicLength;
    sc_proxy_in_signal_port_base* MBurstLength;
    sc_proxy_in_signal_port_base* MBurstPrecise;
    sc_proxy_in_signal_port_base* MBurstSeq;
    sc_proxy_in_signal_port_base* MBurstSingleReq;
    sc_proxy_in_signal_port_base* MDataValid;
    sc_proxy_in_signal_port_base* MDataByteEn;
    sc_proxy_in_signal_port_base* MDataThreadID;
    sc_proxy_in_signal_port_base* MDataLast;
    sc_proxy_in_signal_port_base* MData;
    sc_proxy_in_signal_port_base* MDataInfo;
    sc_proxy_in_signal_port_base* MRespAccept;
    sc_proxy_in_signal_port_base* MThreadBusy;
    sc_proxy_in_signal_port_base* MFlag;
    sc_proxy_in_signal_port_base* MError;
    
    // outs
    sc_proxy_out_signal_port_base* SReset_n;
    sc_proxy_out_signal_port_base* SCmdAccept;
    sc_proxy_out_signal_port_base* SThreadBusy;
    sc_proxy_out_signal_port_base* SDataAccept;
    sc_proxy_out_signal_port_base* SDataThreadBusy;
    sc_proxy_out_signal_port_base* SResp;
    sc_proxy_out_signal_port_base* SRespInfo;
    sc_proxy_out_signal_port_base* SRespLast;
    sc_proxy_out_signal_port_base* SThreadID;
    sc_proxy_out_signal_port_base* SData;
    sc_proxy_out_signal_port_base* SDataInfo;
    sc_proxy_out_signal_port_base* SFlag;
    sc_proxy_out_signal_port_base* SInterrupt;
    sc_proxy_out_signal_port_base* SError;
    
    OCP2_TL0_SlavePorts() :
        MReset_n( NULL ),
        MCmd( NULL ),
        MConnID( NULL ),
        MThreadID( NULL ),
        MByteEn( NULL ),
        MReqInfo( NULL ),
        MReqLast( NULL ),
        MAddr( NULL ),
        MAddrSpace( NULL ),
        MAtomicLength( NULL ),
        MBurstLength( NULL ),
        MBurstPrecise( NULL ),
        MBurstSeq( NULL ),
        MBurstSingleReq( NULL ),
        MDataValid( NULL ),
        MDataByteEn( NULL ),
        MDataThreadID( NULL ),
        MDataLast( NULL ),
        MData( NULL ),
        MDataInfo( NULL ),
        MRespAccept( NULL ),
        MThreadBusy( NULL ),
        MFlag( NULL ),
        MError( NULL ),
        
        SReset_n( NULL ),
        SCmdAccept( NULL ),
        SThreadBusy( NULL ),
        SDataAccept( NULL ),
        SDataThreadBusy( NULL ),
        SResp( NULL ),
        SRespInfo( NULL ),
        SRespLast( NULL ),
        SThreadID( NULL ),
        SData( NULL ),
        SDataInfo( NULL ),
        SFlag( NULL ),
        SInterrupt( NULL ),
        SError( NULL )
        {}
};

// Template function implementation
template<typename T>
void
sc_proxy_in_signal_port<T>::getLongValue( sc_unsigned& val ) const {
    val = sc_in<T>::read();
}
template<typename T>
void
sc_proxy_in_signal_port<T>::getLongValue( uint64& val ) const {
    val = sc_in<T>::read();
}
template<typename T>
void
sc_proxy_in_signal_port<T>::getLongValue( unsigned int& val ) const {
    val = sc_in<T>::read();
}

template<typename T>
unsigned int
sc_proxy_in_signal_port<T>::getValue() const {
    return sc_in<T>::read();
}

template<typename T>
sc_event_finder&
sc_proxy_in_signal_port<T>::defaultEvent() const {
    return sc_in<T>::value_changed();
}

template<int N>
void
sc_proxy_in_signal_port<sc_bv<N> >::getLongValue( sc_unsigned& val ) const {
    val = sc_in<sc_bv<N> >::read();
}
template<int N>
void
sc_proxy_in_signal_port<sc_bv<N> >::getLongValue( uint64& val ) const {
    val = sc_in<sc_bv<N> >::read().to_uint64();
}
template<int N>
void
sc_proxy_in_signal_port<sc_bv<N> >::getLongValue( unsigned int& val ) const {
    val = sc_in<sc_bv<N> >::read().to_uint();
}
template<int N>
unsigned int
sc_proxy_in_signal_port<sc_bv<N> >::getValue() const {
    return sc_in<T>::read().get_word(0);
}

template<int N>
sc_event_finder&
sc_proxy_in_signal_port<sc_bv<N> >::defaultEvent() const {
    return sc_in<T>::value_changed();
}

template<typename T>
void
sc_proxy_out_signal_port<T>::initValue( unsigned int val ) {
    m_shadow = val;
    sc_out<T>::initialize( m_shadow );
}

template<typename T>
void
sc_proxy_out_signal_port<T>::putValue( int val ) {
    m_shadow = static_cast<unsigned int>( val );
    sc_out<T>::write( m_shadow );
}

template<typename T>
void
sc_proxy_out_signal_port<T>::putValue( unsigned int val ) {
    m_shadow = val;
    sc_out<T>::write( m_shadow );
}

template<typename T>
void
sc_proxy_out_signal_port<T>::putValue( uint64 val ) {
    m_shadow = val;
    sc_out<T>::write( m_shadow );
}

template<typename T>
void
sc_proxy_out_signal_port<T>::putValue( const sc_unsigned& val ) {
    m_shadow = val.to_uint64();
    sc_out<T>::write( m_shadow );
}

template<>
inline void
sc_proxy_out_signal_port<bool>::putValue( const sc_unsigned& val ) {
    sc_out<bool>::write( val[0] );
}

#endif // _OCP_TL0_PORT_H_
