//
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//       Original Authors :
//                Alan Kamas, for Sonics Inc.
//                Anssi Haverinen, Nokia Inc.
//                Joe Chou, Sonics Inc.
//                James Aldis, Texas Instruments, Inc.
//           $Id: ocp_tl1_ocpmongen_cl.h,v 1.4 2007/02/03 03:19:54 halexan Exp $
//
//
//  Description : OCP Transaction Level Channel
//                - generate an OCP monitor file
//
//  Old History :
//             12/14/2003 Original revision for OCP 2.0
//             07/25/2004 Minor bug fixes for printouts
//
//             5/08/2004 Added support for gzipped trace files using gzstream
//
//                       Changes:
//                         new include files
//                         monFile replaced everywhere by a pointer monFileptr
//                         initialisation of monFile changed in constructor
//                         added a method for testing for ".gz" in filename
//                         cleaned up a few messy tabs here and there
//
//
// ============================================================================


#ifndef __OCP_TL1_OCPMONGEN_CL_LEGACY_H__
#define __OCP_TL1_OCPMONGEN_CL_LEGACY_H__


#ifdef GZSTREAM_AVAILABLE
#include "gzstream.h"
#else
typedef ofstream ogzstream;
#endif


template <typename TdataCl>
class OCP_TL1_OCPMonGenCl {
  public:

    typedef typename TdataCl::DataType Td;
    typedef typename TdataCl::AddrType Ta;

    // Constructor
    OCP_TL1_OCPMonGenCl(std::string my_ocpmon_prefix, std::string file_name)
    : ocpmon_prefix(my_ocpmon_prefix)
    {
        // Open file for writing
        if (ends_dotgz(file_name)) {
          monFileptr = new ogzstream(file_name.c_str());
        } else {
          monFileptr = new ofstream(file_name.c_str());
        }

        // Did the initialization work?
        // Is it open for writing?
        if (!(*monFileptr)) {
            cout << "Error: OCP Monitor for OCP named " << ocpmon_prefix << " - cannot open OCP monitor file: " << file_name
                 << endl;
            // give up
            exit(1100);
        }

        // clean up the prefix (in case there are leading and trailing '_' chars)
        if ((my_ocpmon_prefix[0]=='_') && (my_ocpmon_prefix[my_ocpmon_prefix.length()-1]=='_')) {
            ocpmon_prefix = my_ocpmon_prefix.substr(1,my_ocpmon_prefix.length()-2);
        }
    }

    // Destructor
    ~OCP_TL1_OCPMonGenCl()
    {
        // close out the file
        delete monFileptr;
    }

    void
    monitorHeader( ParamCl<TdataCl> *mParamCl )
    {
#ifndef OCPVERSION
#define IDEFINED_OCPVERSION
#define OCPVERSION "ocp-2.2"
#endif
        // Send out the header information
        *monFileptr << "# ocpversion=" << OCPVERSION << std::endl;
#ifdef IDEFINED_OCPVERSION
#undef IDEFINED_OCPVERSION
#undef OCPVERSION
#endif
        *monFileptr << "# name=" << ocpmon_prefix << "_ocpmon" << std::endl;

        *monFileptr << "# broadcast_enable=" << mParamCl->broadcast_enable << std::endl;
        *monFileptr << "# burst_aligned=" << mParamCl->burst_aligned << std::endl;
        *monFileptr << "# burstseq_dflt1_enable=" << mParamCl->burstseq_dflt1_enable << std::endl;
        *monFileptr << "# burstseq_dflt2_enable=" << mParamCl->burstseq_dflt2_enable << std::endl;
        *monFileptr << "# burstseq_incr_enable=" << mParamCl->burstseq_incr_enable << std::endl;
        *monFileptr << "# burstseq_strm_enable=" << mParamCl->burstseq_strm_enable << std::endl;
        *monFileptr << "# burstseq_unkn_enable=" << mParamCl->burstseq_unkn_enable << std::endl;
        *monFileptr << "# burstseq_wrap_enable=" << mParamCl->burstseq_wrap_enable << std::endl;
        *monFileptr << "# burstseq_xor_enable=" << mParamCl->burstseq_xor_enable << std::endl;
        *monFileptr << "# burstseq_blck_enable=" << mParamCl->burstseq_blck_enable << std::endl;
        *monFileptr << "# endian=" << mParamCl->endian << std::endl;
        *monFileptr << "# force_aligned=" << mParamCl->force_aligned << std::endl;
        *monFileptr << "# mthreadbusy_exact=" << mParamCl->mthreadbusy_exact << std::endl;
        *monFileptr << "# mthreadbusy_pipelined=" << mParamCl->mthreadbusy_pipelined << std::endl;
        *monFileptr << "# rdlwrc_enable=" << mParamCl->rdlwrc_enable << std::endl;
        *monFileptr << "# read_enable=" << mParamCl->read_enable << std::endl;
        *monFileptr << "# readex_enable=" << mParamCl->readex_enable << std::endl;
        *monFileptr << "# sdatathreadbusy_exact=" << mParamCl->sdatathreadbusy_exact << std::endl;
        *monFileptr << "# sdatathreadbusy_pipelined=" << mParamCl->sdatathreadbusy_pipelined << std::endl;
        *monFileptr << "# sthreadbusy_exact=" << mParamCl->sthreadbusy_exact << std::endl;
        *monFileptr << "# sthreadbusy_pipelined=" << mParamCl->sthreadbusy_pipelined << std::endl;
        *monFileptr << "# write_enable=" << mParamCl->write_enable << std::endl;
        *monFileptr << "# writenonpost_enable=" << mParamCl->writenonpost_enable << std::endl;
        *monFileptr << "# datahandshake=" << mParamCl->datahandshake << std::endl;
        *monFileptr << "# reqdata_together=" << mParamCl->reqdata_together << std::endl;
        *monFileptr << "# writeresp_enable=" << mParamCl->writeresp_enable << std::endl;
        *monFileptr << "# addr=" << mParamCl->addr << std::endl;
        *monFileptr << "# addr_wdth=" << mParamCl->addr_wdth << std::endl;
        *monFileptr << "# addrspace=" << mParamCl->addrspace << std::endl;
        *monFileptr << "# addrspace_wdth=" << mParamCl->addrspace_wdth << std::endl;
        *monFileptr << "# atomiclength=" << mParamCl->atomiclength << std::endl;
        *monFileptr << "# atomiclength_wdth=" << mParamCl->atomiclength_wdth << std::endl;
        *monFileptr << "# burstlength=" << mParamCl->burstlength << std::endl;
        *monFileptr << "# burstlength_wdth=" << mParamCl->burstlength_wdth << std::endl;
        *monFileptr << "# blockheight=" << mParamCl->blockheight << std::endl;
        *monFileptr << "# blockheight_wdth=" << mParamCl->blockheight_wdth << std::endl;
        *monFileptr << "# blockstride=" << mParamCl->blockstride << std::endl;
        *monFileptr << "# blockstride_wdth=" << mParamCl->blockstride_wdth << std::endl;
        *monFileptr << "# burstprecise=" << mParamCl->burstprecise << std::endl;
        *monFileptr << "# burstseq=" << mParamCl->burstseq << std::endl;
        *monFileptr << "# burstsinglereq=" << mParamCl->burstsinglereq << std::endl;
        *monFileptr << "# byteen=" << mParamCl->byteen << std::endl;
        *monFileptr << "# cmdaccept=" << mParamCl->cmdaccept << std::endl;
        *monFileptr << "# connid=" << mParamCl->connid << std::endl;
        *monFileptr << "# connid_wdth=" << mParamCl->connid_wdth << std::endl;
        *monFileptr << "# dataaccept=" << mParamCl->dataaccept << std::endl;
        *monFileptr << "# datalast=" << mParamCl->datalast << std::endl;
        *monFileptr << "# datarowlast=" << mParamCl->datarowlast << std::endl;
        *monFileptr << "# data_wdth=" << mParamCl->data_wdth << std::endl;
        *monFileptr << "# mdata=" << mParamCl->mdata << std::endl;
        *monFileptr << "# mdatabyteen=" << mParamCl->mdatabyteen << std::endl;
        *monFileptr << "# mdatainfo=" << mParamCl->mdatainfo << std::endl;
        *monFileptr << "# mdatainfo_wdth=" << mParamCl->mdatainfo_wdth << std::endl;
        *monFileptr << "# mdatainfobyte_wdth=" << mParamCl->mdatainfobyte_wdth << std::endl;
        *monFileptr << "# mthreadbusy=" << mParamCl->mthreadbusy << std::endl;
        *monFileptr << "# reqinfo=" << mParamCl->reqinfo << std::endl;
        *monFileptr << "# reqinfo_wdth=" << mParamCl->reqinfo_wdth << std::endl;
        *monFileptr << "# reqlast=" << mParamCl->reqlast << std::endl;
        *monFileptr << "# reqrowlast=" << mParamCl->reqrowlast << std::endl;
        *monFileptr << "# resp=" << mParamCl->resp << std::endl;
        *monFileptr << "# respaccept=" << mParamCl->respaccept << std::endl;
        *monFileptr << "# respinfo=" << mParamCl->respinfo << std::endl;
        *monFileptr << "# respinfo_wdth=" << mParamCl->respinfo_wdth << std::endl;
        *monFileptr << "# resplast=" << mParamCl->resplast << std::endl;
        *monFileptr << "# resprowlast=" << mParamCl->resprowlast << std::endl;
        *monFileptr << "# sdata=" << mParamCl->sdata << std::endl;
        *monFileptr << "# sdatainfo=" << mParamCl->sdatainfo << std::endl;
        *monFileptr << "# sdatainfo_wdth=" << mParamCl->sdatainfo_wdth << std::endl;
        *monFileptr << "# sdatainfobyte_wdth=" << mParamCl->sdatainfobyte_wdth << std::endl;
        *monFileptr << "# sdatathreadbusy=" << mParamCl->sdatathreadbusy << std::endl;
        *monFileptr << "# sthreadbusy=" << mParamCl->sthreadbusy << std::endl;
        *monFileptr << "# threads=" << mParamCl->threads << std::endl;
        *monFileptr << "# tags=" << mParamCl->tags << std::endl;
        *monFileptr << "# taginorder=" << mParamCl->taginorder << std::endl;
        *monFileptr << "# control=" << mParamCl->control << std::endl;
        *monFileptr << "# controlbusy=" << mParamCl->controlbusy << std::endl;
        *monFileptr << "# control_wdth=" << mParamCl->control_wdth << std::endl;
        *monFileptr << "# controlwr=" << mParamCl->controlwr << std::endl;
        *monFileptr << "# interrupt=" << mParamCl->interrupt << std::endl;
        *monFileptr << "# merror=" << mParamCl->merror << std::endl;
        *monFileptr << "# mflag=" << mParamCl->mflag << std::endl;
        *monFileptr << "# mflag_wdth=" << mParamCl->mflag_wdth << std::endl;
        *monFileptr << "# mreset=" << mParamCl->mreset << std::endl;
        *monFileptr << "# serror=" << mParamCl->serror << std::endl;
        *monFileptr << "# sflag=" << mParamCl->sflag << std::endl;
        *monFileptr << "# sflag_wdth=" << mParamCl->sflag_wdth << std::endl;
        *monFileptr << "# sreset=" << mParamCl->sreset << std::endl;
        *monFileptr << "# status=" << mParamCl->status << std::endl;
        *monFileptr << "# statusbusy=" << mParamCl->statusbusy << std::endl;
        *monFileptr << "# statusrd=" << mParamCl->statusrd << std::endl;
        *monFileptr << "# status_wdth=" << mParamCl->status_wdth << std::endl;
        *monFileptr << "##" << std::endl;
    }

    // A helper function to print out monitor dump
    void
    printBinIf( bool my_condition, unsigned int my_value, int my_bit_width = 1 )
    {
        *monFileptr << " " << std::setw(1);
        if (my_condition) {
            // print out binary with MSB on left and bit zero on right
            unsigned int my_one = 1 << (my_bit_width-1);
            for (int i=0;i<my_bit_width;i++) {
                if (my_value & my_one) {
                    *monFileptr << '1';
                } else {
                    *monFileptr << '0';
                }
                my_one = my_one>>1;
            }
        } else {
            // condition not met. Print "don't cares"
            for (int i=0;i<my_bit_width;i++) {
                *monFileptr << "x";
            }
        }
    }

    // A helper function to print out monitor dump
    void
    printIf( bool my_condition, int my_value, int my_bit_width = 4 )
    {
        int my_hex_size = 1 + (my_bit_width-1)/4;
        if (my_condition) {
            *monFileptr << " " << std::setw(my_hex_size) << std::setfill('0') << my_value;
        } else {
            // condition not met. Print "don't cares"
            *monFileptr << " ";
            for (int i=0;i<my_hex_size;i++) {
                *monFileptr << "x";
            }
        }
    }

    // Prints out one line of ocp state information in ocpMon format
    //  - should be called every ocp clock tick
    //
    void
    monitorOut( ParamCl<TdataCl> *mParamCl,
            sc_core::sc_time dtime,
            const OCPRequestGrp<Td,Ta>& curRequest,
            const bool& commandAccepted,
            const OCPDataHSGrp<Td>& curDataHS,
            const bool& dataAccepted,
            const OCPResponseGrp<Td>& curResponse,
            const bool& responseAccepted,
            const unsigned int& mySThreadBusy,
            const unsigned int& mySDataThreadBusy,
            const unsigned int& myMThreadBusy,
            const bool& myMReset_n,
            const bool& mySReset_n,
            const bool& myMError,
            const unsigned int& myMFlag,
            const bool& mySError,
            const unsigned int& mySFlag,
            const bool& mySInterrupt,
            const unsigned int& myControl,
            const bool& myControlWr,
            const bool& myControlBusy,
            const unsigned int& myStatus,
            const bool& myStatusRd,
            const bool& myStatusBusy)
    {
      if (dtime>SC_ZERO_TIME) {
        bool requestInProgress = (curRequest.MCmd != OCP_MCMD_IDLE);
        bool writerequestInProgress = (curRequest.MCmd == OCP_MCMD_WR ||
                                       curRequest.MCmd == OCP_MCMD_WRNP ||
                                       curRequest.MCmd == OCP_MCMD_BCST);
        bool dataInProgress = curDataHS.MDataValid;
        bool responseInProgress = (curResponse.SResp != OCP_SRESP_NULL);

        // Field Description : Required OCP Parameter : Width(bits) : Format

        // Simulation Time	None		float in (ns)
        // the time is decimal
        *monFileptr << dec;

        // Clk
        *monFileptr << std::setw(12) << std::setfill(' ') <<
           dtime.to_default_time_units();
        *monFileptr << " ";

        // most other values are in hex
        *monFileptr << hex;

        // MReset_n	mreset parameter is 1	Always 1	hexadecimal
        if (mParamCl->mreset) {
            *monFileptr << " " << std::setw(1) << myMReset_n;
        }

        // SReset_n	sreset parameter is 1	Always 1	hexadecimal
        if (mParamCl->sreset) {
            *monFileptr << " " << std::setw(1) << mySReset_n;
        }

        // MCmd	None	Always 3	hexadecimal
        if (requestInProgress) {
            *monFileptr << " " << curRequest.MCmd;
        } else {
            // No request in progress so this value must be Idle
            *monFileptr << " " << OCP_MCMD_IDLE;
        }

        // MAddr	addr is 1	addr_wdth	hexadecimal
        if (mParamCl->addr) {
          int addr_size = 1 + ((mParamCl->addr_wdth)-1)/4;
          if (requestInProgress) {
            *monFileptr << " " << std::setw(addr_size) << std::setfill('0') << curRequest.MAddr;
          } else {
            // No request in progress so this value is not defined right now
            *monFileptr << " ";
            for (int i=0;i<addr_size;i++) {
              *monFileptr << "x";
            }
          }
        }
        // MAddrSpace	addrspace is 1	addrspace_wdth	hexadecimal
        if (mParamCl->addrspace) {
            printIf(requestInProgress, curRequest.MAddrSpace, mParamCl->addrspace_wdth);
        }

        // MByteEn	byteen	data_wdth / 8	hexadecimal
        if (mParamCl->byteen) {
            unsigned dwByteEn = curRequest.MByteEn;
            unsigned dwByteEnWidth = (mParamCl->data_wdth)/8;
            unsigned dwByteEnMask = 0x0;
            for(unsigned i = 0; i < dwByteEnWidth; i++)
            {
              dwByteEnMask <<= 1;
              dwByteEnMask |= 0x1;
            }
            dwByteEn = dwByteEn & dwByteEnMask;
            printIf(requestInProgress, dwByteEn, dwByteEnWidth);
        }

        // MConnID	connid is 1	connid_wdth	hexadecimal
        if (mParamCl->connid) {
            printIf(requestInProgress, curRequest.MConnID, mParamCl->connid_wdth);
        }

        // MReqInfo	reqinfo is 1	reqinfo_wdth	hexadecimal
        if (mParamCl->reqinfo) {
            printIf(requestInProgress, curRequest.MReqInfo, mParamCl->reqinfo_wdth);
        }

        // MThreadID	threads > 1	threadid_wdth	hexadecimal
        int thread_nybbles = 1 + ( ( OcpIp::ceilLog2( mParamCl->threads ) - 1 ) / 4 );
        if (mParamCl->threads > 1) {
            printIf(requestInProgress, curRequest.MThreadID, thread_nybbles * 4);
        }

        // MAtomicLength	atomiclength is 1	atomiclength_wdth	hexadecimal
        if (mParamCl->atomiclength) {
            printIf(requestInProgress, curRequest.MAtomicLength, mParamCl->atomiclength_wdth);
        }

        // MBurstLength	burstlength is 1	burstlength_wdth	hexadecimal
        if (mParamCl->burstlength) {
            printIf(requestInProgress, curRequest.MBurstLength, mParamCl->burstlength_wdth);
        }

        // MBlockHeight blockheight is 1	blockheight_wdth	hexadecimal
        if (mParamCl->blockheight) {
            printIf(requestInProgress, curRequest.MBlockHeight, mParamCl->blockheight_wdth);
        }

        // MBlockStride blockstride is 1	blockstride_wdth	hexadecimal
        if (mParamCl->blockstride) {
            printIf(requestInProgress, curRequest.MBlockStride, mParamCl->blockstride_wdth);
        }

        // MBurstPrecise	burstprecise is 1	Always 1	hexadecimal
        if (mParamCl->burstprecise) {
            printIf(requestInProgress, curRequest.MBurstPrecise, 1);
        }

        // MBurstSeq	burstseq is 1	Always 3	hexadecimal
        if (mParamCl->burstseq) {
            printIf(requestInProgress, curRequest.MBurstSeq, 3);
        }

        // MBurstSingleReq	burstsinglereq is 1	Always 1	hexadecimal
        if (mParamCl->burstsinglereq) {
            printIf(requestInProgress, curRequest.MBurstSingleReq, 1);
        }

        // MReqLast	reqlast is 1	Always 1	hexadecimal
        if (mParamCl->reqlast) {
            printIf(requestInProgress, curRequest.MReqLast, 1);
        }

        // MReqRowLast	reqrowlast is 1	Always 1	hexadecimal
        if (mParamCl->reqrowlast) {
            printIf(requestInProgress, curRequest.MReqRowLast, 1);
        }

        // SCmdAccept	cmdaccept is 1	Always 1	hexadecimal
        if (mParamCl->cmdaccept) {
          *monFileptr << " " << commandAccepted;
        }

        // SThreadBusy	sthreadbusy is 1	threads	hexadecimal
        if (mParamCl->sthreadbusy) {
            printIf(true , mySThreadBusy, mParamCl->threads);
        }

        // MData	mdata is 1	data_wdth	hexadecimal
        if (mParamCl->mdata) {
            int my_data_size = 1 + ((mParamCl->data_wdth)-1)/4;
            if ( mParamCl->datahandshake && dataInProgress ) {
                *monFileptr << " " << std::setw(my_data_size) << std::setfill('0') << curDataHS.MData;
            } else if (!(mParamCl->datahandshake) && writerequestInProgress) {
                *monFileptr << " " << std::setw(my_data_size) << std::setfill('0') << curRequest.MData;
            } else {
                // no data - just don't cares
                *monFileptr << " ";
                for (int i=0;i<my_data_size;i++) {
                    *monFileptr << "x";
                }
            }
        }

        // MDataInfo	mdatainfo is 1	mdatainfo_wdth	hexadecimal
        if (mParamCl->datahandshake && mParamCl->mdatainfo) {
            printIf(dataInProgress, curDataHS.MDataInfo, mParamCl->mdatainfo_wdth);
        }

        // MDataValid	datahandshake is 1	Always 1	hexadecimal
        if (mParamCl->datahandshake) {
            *monFileptr << " " << dataInProgress;
        }

        // MDataByteEn	mdatabyteen is 1	data_wdth / 8	hexadecimal
        if ( mParamCl->datahandshake && mParamCl->mdatabyteen) {
            unsigned dwByteEn = curDataHS.MDataByteEn;
            unsigned dwByteEnWidth = (mParamCl->data_wdth)/8;
            unsigned dwByteEnMask = 0x0;
            for(unsigned i = 0; i < dwByteEnWidth; i++)
            {
              dwByteEnMask <<= 1;
              dwByteEnMask |= 0x1;
            }
            dwByteEn = dwByteEn & dwByteEnMask;
            printIf(dataInProgress, dwByteEn, dwByteEnWidth);
        }

        // MDataThreadID	threads > 1 and datahandshake is 1	threadid_wdth	hexadecimal
        if ( (mParamCl->datahandshake) && (mParamCl->threads > 1)) {
            printIf(dataInProgress, curDataHS.MDataThreadID, thread_nybbles * 4 );
        }

        // MDataLast	datalast is 1	Always 1	hexadecimal
        if ( mParamCl->datahandshake && mParamCl->datalast) {
            printIf(dataInProgress, curDataHS.MDataLast, 1);
        }

        // MDataRowLast	datarowlast is 1	Always 1	hexadecimal
        if ( mParamCl->datahandshake && mParamCl->datarowlast) {
            printIf(dataInProgress, curDataHS.MDataRowLast, 1);
        }

        // SDataAccept	dataaccept is 1	Always 1	hexadecimal
        if ((mParamCl->datahandshake) && (mParamCl->dataaccept) ) {
            *monFileptr << " " << dataAccepted;
        }


        // SDataThreadBusy	sdatathreadbusy is 1	threads	hexadecimal
        if ( (mParamCl->datahandshake) && (mParamCl->sdatathreadbusy) ) {
            printIf(true , mySDataThreadBusy, mParamCl->threads);
        }

        // SResp	resp is 1	Always 2	hexadecimal
        if (mParamCl->resp) {
            if (responseInProgress) {
                *monFileptr << " " << curResponse.SResp;
            } else {
                // No request in progress so this value must be Idle
                *monFileptr << " " << OCP_SRESP_NULL;
            }
        }

        // SRespInfo	respinfo is 1	respinfo_wdth	hexadecimal
        if ((mParamCl->resp) && (mParamCl->respinfo)) {
            printIf(responseInProgress , curResponse.SRespInfo, mParamCl->respinfo_wdth);
        }

        // SThreadID	threads > 1 and resp is 1	threadid_wdth	hexadecimal
        if ((mParamCl->resp) && (mParamCl->threads > 1) ) {
            printIf(responseInProgress, curResponse.SThreadID, thread_nybbles * 4);
        }

        // SData	sdata is 1	data_wdth	hexadecimal
        if ((mParamCl->sdata) && (mParamCl->resp) ) {
            int my_data_size = 1 + ((mParamCl->data_wdth)-1)/4;
            if (responseInProgress) {
                *monFileptr << " " << std::setw(my_data_size) << std::setfill('0') << curResponse.SData;
            } else {
                // no data - just don't cares
                *monFileptr << " ";
                for (int i=0;i<my_data_size;i++) {
                    *monFileptr << "x";
                }
            }
        }

        // SDataInfo	sdatainfo is 1	sdatainfo_wdth	hexadecimal
        if ((mParamCl->sdatainfo) && (mParamCl->resp) ) {
            printIf(responseInProgress , curResponse.SDataInfo, mParamCl->sdatainfo_wdth);
        }

        // SRespLast	resplast is 1	Always 1	hexadecimal
        if ( (mParamCl->resp) && (mParamCl->resplast)) {
            printIf(responseInProgress , curResponse.SRespLast, 1);
        }

        // SRespRowLast	resprowlast is 1	Always 1	hexadecimal
        if ( (mParamCl->resp) && (mParamCl->resprowlast)) {
            printIf(responseInProgress , curResponse.SRespRowLast, 1);
        }

        // MRespAccept	respaccept is 1	Always 1	hexadecimal
        if (mParamCl->respaccept) {
          *monFileptr << " " << responseAccepted;
        }

        // MThreadBusy	mthreadbusy is 1	threads	hexadecimal
        if (mParamCl->mthreadbusy) {
            printIf(true , myMThreadBusy, mParamCl->threads);
        }

        // MTagID	tags > 1	tagid_wdth	hexadecimal
        int tag_nybbles = 1 + ( ( OcpIp::ceilLog2( mParamCl->tags ) - 1 ) / 4 );
        if (mParamCl->tags > 1) {
            printIf(requestInProgress, curRequest.MTagID, tag_nybbles * 4);
        }

        // MTagInOrder	taginorder is 1	Always 1	hexadecimal
        if (mParamCl->taginorder) {
            printIf(requestInProgress, curRequest.MTagInOrder, 1);
        }

        // STagID	resp && tags > 1	tagid_wdth	hexadecimal
        if (mParamCl->resp && mParamCl->tags > 1) {
            printIf(responseInProgress, curResponse.STagID, tag_nybbles * 4);
        }

        // STagInOrder	resp && taginorder is 1	Always 1	hexadecimal
        if (mParamCl->resp && mParamCl->taginorder) {
            printIf(responseInProgress, curResponse.STagInOrder, 1);
        }

        // MDataTagID	datahandshake && tags > 1	tagid_wdth	hexadecimal
        if (mParamCl->datahandshake && mParamCl->tags > 1) {
            printIf(dataInProgress, curDataHS.MDataTagID, tag_nybbles * 4);
        }

        // MFlag	mflag is 1	mflag_wdth	binary
        if (mParamCl->mflag) {
            printIf(true, myMFlag, mParamCl->mflag_wdth);
        }

        // MError	merror is 1	Always 1	binary
        if (mParamCl->merror) {
            printBinIf(true, myMError, 1);
        }

        // SFlag	sflag is 1	sflag_wdth	binary
        if (mParamCl->sflag) {
            printIf(true, mySFlag, mParamCl->sflag_wdth);
        }

        // Serror	serror is 1	Always 1	binary
        if (mParamCl->serror) {
            printBinIf(true, mySError, 1);
        }

        // SInterrupt	interrupt is 1	Always 1	binary
        if (mParamCl->interrupt) {
            printBinIf(true, mySInterrupt, 1);
        }

        // Control	control is 1	control_wdth	hexadecimal
        if (mParamCl->control) {
            printIf(true , myControl, mParamCl->control_wdth);
        }

        // ControlWr	controlwr is 1	Always 1	binary
        if (mParamCl->controlwr) {
            printBinIf(true, myControlWr, 1);
        }

        // ControlBusy	controlbusy is 1	Always 1	binary
        if (mParamCl->controlbusy) {
            printBinIf(true, myControlBusy, 1);
        }

        // Status	status is 1	status_wdth	hexadecimal
        if (mParamCl->status) {
            printIf(true , myStatus, mParamCl->status_wdth);
        }

        // StatusRd	statusrd is 1	Always 1	binary
        if (mParamCl->statusrd) {
            printBinIf(true, myStatusRd, 1);
        }

        // StatusBusy	statusbusy is 1	Always 1	binary
        if (mParamCl->statusbusy) {
            printBinIf(true, myStatusBusy, 1);
        }

        // finish off the line
        *monFileptr << endl;
      }
    }

    protected:
        ostream *monFileptr;    // starts out with no file.
        std::string ocpmon_prefix;   // name of the OCP channel

        // method for deciding whether to generate a compressed trace
        bool ends_dotgz(std::string fname) {
          int p = fname.rfind(".gz");
          int lenf = fname.length();

          return(p==lenf-3);
        };
  };


#endif

