///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2008-2009
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Herve Alexanian - Sonics, inc.
//
//          $Id:
//
//  Description :  This file defines utility classes designed to be used as instance
//                 specific extensions on a tlm generic payload to ease OCP burst
//                 decoding
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef OCPIP_VERSION
  #error ocp_txn_inst_extensions.h may not be included directly. Use #include "ocpip.h" or #include "ocpip_X_X_X.h" (where desired ocp version is X.X.X)
#endif

namespace OCPIP_VERSION{

struct ocp_txn_track_error : public std::exception {
    std::string m_msg;
    ocp_txn_track_error( const std::string& msg ) :
        m_msg( msg )
        {}
    ~ocp_txn_track_error() throw() {};
    const char* what() const throw() {
        return m_msg.c_str();
    }
};

// single threaded burst counter. Counts all bursts in progress
class ocp_txn_track_base {
  public:
    struct txn_position {
        uint16_t count;
        int16_t  remain; // neg count means unknown (imprecise in progress)
        uint16_t row_count;
        uint16_t row_remain;
    };
	
    struct txn_status {
        int32_t  m_expect_count;
        uint32_t m_row_length;
	bool     m_precise;
        bool     m_single_resp;
        uint32_t m_req_count;
        uint32_t m_datahs_count;
        uint32_t m_resp_count;
        txn_status() :
            m_expect_count  (-1),
            m_row_length    ( 1),
	    m_precise       ( 1),
            m_single_resp   ( 0),
            m_req_count     ( 0),
            m_datahs_count  ( 0),
            m_resp_count    ( 0)
            {}
        inline bool complete_req() const {
            return m_expect_count > 0 &&
                static_cast<uint32_t>( m_expect_count ) == m_req_count;
        }
        inline bool complete_datahs() const {
            return complete_req() &&
                static_cast<uint32_t>( m_expect_count ) == m_datahs_count;
        }
        inline bool complete() const {
            return ( complete_datahs() &&
                     ( static_cast<uint32_t>( m_expect_count ) == m_resp_count ||
                       m_single_resp && m_resp_count == 1 ) );
        }
    };

    ocp_txn_track_base( const ocp_parameters* p_param ) :
        m_p_param( p_param )
	, m_byte_width( (m_p_param->data_wdth+7) >> 3 )
        {}

    virtual ~ocp_txn_track_base() {}
    uint64_t num_open() const {
        return m_burst.size();
    }
    void reset() {
        m_burst.clear();
    }

  protected:
    void purge() {
        // Erase up to the first incomplete burst
	std::deque<txn_status>::iterator pos = m_burst.begin();
        while ( pos != m_burst.end() && pos->complete() ) {
            ++pos;
        }
        m_burst.erase( m_burst.begin(), pos );
    }

    std::deque<txn_status> m_burst;
    const ocp_parameters*  m_p_param;
    uint32_t               m_byte_width;
    typedef infr::bind_checker<tlm::tlm_base_protocol_types>::ext_support_type ext_support_type;
};

class ocp_txn_track : public ocp_txn_track_base {
  public:
    typedef ocp_txn_track_base::txn_position txn_position;
    typedef ocp_txn_track_base::txn_status txn_status;
    
    ocp_txn_track( const ocp_parameters* p_param, bool tl1_n_tl2=false ) :
	ocp_txn_track_base( p_param )
	, m_tl1_n_tl2( tl1_n_tl2 )
    {	
    }

    bool is_tl1() const {
	return m_tl1_n_tl2;
    }
    bool is_tl2() const {
	return !m_tl1_n_tl2;
    }


    bool has_pending_request() {
	std::deque<txn_status>::iterator pos = m_burst.begin();
	while (pos != m_burst.end()) {
	    if(pos->complete_req() == false)
		return true;
	    pos++;
	}
	return false;
    }
    bool has_pending_datahs() {
	std::deque<txn_status>::iterator pos = m_burst.begin();
	while (pos != m_burst.end()) {
	    if(pos->complete_datahs() == false)
		return true;
	    pos++;
	}
	return false;
    }

    // The main method. track_phase returns a txn_position quadruplet with
    // count (beat count in the current burst)
    // remain (how many beats remain, negative number if imprecise)
    // row_count (beat count in the current row)
    // row_remain (how many beats remain in the current row)    
    txn_position track_phase ( tlm::tlm_generic_payload& txn, const tlm::tlm_phase& phase ) {
	switch ( phase ) {
	case tlm::BEGIN_REQ            : return track_request( txn );
	case tlm::BEGIN_RESP           : return track_response( txn );
	default:
	    if ( phase == BEGIN_DATA )
		return track_data( txn );
	    else 
		throw ( ocp_txn_track_error( "Can only track dataflow begin phases: BEGIN_REQ/BEGIN_DATA/BEGIN_RESP" ) );
	}
    }

    txn_position track_request( tlm::tlm_generic_payload& txn ) {
	ext_support_type ext_support( m_p_param->data_wdth );
	bool new_req = ( m_burst.empty() || m_burst.back().complete_req() );
	if ( new_req ) {
	    m_burst.push_back( txn_status() );
	}
	txn_status& cur_burst = m_burst.back();

	bool single_req = false; // can only be set in new_req block
	if ( new_req ) {
	    // get invariant data now
	    burst_sequence* b_seq; 
	    bool has_b_seq= ext_support.get_extension(b_seq, txn);
	    bool blck_seq = has_b_seq && b_seq->value.sequence == BLCK;
	    bool incr_seq = ( (has_b_seq && b_seq->value.sequence == INCR) ||
			      (txn.get_streaming_width()>=txn.get_data_length()) );
	    uint32_t block_height = blck_seq ? b_seq->value.block_height : 1;
	    assert( block_height > 0 );
	    bool precise    = !m_p_param->burstprecise || ext_support.get_extension<imprecise>(txn)==NULL;
	    single_req = m_p_param->burstsinglereq && ext_support.get_extension<srmd>(txn);
    
	    // burst length
	    unsigned int length;
	    burst_length* b_len;
	    if ( ext_support.get_extension<burst_length>(b_len, txn) ) {
		length = b_len->value;
	    } else {	    
		unsigned int offset = calculate_ocp_address_offset( txn, m_byte_width );
		uint64_t dummy_addr;
		length = calculate_ocp_burst_length(
		    txn, m_byte_width, offset,
		    m_p_param->burst_aligned && incr_seq, dummy_addr, has_b_seq ? b_seq: NULL );
	    }

	    // Set up expected count
	    if ( !m_p_param->burstprecise || precise ) {
		cur_burst.m_precise      = precise;
		cur_burst.m_row_length   = length;
		cur_burst.m_expect_count = block_height * cur_burst.m_row_length;
	    }
	}

	// Update request count
	if ( single_req ) {
	    assert( new_req );
	    cur_burst.m_req_count = cur_burst.m_expect_count;
	    if ( txn.is_write() && m_p_param->writeresp_enable ) {
		// srmd write expects only one response, not the expect_count
		cur_burst.m_single_resp = true;
	    }
	} else {
	    if ( is_tl2() ) {
		word_count* word_count_info;
		bool has_word_count= ext_support.get_extension(word_count_info, txn);
		if ( has_word_count )
		    cur_burst.m_req_count += word_count_info->value.request_wc;
		else {
		    // completes the burst
		    cur_burst.m_req_count = cur_burst.m_expect_count;
		}
	    } else {
		cur_burst.m_req_count++;
	    }
	}

	// Update expected count if at the end of an imprecise burst
	if ( !cur_burst.m_precise ) {
	    // must have explicit burst_len
	    burst_length* b_len;
	    bool has_b_len=ext_support.get_extension<burst_length>(b_len, txn);
	    if ( !has_b_len )
		throw ( ocp_txn_track_error( "Ocp transaction for imprecise burst must define burst_length extension" ) );
	    if ( b_len->value == 1 ) {
		cur_burst.m_expect_count = cur_burst.m_req_count;
	    }
	}

	// Set up data and response count if not expected
	if ( !txn.is_write() || !m_p_param->datahandshake ) {
	    cur_burst.m_datahs_count = cur_burst.m_expect_count;
	}
	if ( txn.is_write() && !m_p_param->writeresp_enable ) {
	    cur_burst.m_resp_count = cur_burst.m_expect_count;
	}

	txn_position ret;
	ret.count     = cur_burst.m_req_count;
	ret.remain    = cur_burst.m_expect_count - ret.count;
	ret.row_count  = ( (int)( ret.count - 1 ) % cur_burst.m_row_length ) + 1;
	ret.row_remain = cur_burst.m_row_length - ret.row_count;

	purge();
	return ret;
    }

    txn_position track_data( tlm::tlm_generic_payload& txn ) {
	if ( m_burst.empty() )
	    throw ( ocp_txn_track_error( "Received Data phase before Request" ) );

	ext_support_type ext_support( m_p_param->data_wdth );

	// Find the first burst still expecting data
	std::deque<txn_status>::iterator pos = m_burst.begin();
	while ( pos->complete_datahs() ) {
	    if ( ++pos == m_burst.end() ) {
		throw( ocp_txn_track_error(
			   "Received too many Data phases for pending Requests" ) );
	    }
	}
	txn_status& cur_burst = *pos;
	assert( cur_burst.m_expect_count == -1 ||
		cur_burst.m_datahs_count < static_cast<uint32_t>( cur_burst.m_expect_count ) );

	if ( is_tl2() ) {
	    word_count* word_count_info;
	    bool has_word_count= ext_support.get_extension(word_count_info, txn);
	    if ( has_word_count )
		cur_burst.m_datahs_count += word_count_info->value.data_wc;
	    else {
		// completes the burst
		cur_burst.m_datahs_count = cur_burst.m_expect_count;
	    }
	} else {
	    cur_burst.m_datahs_count++;
	}

	txn_position ret;
	ret.count     = cur_burst.m_datahs_count;
	ret.remain    = cur_burst.m_expect_count - ret.count;
	ret.row_count  = ( (int)( ret.count - 1 ) % cur_burst.m_row_length ) + 1;
	ret.row_remain = cur_burst.m_row_length - ret.row_count;

	purge();
	return ret;
    }

    txn_position track_response( tlm::tlm_generic_payload& txn ) {
	if ( m_burst.empty() )
	    throw ( ocp_txn_track_error( "Received Response phase before Request" ) );

	ext_support_type ext_support( m_p_param->data_wdth );

	// Find the first burst still expecting responses
	std::deque<txn_status>::iterator pos = m_burst.begin();
	while ( pos->complete() ) {
	    if ( ++pos == m_burst.end() ) {
		throw( ocp_txn_track_error(
			   "Received too many Response phases for pending Requests" ) );
	    }
	}
	txn_status& cur_burst = *pos;
	if (!( cur_burst.m_expect_count == -1 ||
	       cur_burst.m_resp_count < static_cast<uint32_t>( cur_burst.m_expect_count ) ||
	       ( cur_burst.m_resp_count < 1 && cur_burst.m_single_resp ) ) ) {
	    throw ( ocp_txn_track_error( "Received too many response phases for pending Requests" ) );
	}

	if ( is_tl2() ) {
	    word_count* word_count_info;
	    bool has_word_count= ext_support.get_extension(word_count_info, txn);
	    if ( has_word_count )
		cur_burst.m_resp_count += word_count_info->value.response_wc;
	    else {
		// completes the burst
		cur_burst.m_resp_count = cur_burst.m_expect_count;
	    }
	} else {
	    cur_burst.m_resp_count++;
	}

	txn_position ret;
	ret.count     = cur_burst.m_resp_count;
	ret.remain    = cur_burst.m_expect_count - ret.count;
	ret.row_count  = ( (int)( ret.count - 1 ) % cur_burst.m_row_length ) + 1;
	ret.row_remain = cur_burst.m_row_length - ret.row_count;
	if ( ret.count == 1 && cur_burst.m_single_resp ) {
	    ret.remain    = 0;
	    ret.row_remain = 0;
	}

	if ( ret.remain == 0 && !cur_burst.complete_datahs() ) {
	    throw( ocp_txn_track_error(
		       "Received SRMD Write Response phase before all Data phases completed" ) );
	}

	purge();
	return ret;
    }

private:
    bool m_tl1_n_tl2;
};

struct txn_position: public tlm_utils::instance_specific_extension<txn_position> {
    ocp_txn_track_base::txn_position req_position;
    ocp_txn_track_base::txn_position dh_position;
    ocp_txn_track_base::txn_position resp_position;
};

// Burst invariant fields storage utility
template <uint32_t BUSWIDTH, uint32_t ADDRWIDTH>
struct txn_burst_invariant:
public tlm_utils::instance_specific_extension<txn_burst_invariant<BUSWIDTH,ADDRWIDTH> > {
    typedef typename ocp_data_class_unsigned<BUSWIDTH,ADDRWIDTH>::AddrType addr_type;

    sc_core::sc_time          start_time;
    
    // gp invariants
    uint32_t                  streaming_width;
    uint32_t                  data_length;
    
    // ocp request info invariants
    ocpip_legacy::OCPMCmdType cmd;
    uint32_t                  threadid;
    uint32_t                  tagid;
    uint32_t                  connid;
    uint32_t                  addr_space;
    uint32_t                  atomic_length;
    uint32_t                  burst_length;
    bool                      precise;
    bool                      srmd;
    
    // burst sequence. Copy of the extension only defined when
    // burst_seq_ext_valid=true
    burst_seq_info            burst_seq_ext;
    bool                      burst_seq_ext_valid;
    burst_seqs                get_sequence() const {
	if ( burst_seq_ext_valid )
	    return burst_seq_ext.sequence;
	else
	    return (streaming_width < data_length) ? STRM : INCR;
    }

    addr_type get_unkn_address( uint32_t beat ) const {
	const uint32_t byte_width = (BUSWIDTH+7)>>3;
	burst_seqs seq = get_sequence();
	if (byte_width>=burst_seq_ext.unkn_dflt_bytes_per_address) {
	    if (seq == DFLT1) {
		uint32_t offset=burst_seq_ext.unkn_dflt_addresses[0]%byte_width;
		if (beat==1){ //first chunk
		    return burst_seq_ext.unkn_dflt_addresses[0]-offset; //align first address to our buswidth
		}
		offset/=burst_seq_ext.unkn_dflt_bytes_per_address;
		return burst_seq_ext.unkn_dflt_addresses[(beat-1)*(byte_width/burst_seq_ext.unkn_dflt_bytes_per_address)-offset];
	    }
	    //DFLT2, UNKN no packing
	    unsigned int offset=burst_seq_ext.unkn_dflt_addresses[beat-1]%byte_width;
	    assert(burst_seq_ext.unkn_dflt_addresses.size()>(beat-1));
	    return burst_seq_ext.unkn_dflt_addresses[beat-1]-offset;
	}
	else{ //DFLT1, UNKN split
	    uint32_t unkn_pseudo_b_len=(burst_seq_ext.unkn_dflt_bytes_per_address+byte_width-1)/byte_width;
	    unsigned int cnt, row;
	    row=(beat-1)/unkn_pseudo_b_len;
	    cnt=beat-(row*unkn_pseudo_b_len)-1;
	    assert(burst_seq_ext.unkn_dflt_addresses.size()>row);
	    return burst_seq_ext.unkn_dflt_addresses[row]+byte_width*cnt;
	}
    }

    static
    ocpip_legacy::OCPMCmdType decode_cmd( tlm::tlm_generic_payload& txn,
					  infr::bind_checker<tlm::tlm_base_protocol_types>::ext_support_type& ext_support ) {
	lock* lck;
	semaphore * sem;
	if (ext_support.template get_extension<lock>(lck, txn) && txn.is_read()) {
	    return ocpip_legacy::OCP_MCMD_RDEX;
	} else if (ext_support.template get_extension<semaphore>(sem, txn)){
	    if (txn.is_read()) return txn.is_read() ?
		ocpip_legacy::OCP_MCMD_RDL: ocpip_legacy::OCP_MCMD_WRC;
	} else if (ext_support.template get_extension<nonposted>(txn)){
	    if (txn.is_write()) return ocpip_legacy::OCP_MCMD_WRNP;
	    else
		std::cerr<<"Warning: Non posted non-write cannot be decoded for OCP"
			 <<std::endl;
	} else if (ext_support.template get_extension<broadcast>(txn)){
	    if (txn.is_write()) return ocpip_legacy::OCP_MCMD_BCST;
	    else
		std::cerr<<"Warning: Broadcast non-write cannot be decoded for OCP"
			 <<std::endl;
	} else if (txn.is_write()) {
	    return ocpip_legacy::OCP_MCMD_WR;
	} else if (txn.is_read()) {
	    return ocpip_legacy::OCP_MCMD_RD;
	}
	return ocpip_legacy::OCP_MCMD_IDLE;
    }

    // extract an instance from a generic payload
    static 
    txn_burst_invariant init_from( tlm::tlm_generic_payload& txn,
				   const ocp_parameters& params,
				   infr::bind_checker<tlm::tlm_base_protocol_types>::ext_support_type& ext_support ) {

	txn_burst_invariant inv;
	inv.streaming_width = txn.get_streaming_width();
	inv.data_length     = txn.get_data_length();

	inv.cmd = decode_cmd(txn, ext_support);

	inv.threadid = inv.tagid = 0;
	thread_id* th_id; 
	if ( ext_support.template get_extension(th_id, txn) )
	    inv.threadid = th_id->value;
	tag_id* tg_id; 
	if ( ext_support.template get_extension(tg_id, txn) )
	    inv.tagid = tg_id->value;

	conn_id* conn_id_info;
	bool has_conn_id=ext_support.template get_extension(conn_id_info, txn);
	inv.connid = has_conn_id ? conn_id_info->value : 0;

	address_space* address_space_info; 
	bool has_address_space=ext_support.template get_extension(address_space_info, txn);
	inv.addr_space = has_address_space ? address_space_info->value : 0;

	OCPIP_VERSION::atomic_length* atomic_length_info;
	bool has_atomic_length=ext_support.template get_extension(atomic_length_info, txn);
	inv.atomic_length = has_atomic_length ? atomic_length_info->value : 0;

	burst_sequence* b_seq;
	if ( inv.burst_seq_ext_valid=ext_support.template get_extension(b_seq, txn) ) {
	    inv.burst_seq_ext      = b_seq->value;
	}

	OCPIP_VERSION::burst_length* b_len; 
	if ( ext_support.template get_extension(b_len, txn) ) {
	    inv.burst_length      = b_len->value;
	} else {
	    const uint32_t byte_width=(BUSWIDTH+7)>>3;
	    unsigned int offset = calculate_ocp_address_offset( txn, byte_width );
	    uint64_t dummy_addr;
	    inv.burst_length = calculate_ocp_burst_length(
		txn, byte_width, offset,
		params.burst_aligned && inv.get_sequence()==INCR, dummy_addr,
		inv.burst_seq_ext_valid ? b_seq : NULL );
	}

	inv.precise = !params.burstprecise ||
	    ext_support.template get_extension<imprecise>(txn)==NULL;
	inv.srmd = params.burstsinglereq &&
	    ext_support.template get_extension<OCPIP_VERSION::srmd>(txn);

	inv.start_time = sc_core::sc_time_stamp();
	return inv;
    }
};
}
