// 
//  (c) Copyright OCP-IP 2006
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Tim Kogel, CoWare, Inc
//          $Id:
//
//  Description : definition of simple OCP TL3 bus
//
// ============================================================================

//TODO: add acquistions and releases

#include "ocp_tl3_bus.h"

ocp_tl3_bus::ocp_tl3_bus(sc_core::sc_module_name mod):
  sc_core::sc_module(mod),
  ocp_s("ocp_s"),
  ocp_m("ocp_m"),
  m_index(~0),
  m_previous_read_index(~0),
  m_outstanding_reads(0),
  m_clk_period(1, SC_NS)
{

  m_active_master_id=-1;

  SC_METHOD(request_method);
  sensitive<<m_req_event;
  dont_initialize();

  SC_METHOD(response_method);
  sensitive<<m_resp_event;
  dont_initialize();
  
  m_dont_resp_acc=false;
  m_state=m_resp_state=0;
  
  ocp_s.register_nb_transport_fw(this, &ocp_tl3_bus::nb_transport_fw);
  ocp_m.register_nb_transport_bw(this, &ocp_tl3_bus::nb_transport_bw);
}

void ocp_tl3_bus::start_of_simulation(){
  std::cout<<"Start of sim callback!"<<std::endl
           <<"  Connected masters: "<<ocp_s.size()<<std::endl
           <<"  Connected slaves:  "<<ocp_m.size()<<std::endl;
  m_pending_reqs.resize(ocp_s.size(), std::pair<tlm::tlm_generic_payload*, sc_core::sc_time>(NULL, sc_core::SC_ZERO_TIME));
  m_running_resp=m_pending_resp=m_running_req=NULL;
}

void
ocp_tl3_bus::request_method() {  
  unsigned int n_active_masters = 0;
  switch(m_state){
    case 0:
      std::cout << sc_core::sc_time_stamp() << ", " << sc_core::sc_module::name() << ", collecting requests... active masters : ";
      for (unsigned int i=0; i<m_pending_reqs.size(); i++) {
        if(m_pending_reqs[i].first!=NULL && m_pending_reqs[i].second<sc_core::sc_time_stamp()){
          std::cout << i << " ";
          n_active_masters++;
          m_active_master_id = i; 
        }
      }
      std::cout << std::endl;
      if(!n_active_masters) {
        std::cout<<" suspended"<<std::endl;
        return;
      }
      
      
      // save initiator id for the response path
      // based on the address, forward the request to target slave:
      m_index = (m_pending_reqs[m_active_master_id].first->get_address() / 0x100) % ocp_m.size();

      if (m_pending_reqs[m_active_master_id].first->is_read()) { //we can only start the next read if the read bus is free, or it is a read to the same location
        if ( m_previous_read_index != m_index ) { //location changed
          std::cout << sc_core::sc_time_stamp() << ", " << sc_core::sc_module::name() 
                    << ", m_outstanding_reads :" << m_outstanding_reads
                    << ", m_previous_read_index :" << m_previous_read_index
                    << ", m_index :" << m_index <<std::endl;
          if ( m_outstanding_reads > 0 ) { //gotta wait for the current read(s) to end
            m_state=1; sc_core::next_trigger(m_resp_bus_free); return;
    case 1: ;
          }
          m_previous_read_index = m_index;	    
        }
        m_outstanding_reads++;
        m_reading_masters.push_back(m_active_master_id);
      }

      if (m_running_req!=NULL || m_req_end_time>sc_core::sc_time_stamp()) {
        std::cout << sc_core::sc_time_stamp() << ", " << sc_core::sc_module::name()
                  << ", target " << m_index << " is not yet ready \n";
        //wait( ocp_m[m_index]->RequestEndEvent() );
        m_state=2; sc_core::next_trigger(m_req_end_event); return;
    case 2:;
      }

      do_req();
      // wait for transfer delay
      std::cout << sc_core::sc_time_stamp() << ", " << sc_core::sc_module::name() 
                << "::request_thread() transfer-delay : " 
                << m_clk_period << std::endl;
      
      //note that do_req and do_req_acc are atomic. So the init will always see END_REQ followed by BEGIN_RESP
      // because the END_REQ comes from do_req_acc, while BEGIN_RESP comes from the resp_method
      do_req_acc();
      m_state=0;
      m_req_event.notify(m_clk_period); //restart this method
  }
}
    

void
ocp_tl3_bus::response_method() {  
  switch(m_resp_state){
    case 0:
      std::cout << sc_core::sc_time_stamp() << ", " << sc_core::sc_module::name() << "::response_method(),"
                << " get resp"<<std::endl;
       
      // send the reponse to initiating master:
      assert(m_reading_masters.size());
      if ( m_running_resp!=NULL || m_resp_end_time>sc_core::sc_time_stamp()) {
        std::cout << sc_core::sc_module::name() << " our master is not yet ready, so we'll wait for it\n";
        m_resp_state=1; next_trigger(m_resp_end_event); return;
    case 1:;
      }
      do_resp();

      m_outstanding_reads--;
      std::cout << sc_core::sc_time_stamp() << ", " << sc_core::sc_module::name() 
                << ", response_thread has sent response, outstanding reads : " 
                << m_outstanding_reads << endl;
      if (m_outstanding_reads==0) m_resp_bus_free.notify(m_clk_period);
      do_resp_acc();
      
      m_pending_resp->release(); //we are done now towards the slave. Let's release
      m_pending_resp=NULL;
      //ocp_m[m_previous_read_index]->acceptResponse();    
      // transaction completed. 
      m_resp_state=0;
  }
}

void ocp_tl3_bus::do_req(){
  std::cout << sc_core::sc_time_stamp() << ", " << sc_core::sc_module::name() <<" sending req from "<<m_active_master_id<<" to "<<m_index<<std::endl;
  m_ph=tlm::BEGIN_REQ;
  m_time=sc_core::SC_ZERO_TIME;
  assert(m_running_req==NULL);
  assert(m_pending_reqs[m_active_master_id].first);
  m_running_req=m_pending_reqs[m_active_master_id].first;
  
  m_running_req->acquire();
  
  switch(ocp_m[m_index]->nb_transport_fw(*m_running_req, m_ph, m_time)){
    case tlm::TLM_COMPLETED:
      if (m_running_req->is_read()){ //writes do not have resps in this example
        m_ph=tlm::BEGIN_RESP;
        m_dont_resp_acc=true;
        }
      else
        m_ph=tlm::END_REQ;
    case tlm::TLM_UPDATED:
        nb_transport_bw(m_index, *m_running_req, m_ph, m_time);
    case tlm::TLM_ACCEPTED: 
      break;
  }
}

void ocp_tl3_bus::do_req_acc(){
  std::cout << sc_core::sc_time_stamp() << ", " << sc_core::sc_module::name() <<" accepting req from "<<m_active_master_id<<" in "<<m_clk_period<<std::endl;
  m_ph=tlm::END_REQ;
  m_time=m_clk_period;
  assert(m_pending_reqs[m_active_master_id].first);
  switch(ocp_s[m_active_master_id]->nb_transport_bw(*m_pending_reqs[m_active_master_id].first, m_ph, m_time)){
    case tlm::TLM_ACCEPTED: 
    case tlm::TLM_UPDATED: break;
    case tlm::TLM_COMPLETED:
      assert(0 && "TLM_COMPLETED not allowed in response to END_REQ"); exit(666);
      break;
  }
  m_pending_reqs[m_active_master_id].first=NULL;
}

void ocp_tl3_bus::do_resp(){
  std::cout << sc_core::sc_time_stamp() << ", " << sc_core::sc_module::name() <<" sending resp from to "<<m_reading_masters.front()<<std::endl;
  m_ph=tlm::BEGIN_RESP;
  m_time=sc_core::SC_ZERO_TIME;
  assert(m_running_resp==NULL);
  assert(m_pending_resp!=NULL);
  m_running_resp=m_pending_resp;
  switch(ocp_s[m_reading_masters.front()]->nb_transport_bw(*m_running_resp, m_ph, m_time)){
    case tlm::TLM_ACCEPTED: 
      break;
    case tlm::TLM_COMPLETED:
      m_ph=tlm::END_RESP;
    case tlm::TLM_UPDATED:
      switch (m_ph){
        case tlm::END_RESP:
          nb_transport_fw(m_reading_masters.front(), *m_running_resp, m_ph, m_time);
        default:;
      }
      break;
  }
  m_reading_masters.pop_front();
}

void ocp_tl3_bus::do_resp_acc(){
  if (m_dont_resp_acc) {
    std::cout << sc_core::sc_time_stamp() << ", " << sc_core::sc_module::name() <<" skipping accepting resp from "<<m_previous_read_index<<" in "<<m_clk_period<<std::endl;
    m_dont_resp_acc=false;
    return;
  }
  std::cout << sc_core::sc_time_stamp() << ", " << sc_core::sc_module::name() <<" accepting resp from "<<m_previous_read_index<<" in "<<m_clk_period<<std::endl;
  m_ph=tlm::END_RESP;
  m_time=m_clk_period;
  assert(m_pending_resp);
  ocp_m[m_previous_read_index]->nb_transport_fw(*m_pending_resp, m_ph, m_time);
}

tlm::tlm_sync_enum ocp_tl3_bus::nb_transport_fw(unsigned int index, tlm::tlm_generic_payload& gp, tlm::tlm_phase& ph, sc_core::sc_time & t){
  std::cout<<sc_core::sc_time_stamp()<<"(+"<<t<<"), "<<sc_core::sc_module::name()<<": "<<ph<<" from master "<<index<<std::endl;
  switch(ph){
    case tlm::BEGIN_REQ:
      assert(m_pending_reqs[index].first==NULL); //there must not be a pending req
      m_pending_reqs[index].first=&gp;
      m_pending_reqs[index].second=sc_core::sc_time_stamp()+t;
      m_req_event.notify(t+m_clk_period); //arbitrate in next clk cycle
      break;
    case tlm::END_RESP:
      assert(m_running_resp);
      m_running_resp=NULL;
      m_resp_end_time=sc_core::sc_time_stamp()+t+m_clk_period;
      m_resp_end_event.notify(t+m_clk_period);
      break;
  }
  return tlm::TLM_ACCEPTED;
}

tlm::tlm_sync_enum ocp_tl3_bus::nb_transport_bw(unsigned int index, tlm::tlm_generic_payload& gp, tlm::tlm_phase& ph, sc_core::sc_time & t){
  std::cout<<sc_core::sc_time_stamp()<<"(+"<<t<<"), "<<sc_core::sc_module::name()<<": "<<ph<<" from slave "<<index<<std::endl;
  switch(ph){
    case tlm::END_REQ:
      assert(m_running_req);
      if (m_running_req->is_write()) m_running_req->release(); //done with it towards the slave, so let's release it
      m_running_req=NULL;
      m_req_end_time=sc_core::sc_time_stamp()+t+m_clk_period;
      m_req_end_event.notify(t+m_clk_period);
      break;
    case tlm::BEGIN_RESP:
      if (m_running_req==&gp){ //BEGIN_RESP covers END_REQ
        m_running_req=NULL;
        m_req_end_time=sc_core::sc_time_stamp()+t+m_clk_period;
        m_req_end_event.notify(t+m_clk_period);
      }
      assert(m_pending_resp==NULL); //there must not be a pending resp
      m_pending_resp=&gp;
      m_resp_event.notify(t+m_clk_period); //arbitrate in next clk cycle
      break;
  }
  return tlm::TLM_ACCEPTED;  
}


