// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Albert Chitemyan, Sonics Inc. 
//                
//         $Id: 
//
//  Description : OCP API - Layer-2 Simple Example slave
//    Features:
//    - Synchronous
//    - Non-blocking methods are used
//    - Not pipelined
// ============================================================================

#include "slave.h"

using namespace std;

///////////////////////////////////////////////////////////
// 
// In this example we are going to demonstrate how
// to use word_count payload extension.
// 
// First, master sends 8-length write burst using word count
// extension.
// It sends 3 words in one call and then 5 remaining words in 
// another call.
//
// Second, master sends 4-length read burst.
// Slave sends 3 word response and then remaining 1 word.
//
///////////////////////////////////////////////////////////

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Slave<BUSWIDTH, ADDRWIDTH>::Slave (sc_core::sc_module_name name_)
  : sc_core::sc_module (name_), tp_socket("tp_socket"), req(NULL), ongoing_rsp(false) {

  // initialize common members
  last_request = tlm::TLM_IGNORE_COMMAND;
  chunk_counter = 0;
  sent_word_cnt = 0;

  for (int i=0; i < 4; i++)
     response_data[i] = 720000 + i;

  SC_METHOD(proc);
    sensitive << clk.pos();
    dont_initialize(); // make sure we execute only at clock edge

  tp_socket.register_nb_transport_fw(this, &Slave::nb_transport);
}

// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Slave<BUSWIDTH, ADDRWIDTH>::~Slave(){}

// ----------------------------------------------------------------------------
//  Method : Slave::proc()
//
//  Synchronous slave process
//
// ----------------------------------------------------------------------------

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> void Slave<BUSWIDTH, ADDRWIDTH>::proc()
{

  // If a previous read-burst has been served, we check for new burst 
  if (last_request == tlm::TLM_IGNORE_COMMAND) {
    if (req){
      last_request = req->get_command();

      // get request_word_count
      ocpip::word_count* word_count_info;
      tp_socket.get_extension<ocpip::word_count>(word_count_info, *req);
      unsigned int current_chunk = word_count_info->value.request_wc;

      if (last_request == tlm::TLM_READ_COMMAND)
#ifdef DEBUG_TL2                  
      {
#endif
        rsp = req;

        burst_length = get_burst_length();
#ifdef DEBUG_TL2                  
        cout << "Slave got READ burst " 
             << "   word_count=" << current_chunk 
             << "                      addr=" << rsp->get_address() 
             << "                             time=" << sc_core::sc_time_stamp() << endl;
      }
      else {

        cout << "Slave got WRITE burst " 
             << "   word_count=" << current_chunk 
             << "   data={";

        for (unsigned int i = 0; i < current_chunk; ++i)
            cout << ((Td*)(req->get_data_ptr()))[i] << "   " ;
        cout << "}"
             << "             time=" << sc_core::sc_time_stamp() << endl;
      }
#endif      

      time = sc_core::SC_ZERO_TIME;
      phase = tlm::END_REQ;
      tlm::tlm_sync_enum retVal = tp_socket->nb_transport_bw(*req, phase, time);

      switch(retVal){
        case tlm::TLM_ACCEPTED: break;
        case tlm::TLM_UPDATED:
          cerr<<"Error in " << name() << " got unexpected phase update " << phase << " in response to END_REQ" << endl;
          exit(1);
          break;
        case tlm::TLM_COMPLETED:;
      }
      req = NULL;
    }
  }

  // If request was write, there is no respose
  if (last_request == tlm::TLM_WRITE_COMMAND) {
    last_request = tlm::TLM_IGNORE_COMMAND;
  }
  else 
  if (last_request == tlm::TLM_READ_COMMAND) {	
    if (!ongoing_rsp) 
    {
      assert(rsp);

      generate_read_response();

      // Send response
      time = sc_core::SC_ZERO_TIME;
      phase = tlm::BEGIN_RESP;
      tlm::tlm_sync_enum retVal = tp_socket->nb_transport_bw(*rsp,phase, time);

      switch(retVal){
        case tlm::TLM_ACCEPTED: 
          ongoing_rsp = true; 
          break;
        case tlm::TLM_UPDATED:
          if (phase != tlm::END_RESP) {
            std::cerr<<"Error in "<< name() <<" got unexpected phase update "<< phase <<" in response to BEGIN_RESP"<<std::endl;
            exit(1);
          }
          break;
        case tlm::TLM_COMPLETED: break;
      }

#ifdef DEBUG_TL2
      cout << "Slave sent response     " 
           << " word_count=" << word_count_info->value.response_wc
           << "   data={  ";

      for (unsigned int i = 0; i < word_count_info->value.response_wc; ++i)
           cout << ((Td*)(rsp->get_data_ptr()))[i] << "   " ;
      cout << "}"
	       << "               time=" << sc_core::sc_time_stamp() << endl;
#endif
      if (sent_word_cnt == burst_length) 
      {
         last_request = tlm::TLM_IGNORE_COMMAND;
         rsp = NULL;
      }
    } // end of if (!ongoing)
  } 
} // end of method

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
tlm::tlm_sync_enum Slave<BUSWIDTH, ADDRWIDTH>::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim)
{
  switch(ph){
    case tlm::BEGIN_REQ: 
      req = &txn;     // get burst 
      break;
    case tlm::END_RESP: 
          ongoing_rsp = false; 
      break;
    default:
      std::cerr<<"Error in "<<name()<<" got unexpected phase "<<phase<<" in nb_transport_fw"<<std::endl;
      exit(1);        
  }
  return tlm::TLM_ACCEPTED;
}

// We got 4-length read burst 
// and will send response data in 2 chunks:
// in first call we will send 3 data
// then, in second call, remaining 1 data
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Slave<BUSWIDTH, ADDRWIDTH>::generate_read_response()
{
   unsigned int current_resp_chunk;

   // first chunk of read response  
   if (chunk_counter == 0)
    current_resp_chunk = 3;
   // second chunk of read response 
   else if (chunk_counter == 1)
     current_resp_chunk = 1;

   sent_word_cnt += current_resp_chunk;

   // get request_word_count
   tp_socket.get_extension<ocpip::word_count>(word_count_info, *rsp);
   unsigned int req_wc = word_count_info->value.request_wc;;
  
   // do not send more data then required
   assert (current_resp_chunk <= req_wc);

   word_count_info->value.response_wc = current_resp_chunk;
   tp_socket.validate_extension<ocpip::word_count>(*rsp);

   rsp->set_response_status(tlm::TLM_OK_RESPONSE);

   if (chunk_counter == 0) // first chunk of data
      memcpy(  ((Td*) (rsp->get_data_ptr())), response_data, sizeof(unsigned int) * current_resp_chunk);
   else if (chunk_counter == 1)
      memcpy(  ((Td*) (rsp->get_data_ptr())), response_data+3, sizeof(unsigned int) * current_resp_chunk);
  
   chunk_counter++;
}

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
unsigned int Slave<BUSWIDTH, ADDRWIDTH>::get_burst_length()
{
  assert(rsp);
 
  ocpip::burst_length* b_len;
  tp_socket.get_extension<ocpip::burst_length>(b_len, *rsp);
  return b_len->value;
}

// ----------------------------------------------------------------------------
//
//  Instantiation of the Slave
//
// ----------------------------------------------------------------------------
template class Slave< 32,32 >; 
