// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Albert Chitemyan, Sonics Inc. 
//   
//         $Id: 
//
//  Description : OCP API - Layer-2 test Master
//    Features:
//    - Synchronous
//    - Non-blocking methods are used
// ============================================================================

#include <cstdlib> 
#include <memory.h>
using namespace std;

#include "master.h"

///////////////////////////////////////////////////////////
// 
// In this example we are going to demonstrate how
// to use word_count payload extension.
// 
// First, master sends 8-length write burst using word count
// extension.
// It sends 3 words in one call and then 5 remaining words in 
// another call.
//
// Second, master sends 4-length read burst.
// Slave sends 3 word response and then remaining 1 word.
//
///////////////////////////////////////////////////////////

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Master<BUSWIDTH, ADDRWIDTH>::Master (sc_core::sc_module_name name_): 
  sc_core::sc_module (name_), ip_socket("ip_socket", ocpip::ocp_master_socket_tl2<32>::mm_txn_with_data()), ongoing_req(false), rsp(NULL) {
    
  // initialize common members
  transaction_count = 0;
  write_burst_done = false;
  sent_word_cnt = 0;
  chunk_counter = 0;

  for (int i=0; i < 8; i++) 
     write_data[i] = 10000 + i;

  SC_METHOD(proc);
    sensitive<<clk.pos(); 
    dont_initialize();

  ip_socket.register_nb_transport_bw(this, &Master::nb_transport);
}

// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Master<BUSWIDTH, ADDRWIDTH>::~Master() {}

// ----------------------------------------------------------------------------
//  Method : Master::proc()
//
//  Synchronous Master process
//
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> void Master<BUSWIDTH, ADDRWIDTH>::proc() 
{
  // check if channel is free
  // we are going to send 3 transactions in this example
  if (!ongoing_req && transaction_count < 3) {
    // first, we send write burst 
    if (!write_burst_done)
      generate_write_burst();
    else
      generate_read_burst();
      
    // Send burst 
    time  = sc_core::SC_ZERO_TIME;
    phase = tlm::BEGIN_REQ;
    tlm::tlm_sync_enum slave_answer = ip_socket->nb_transport_fw(*req, phase, time);

    switch(slave_answer){
      case tlm::TLM_ACCEPTED: 
        ongoing_req = true; 
        break;
      case tlm::TLM_UPDATED: 
        switch (phase){
          case tlm::END_REQ:

            if (req->get_command() == tlm::TLM_WRITE_COMMAND) 
            {    
                if (sent_word_cnt == burst_len) {
                   write_burst_done = true;
                   ip_socket.release_transaction(req);
                }
            }
            break;
          default:
            std::cerr<<"Error in "<<name()<<" : got unexpected phase update to "<<phase<<" in response to BEGIN_REQ."<<std::endl;
            exit(1);                
        }
        break;
      case tlm::TLM_COMPLETED:;
    }

#ifdef DEBUG_TL2
    cout << 
    endl << "------------------------------------------------------------------------------------ time=" 
         << sc_core::sc_time_stamp() << endl;

    cout << "Master sent burst with length " <<  burst_len
         << "    word_count=" << word_count_info->value.request_wc;

    if (req->get_command() == tlm::TLM_WRITE_COMMAND) {
      cout << "   WRITE "
           << "   addr=" << req->get_address()
           << "   data={  "; 

      for (unsigned int i = 0; i < word_count_info->value.request_wc; ++i)
           cout << ((Td*)(req->get_data_ptr()))[i] << "  " ;
      cout << "}"; 


    } else {
      cout << "   READ "
           << "   addr=" << req->get_address();
    }

    cout << endl;
#endif
      
    transaction_count++;
  }

  if (rsp) 
  {
    time = sc_core::SC_ZERO_TIME;
    phase = tlm::END_RESP;
    ip_socket->nb_transport_fw(*rsp, phase, time);  

    if (rsp->get_response_status() == tlm::TLM_OK_RESPONSE) {
#ifdef DEBUG_TL2
      cout << "Master got valid response    word_count=" << word_count_info->value.response_wc
           << "   data={  ";

     for (unsigned int i = 0; i < word_count_info->value.response_wc; ++i)
           cout << ((Td*)(rsp->get_data_ptr()))[i] << "  " ;
      cout << "}"
	       << "                time=" << sc_core::sc_time_stamp() << endl;
#endif
    }
    ip_socket.release_transaction(rsp); //we are done with this txn now
    rsp = NULL;
  }
} // end of method

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
tlm::tlm_sync_enum Master<BUSWIDTH, ADDRWIDTH>::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim) 
{
  switch(ph){
    case tlm::END_REQ:
      ongoing_req = false;

      if (req->get_command() == tlm::TLM_WRITE_COMMAND)
      {
        if (sent_word_cnt == burst_len) {
           write_burst_done = true;
           ip_socket.release_transaction(req);
        }
      }      
      break;
    case tlm::BEGIN_RESP: 
      rsp = &txn;
      break;
    default:
      std::cerr<<"Error in "<<name()<<" : got unexpected phase "<<ph<<" in nb_transport_bw"<<std::endl;
      exit(1);
  }
  return tlm::TLM_ACCEPTED;
}


template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Master<BUSWIDTH, ADDRWIDTH>::generate_write_burst() 
{
  unsigned int current_chunk;

  // first chunk of the write burst 
  if (chunk_counter == 0)
    current_chunk = 3;
  // second chunk of the write burst
  else if (chunk_counter == 1)
    current_chunk = 5;

  sent_word_cnt += current_chunk; 

  burst_len = 8;
  tlm::tlm_command cmd = tlm::TLM_WRITE_COMMAND;

  // allocate burst once, when first chunk is sending
  if (chunk_counter == 0) {
    allocate_transaction(cmd, burst_len);
  }
  // update word_count information
  ip_socket.get_extension<ocpip::word_count>(word_count_info, *req);
  word_count_info->value.request_wc = current_chunk;
  ip_socket.validate_extension<ocpip::word_count>(*req);

  // fill data for each chunk
  if (chunk_counter == 0) // first chunk of data
    memcpy(  ((Td*) (req->get_data_ptr())), write_data, sizeof(unsigned int) * current_chunk);
  else if (chunk_counter == 1)
    memcpy(  ((Td*) (req->get_data_ptr())), write_data+3, sizeof(unsigned int) * current_chunk);

  chunk_counter++;
}

// Sending READ burst in one call, but response will
// be returned in 2 calls
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Master<BUSWIDTH, ADDRWIDTH>::generate_read_burst()
{
  tlm::tlm_command cmd = tlm::TLM_READ_COMMAND;
  burst_len = 4;

  allocate_transaction(cmd, burst_len);

  ip_socket.get_extension<ocpip::word_count>(word_count_info, *req);
  word_count_info->value.request_wc = burst_len;
  ip_socket.validate_extension<ocpip::word_count>(*req);
}


// Generates transaction byusing command type and burst length
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Master<BUSWIDTH, ADDRWIDTH>::allocate_transaction(tlm::tlm_command cmd, unsigned int b_length)
{
  // burst start address
  unsigned int addr = 0x160;

  req = ip_socket.get_transaction();

  req->set_command(cmd);
  req->set_address(addr);
  ip_socket.reserve_data_size(*req, b_length * sizeof(Td));
  req->set_streaming_width(8 * sizeof(Td));

  ocpip::burst_sequence* b_seq;
  ip_socket.get_extension<ocpip::burst_sequence>(b_seq, *req);
  b_seq->value.sequence = ocpip::INCR;
  ip_socket.validate_extension<ocpip::burst_sequence>(*req);

  ocpip::burst_length* b_len;
  ip_socket.get_extension<ocpip::burst_length>(b_len, *req);
  b_len->value = b_length;
  ip_socket.validate_extension<ocpip::burst_length>(*req);
}

// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------
template class Master< 32,32 >;

